"""

    PROJECT: flex_toolbox
    FILENAME: variables.py
    AUTHOR: David NAISSE
    DATE: August 5th, 2024

    DESCRIPTION: variables for the module
"""

import os
from tqdm import tqdm
import yaml 

from src.utils import download_file, get_sdk_version_mapping

useful_schemas = [
    "APIQuery",
    "ObjectAPIQuery",
    "AccountProperty",
    "NewAccountProperty",
    "AccountPropertyUpdate",
    "Account",
    "NewAccount",
    "AccountUpdate",
    "AccountAPIQuery",
    "Action",
    "NewAction",
    "ActionUpdate",
    "ActionAPIQuery",
    "Asset",
    "NewAssetPlaceholder",
    "AssetUpdate",
    "AssetAPIQuery",
    "Collection",
    "CreateCollectionRequest",
    "UpdateCollection",
    "EventHandler",
    "NewEventHandler",
    "EventHandlerUpdate",
    "EventHandlerAPIQuery",
    "Event",
    "NewEvent",
    "EventAPIQuery",
    "Group",
    "NewGroup",
    "GroupUpdate",
    "GroupAPIQuery",
    "Job",
    "NewJob",
    "JobAPIQuery",
    "MessageTemplate",
    "NewMessageTemplate",
    "MessageTemplateUpdate",
    "MessageTemplateAPIQuery",
    "MetadataDefinition",
    "NewMetadataDefinition",
    "MetadataDefinitionUpdate",
    "MetadataDefinitionAPIQuery",
    "ObjectType",
    "ObjectTypeAPIQuery",
    "Profile",
    "NewProfile",
    "ProfileUpdate",
    "ProfileAPIQuery",
    "Quota",
    "NewQuota",
    "QuotaUpdate",
    "QuotaAPIQuery",
    "Resource",
    "NewResource",
    "ResourceUpdate",
    "ResourceAPIQuery",
    "Role",
    "NewRole",
    "RoleUpdate",
    "ObjectAPIQuery",
    "TagCollection",
    # "qlsdkjfq",  # custom tag collection create
    "TaskDefinition",
    "NewTaskDefinition",
    "TaskDefinitionUpdate",
    "TaskDefinitionAPIQuery",
    "Task",
    "TaskStatus",
    "TaskAPIQuery",
    "Taxonomy",
    # "mlqsdkfjq",  # custom create taxonomy/taxon
    # "mlqsdkjfqmsldk",  # custom update taxonomy/taxon
    "Taxon",
    "TimedAction",
    "NewTimedAction",
    "TimedActionUpdate",
    "TimedActionAPIQuery",
    "UserDefinedObjectType",
    "NewUserDefinedObjectType",
    "UserDefinedObjectTypeUpdate",
    "UserDefinedObjectTypeAPIQuery",
    "User",
    "NewUser",
    "UserUpdate",
    "UserAPIQuery",
    "Variant",
    "NewVariant",
    "VariantUpdate",
    "VariantAPIQuery",
    "Wizard",
    "NewWizard",
    "WizardUpdate",
    "WizardAPIQuery",
    "WorkflowDefinition",
    "NewWorkflowDefinition",
    "WorkflowDefinitionUpdate",
    "WorkflowDefinitionAPIQuery",
    "WorkflowInstance",
    "NewWorkflow",
    "WorkflowAPIQuery",
    "Workspace",
    "NewWorkspace",
    "WorkspaceUpdate",
    "WorkspaceAPIQuery",
]


def yaml_property_to_python(key: str, value: dict):

    try:
        match value["type"]:
            case "string":
                return f"{key}: str"
            case "integer":
                return f"{key}: int"
            case "boolean":
                return f"{key}: bool"
            case "object":
                return f"{key}: dict"
            case "array":
                if "type" in value["items"]:
                    return f'{key}: List[{value["items"]["type"][:3]}]'
                else:
                    return f"{key}: List[int]"
            case _:
                return None
    except:
        # custom for collection sharing
        if key in [
            "sharing",
            "fileDetails",
            "mediaRegion",
            "segmentedFileLocation",
            "metadata",
        ]:
            return f"{key}: dict"
        else:
            print(f"FAILED FOR {key}")
            return f"{key}: None"


sdk_version_mapping = get_sdk_version_mapping()
for version in tqdm(sdk_version_mapping):
    try:
        download_file(
            url=f"https://help.dalet.com/daletflex/apis/flex-api-{version}.yml",
            destination=os.path.join("docs", f"{version}.yml"),
        )

        flex_doc = None
        with open(f"docs/{version}.yml", "r") as doc_yml:
            flex_doc = yaml.safe_load(doc_yml)

        schemas = None
        with open(f"docs/schemas_{version}.yml", "w") as schemas_yml:
            schemas = {}

            for s in useful_schemas:
                schemas[s] = flex_doc["components"]["schemas"][s]

            yaml.dump(schemas, schemas_yml, indent=2)

        with open(f"src/schemas/{version.replace('.', '_')}.py", "w") as version_module:
            version_module.write(f"# VERSION: {version}\n")
            version_module.write(
                f"# THIS FILE IS AUTOGENERATED FROM FLEX DOCUMENTATION.\n\n\n"
            )
            version_module.write(f"from typing import List\n\n\n")

            for schema in schemas:

                if any(kw in schema for kw in ["Update", "New", "Create"]):
                    version_module.write(f"class {schema}:\n")

                    properties = []
                    for k, v in schemas[schema]["properties"].items():
                        properties.append(yaml_property_to_python(k, v))
                    joined_props = []
                    names = []

                    # keys and __init__
                    for p in properties:
                        name, type = p.split(": ")
                        if name.endswith("Id") and any(
                            kw in schema for kw in ["Update", "Create", "New"]
                        ):
                            name = name[:-2]
                            type = "dict"
                        elif name.endswith("Ids") and not name == "externalIds" and any(
                            kw in schema for kw in ["Update", "Create", "New"]
                        ):
                            name = name[:-3]
                            type = "List[dict]"
                        else:
                            name = name
                        names.append(name)
                        joined_props.append(f"{name}: {type}")
                    version_module.write(
                        f"\tkeys: List[str] = [\"{'\", \"'.join(names)}\"]\n\n"
                    )
                    version_module.write(
                        f"\tdef __init__(self, {', '.join(joined_props)}):\n"
                    )

                    # attributes
                    for p in properties:
                        name, type = p.split(": ")
                        if name.endswith("Id") and any(
                            kw in schema for kw in ["Update", "Create", "New"]
                        ):
                            value = f"{name[:-2]}.get('id') if {name[:-2]} else None"
                        elif name.endswith("Ids") and not name == "externalIds" and any(
                            kw in schema for kw in ["Update", "Create", "New"]
                        ):
                            value = f"[v['id'] for v in {name[:-3]}]"
                        else:
                            value = f"{name}"
                        version_module.write(f"\t\tself.{name} = {value}\n")

                    version_module.write(f"\n\n")
    except:
        print(f"Failed vor {version}. ")
