#Feed Indexer tests [316-000000-043]

#Author: P.P.A. Kotze, H. Niehaus, T. Glaubach
#Date: 1/9/2020
#Version: 
#0.1 Initial
#0.2 Update after feedback and correction from HN email dated 1/8/2020
#0.3 Rework scu_get and scu_put to simplify
#0.4 attempt more generic scu_put with either jason payload or simple params, remove payload from feedback function
#0.5 create scu_lib
#0.6 1/10/2020 added load track tables and start table tracking also as debug added 'field' command for old scu
#HN: 13/05/2021 Changed the way file name is defined by defining a start time 
# 1.0 2022-05-26 added stowing / unstowing / taking /releasing command authorithy / (de)activating axis
#                added logging
#                changed the session saving function

#Import of Python available libraries

import os, inspect, sys
current_dir = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))
parent_dir = os.path.dirname(current_dir)
if __name__ == '__main__':
    sys.path.insert(0, parent_dir)

import ctypes
import sys
import threading
import warnings
from io import StringIO
import datetime
import time
import requests
import re
import json

from astropy.time import Time
import astropy.units as u
import numpy as np
import pandas as pd

import logging, websockets, asyncio, socket

from websockets.sync.client import connect as ws_sync_connect

import mke_sculib.chan_list_acu as chans
from mke_sculib.helpers import make_zulustr, match_zulutime, get_utcnow, parse_zulutime

class colors:
    HEADER = '\033[95m'
    OKBLUE = '\033[94m'
    OKGREEN = '\033[92m'
    WARNING = '\033[93m'
    FAIL = '\033[91m'
    BLACK = '\033[0m'
    BOLD = '\033[1m'
    UNDERLINE = '\033[4m'

colors_dc = {
    'header': colors.HEADER,
    'blue': colors.OKBLUE,
    'green': colors.OKGREEN,
    'warn': colors.WARNING,
    'red': colors.FAIL,
    'black': colors.BLACK,
    'bold': colors.BOLD,
    'underline': colors.UNDERLINE
}
    
def print_color(msg, color='red'):
    if isinstance(color, str):
        color = colors_dc[color]
    print(f"{color}{msg}{colors.BLACK}")

    

    
def logfun(name, msg, color):
    t = datetime.datetime.utcnow().isoformat()[:-4].replace('T', ' ') + 'Z'
    print(f'{color}[{t} - {name}] {msg}{colors.BLACK}')

def getLogger(name):
    def printlog(msg, color=colors.BLACK): 
        return logfun(name, msg, color)
    return printlog


log = getLogger('sculib')


configs_dc = {
    'full': chans.channels_detailed,
    'normal': chans.channels_normal,
    'reduced': chans.channels_reduced,
    'small': chans.channels_small,
    'hn_fi': chans.channels_hn_feed_indexer_sensors,
    'hn_tilt': chans.channels_hn_tilt_sensors,
}


state_dc = {
    -1: "UNKNOWN",
    0: "Undefined",
    1: "Standby",
    2: "Parked",
    3: "Locked",
    4: "E-Stop",
    6: "Stowed",
    9: "Locked and Stowed (3+9)",
    10: "Activating",
    19: "Deactivating",
    110: "SIP",
    120: "Stop",
    130: "Slew",
    220: "Jog",
    300: "Track",
}

bands_dc = {'Band 1': 1, 'Band 2': 2, 'Band 3': 3, 'Band 4': 4, 'Band 5a': 5, 'Band 5b': 6, 'Band 5c': 7}
bands_dc_inv = {v:k for k, v in bands_dc.items()}


state_dc = {k:v.upper() for k, v in state_dc.items()}
state_dc_inv = {v:k for k, v in state_dc.items()}

state_dc_inv['DEPLOYING'] = 4
state_dc_inv['DEPLOYED'] = 3
state_dc_inv['RETRACTING'] = 2
state_dc_inv['RETRACTED'] = 1



class DataStreamHandler():

    def __init__(self, parent, channels=None, limit_too_old_s = 0.5, verb=True) -> None:
        assert not parent is None, 'parent can not be None'
        self.parent = parent
        self.t_last_remote = None
        self.t_last_local = None

        self.data = {}
        self.channels = channels
        self.thread = None
        self.limit_too_old_s = limit_too_old_s

        self.tickcount = 0

        self.do_test_on_call = True
        self.do_start_on_call = True

        self.verb=verb

    def wait_for_start(self, timeout=10.):
        if self.verb: log('waiting for data stream to start...', color=colors.HEADER)
        t = time.time()
        while (time.time() - t) < timeout and not self.tickcount:
            time.sleep(0.1)
        if self.verb: log('--> stream started OK', color=colors.OKGREEN)

    def __test(self):
        if not self.do_test_on_call:
            return
        
        if self.tickcount == 0 and self.do_start_on_call:
            self.start()
            self.wait_for_start()

        assert self.thread and self.thread.is_alive(), 'no update thread running!'
        assert not self.t_last_local is None, 'not started or no data received yet (timestamp None)'
        dt = (Time.now() - self.t_last_local).to(u.s).value
        assert dt < self.limit_too_old_s, f'the data stored in the stream object is too old ({dt=} >= {self.limit_too_old_s})'
        assert self.data, 'not started or no data received yet (data empty)'

    def get_all(self, as_dict=False):
        return self.get_many(channels=None, as_dict=as_dict)
    
    def get_many(self, channels=None, as_dict=False):
        self.__test()
        keys = self.channels if channels is None else channels

        if as_dict:
            return {k:self.data[k] for k in keys}
        else:
            return [self.data[k] for k in keys]
        
    def getc(self, channels=None, as_dict=False):
        if isinstance(channels, str):
            return {channels: self[channels]} if as_dict else self[channels]
        else:
            return self.get_many(channels=channels, as_dict=as_dict)

    def __getitem__(self, key):
        self.__test()
        return self.data[key]
    
    def __contains__(self, key):
        self.__test()
        return key in self.data
    
    def is_alive(self):
        return False if self.thread is None else self.thread.is_alive()

    def start(self) -> None:
        if self.verb: log('starting data stream in new thread...', color=colors.HEADER)
        assert not self.parent is None, 'parent can not be None'
        if self.channels is None:
            self.channels = self.parent.get_channel_list()
        assert self.channels, 'channels must be iterable and not empty!'
        
        def statusupdate():
            
            with ws_sync_connect(f'ws://{self.parent.ip}:{self.parent.port}/wsstatus') as ws:
                ws.send(json.dumps(self.channels))
                while True:
                    data = json.loads(ws.recv())
                    self.t_last_local = Time.now()
                    self.tickcount += 1
                    self.data = {k:v[0] for k, v in data['fields'].items()}
                    # data['timestamp'] is the local NTP timestamp of the windows system, not the actual PTP time
                    tptp = self.data.get('acu.time.external_ptp', None)
                    self.t_last_remote = Time(tptp, format='mjd') if tptp else Time(data['timestamp'])


        self.thread = threading.Thread(target=statusupdate)
        self.thread.daemon = True
        self.thread.start()
        if self.verb: log('--> datasteam started...', color=colors.OKGREEN)
        return self.thread
    

    def stop_by_exception(self):
        if self.thread and self.thread.is_alive():
            if self.thread_id is None:
                thread_id = next((id_ for id_, thread in threading._active.items() if thread is self.thread))
            else:
                thread_id = self.thread_id
            res = ctypes.pythonapi.PyThreadState_SetAsyncExc(thread_id,ctypes.py_object(SystemExit))
            if res > 1:
                ctypes.pythonapi.PyThreadState_SetAsyncExc(thread_id, 0)
                raise Exception('Exception raise failure')


class TrackHandler(object):
    def __init__(self, scu, t, az, el, activate_logging, config_name, wait_start=True, verb=False):
        self.scu = scu
        self.args = (t, az, el, activate_logging, config_name)
        self.t_end = Time(t[-1], format='mjd')
        self.verb=verb
        self.wait_start = wait_start

    def get_t_remaining(self):
        """estimate of remaining time in tracking table (in seconds)"""
        return (self.t_end - self.scu.t_internal).to(u.s).value

    def __enter__(self):
        scu = self.scu
        t, az, el, activate_logging, config_name = self.args

        scu.stop_program_track(not self.verb)

        scu.wait_duration(5, not self.verb)
        t_start, t_end = Time(t[0], format='mjd'), Time(t[-1], format='mjd')
        if activate_logging and scu.logger_state() != 'STOPPED':
            log('WARNING, logger already recording - attempting to stop and start a fresh logger...', color=colors.WARNING)
            scu.stop_logger()  
            scu.wait_duration(2, not self.verb)
        
        log(f'Uploading tracking table...', color=colors.OKBLUE)
        dt_wait = scu.upload_track_table(t, az, el, wait_for_start=False)
        t_now = scu.t_internal
        if self.wait_start and (t_start - 3*u.s) > t_now:
            log(f'Waiting for track start time. dt={(t_start - t_now).to(u.s) - 3*u.s}', color=colors.OKBLUE)
            scu.wait_until(t_start - 3*u.s, not self.verb)

        if activate_logging:
            # start logging for my testrun
            scu.start_logger(config_name=config_name, stop_if_need=False)
        
        if self.wait_start and (t_start - 3*u.s) > scu.t_internal:
            scu.wait_until(t_start, not self.verb)
        log(f'Waiting for state to be either TRACK or SLEW...', color=colors.OKBLUE)
        scu.wait_state('acu.general_management_and_controller.state', ['TRACK', "SLEW"], timeout=20, query_delay=.25, operator = 'IN')
        return self
    
    def __exit__(self, *args):
        scu = self.scu
        t, az, el, activate_logging, config_name = self.args
        t_start, t_end = Time(t[0], format='mjd'), Time(t[-1], format='mjd')
        scu.wait_until(t_end, not self.verb)
        scu.wait_for_pos('az', az[-1], tolerance=0.05, timeout=20)
        scu.wait_for_pos('el', el[-1], tolerance=0.05, timeout=20)
        if scu.logger_state() != 'STOPPED':
            scu.stop_logger()
        scu.stop_program_track()
        scu.wait_duration(5, not self.verb)

class scu():
    """A SCU interface object, which connects to a SCU controller by OHB Digital Connect GmbH
    for the SKA-MPI Demonstrator Radio Telescope in the Karroo Desert in South Africa as well
    as the MeerKAT Extension Radio telescopes. 
    
    This class can be used to communicate with and control of the Antennas control unit via 
    HTTP rest API.
    """
    def __init__(self, address='', ip='', port='8080', debug=False, lims_az=(-270.0, 270, 3.0), lims_el=(15, 90, 1.35), dish_type='mke', post_put_delay=0.2, use_socket=True, influx_token=''):
        """create an SCU object, which can be used to communicate with 
        and control of the Antennas control unit via HTTP rest API.

        Args:
            ip (str, optional): ip address of the antenna to connect to. Defaults to 'localhost'.
            port (str, optional): port of the antenna to connect to. Defaults to '8080'.
            debug (bool, optional): Set to True, to receive additional information in stdout, when using commands. Defaults to True.
            lims_az (tuple, optional): limits on AZ axis, angle_min, angle_max, speed_max. Defaults to (-270.0, 270, 3.0).
            lims_el (tuple, optional): limits on EL axis, angle_min, angle_max, speed_max. Defaults to (15, 90, 1.35).
            use_socket: (bool, optional): Set to True, to use async websockets for status updates under the hood. Defaults to False.
        """

        if ip and not address:
            address = ip
            ip = ''
            
        if address:
            if isinstance(ip, int) or ip.isdigit():
                port, ip = ip, ''

            assert not ip, 'can not set address and ip simultanious'

            if 'localhost' in address:
                matches = re.findall(r'(localhost):?([0-9]+)?', address)
            else:
                matches = re.findall(r'([0-9\.]+):?([0-9]+)?', address)

            assert len(matches) > 0, 'given antenna addess does not match any known IP or address pattern'
            matches = matches[0]
            if len(matches) == 1:
                ip, port = matches[0], port
            elif len(matches) == 2:
                ip = matches[0]
                port = matches[1] if matches[1] else port

        self.influx_token = influx_token
        self.ip = ip
        self.port = str(port)
        self.debug = debug

        self.call_log = {}

        self.t_start = Time.now()
        
        self.lims_az = lims_az
        self.lims_el = lims_el
        self.dish_type = dish_type.lower()
        self.bands_possible = bands_dc
        self.post_put_delay = post_put_delay

        self.use_socket = use_socket
        self.data_streamer = DataStreamHandler(self)

    def keys(self):
        if self.use_socket:
            return self.data_streamer.data.keys()
        else:
            return self.get_channel_list(with_values=False, with_timestamps=False)
        
    @property
    def address(self):
        return f'http://{self.ip}:{self.port}'
    
    @property
    def version_acu(self):
        if self.dish_type == 'mke':
            chans = ['acu.general_management_and_controller.ds_software_version_major', 
                    'acu.general_management_and_controller.ds_software_version_minor',
                    'acu.general_management_and_controller.ds_software_version_fix']
            return '.'.join([str(s) for s in self.getc(chans)])
        else:
            return str(self.getc('acu.general_management_and_controller.ds_software_version'))

    @property
    def is_simulator(self):
        """indicates, whether or not this object is a simulator a real antenna

        Returns:
            True if it is a simulator
        """
        return hasattr(self, 'telescope')

    @property
    def t_internal(self):
        """internal telescope time as astropy Time object based on MJD format

        Returns:
            astropy.time.Time: the ACU internal time now
        """
        value = self.getc(f'acu.time.internal_time')
        return Time(value, format='mjd')
        # return Time.now()

    @property
    def spem_keys(self):
        
        if self.dish_type == 'skampi':
            return ["p1_encoder_offset_azimuth", "p2_collimation",
                "p3_non_orthog_nasmyth", "p4_e_w_azimuth_tilt",
                "p5_n_s_azimuth_tilt", "p6_declination_error",
                "p7_encoder_offset_elevation", "p8_cos_terx_gray_nasmyth",
                "p9_sin_term_gray_nasmyth"]

        elif self.dish_type == 'mke':
            return [
                "p1_azimuth_encoder_offset",
                "p3_non_orthog_az_el",
                "p4_collimination_az",
                "p5_n_s_azimuth_tilt",
                "p6_e_w_azimuth_tilt",
                "p7_elevation_encoder_offset",
                "p8_grav_vertical_shift_el_az",
                "p9_linear_scale_factor_el",
                "p11_grav_horizontal_shift_el_az",
                "p12_linear_scale_factor_az",
                "p13_az_frequency_cos_part",
                "p14_az_frequency_sin_part",
                "p15_twice_el_frequency_cos_part",
                "p16_twice_az_frequency_sin_part",
                "p17_twice_az_frequency_cos_part",
                "p18_twice_az_frequency_sin_part",
                ]
        else:
            raise ValueError(f'{self.dish_type} is an unknon dish type for SPEM models, only skampi and mke are allowed')
    
    @property
    def spem_param_name_map(self):
        param_name_map = [re.match(r"^[Pp][0-9]+", k) for k in self.spem_keys]
        param_name_map = {m.group().upper():k for m, k in zip(param_name_map, self.spem_keys)}
        return param_name_map
    
    @property
    def stow_pos(self):
        az = 0.0 if self.dish_type == 'mke' else -90.0
        el = 89.75
        return az, el

    @property
    def band_in_focus(self):
        return self.getc('acu.general_management_and_controller.feed_indexer_pos')

    @property
    def azel(self):
        return self.get_azel()
    
    @property
    def state(self):
        return self.get_state()
    
    @property
    def azel_setp(self):
        return tuple(self.getc(['acu.azimuth.p_set', 'acu.elevation.p_set']))
    
    @property
    def azel_shape(self):
        return tuple(self.getc(['acu.azimuth.p_shape', 'acu.elevation.p_shape']))

    @property
    def acu_eventlog(self):
        return self.get_events(as_pandas=True)
    
    @property
    def events(self):
        return self.get_events(as_pandas=True)

    def getc(self, key, as_dict=False):
        """get one or many channel values. key(s) must be string or list of strings

        Args:
            key (string or iterable[str]): the channel names to get

        Returns:
            dict, list or immuteable: type depending on the input.
        """
        try:
            if self.use_socket:
                return self.data_streamer.getc(key, as_dict=as_dict)
        except Exception as err:
            log(f'error while getting data from socket stream... falling back to HTTP api. {err=}', color=colors.WARNING)

        if key is None:
            return dict(self.get_channel_list(with_values=True, with_timestamps=False))
        elif not isinstance(key, str) and hasattr(self, '__len__'):
            allchans = dict(self.get_channel_list(with_values=True, with_timestamps=False))
            if as_dict:
                return {k:allchans[k] for k in key}
            else:
                return [allchans[k] for k in key]
        else:
            return self._get_device_status_value(key)


    def _limit_motion(self, az_pos, el_pos, az_speed=None, el_speed=None):
        def limit(name, x, xmin, xmax):
            if x is None:
                return x
            vlim = max(min(x, xmax), xmin)
            if vlim != x:
                txt = f"WARNING: variable {name} exceeds its allowed limit and was set to {vlim} from {x}"
                log(txt)
                warnings.warn(txt)
            return vlim
        
        angle_min, angle_max, speed_max = self.lims_az
        az_pos = limit('az_pos', az_pos, angle_min, angle_max)
        az_speed = limit('az_speed', az_speed, 1e-10, speed_max)

        angle_min, angle_max, speed_max = self.lims_el
        el_pos = limit('el_pos', el_pos, angle_min, angle_max)
        el_speed = limit('el_speed', el_speed, 1e-10, speed_max)
        return az_pos, el_pos, az_speed, el_speed


    #Direct SCU webapi functions based on urllib PUT/GET
    def _feedback(self, r):
        if self.debug == True:
            log('***Feedback:' +  str(r.request.url) + ' ' + str(r.request.body))
            log(f'{r.status_code}: {r.reason}')
            log("***Text returned:")
            log(r.text)
        elif r.status_code != 200:
            log('***Feedback:' +  str(r.request.url) + ' ' + str(r.request.body), color=colors.WARNING)
            log(f'{r.status_code}: {r.reason}', color=colors.WARNING)
            log("***Text returned:", color=colors.WARNING)
            log(r.text, color=colors.WARNING)
            #log(r.reason, r.status_code)
            #log()

    def ping(self, timeout=10):
        URL = 'http://' + self.ip + ':' + self.port + '/devices/'
        r = requests.get(url = URL, timeout=timeout)
        return r.status_code == 200
    
    #	def scu_get(device, params = {}, r_ip = self.ip, r_port = port):
    def scu_get(self, device, params = {}):
        '''This is a generic GET command into http: scu port + folder 
        with params=payload'''
        URL = 'http://' + self.ip + ':' + self.port + device
        if device != '/devices/statusValue':
            self.call_log[datetime.datetime.utcnow()] = 'GET | ' + device

        if self.debug == True:
            log(f'request.get(**{dict(url = URL, params = params)}):')

        r = requests.get(url = URL, params = params)
        self._feedback(r)
        r.raise_for_status()
        if r.status_code != 200:
            log(f'Statuscode != 200. Returnded: {r.status_code}: {r.reason}', color=colors.WARNING)

        return r

    async def scu_get_async(self, devices, params):
        '''This is a generic GET command into http: scu port + folder 
        with params=payload fur multiple devices and params at the same time'''
        url = self.address
        if not 'aiohttp' in sys.modules:
            import aiohttp

        async with aiohttp.ClientSession() as session:                
            async def get(device, params):
                async with session.get(url + device, params=params) as r:
                    txt = await r.text()
                    r.raise_for_status()
                    return json.loads(txt)
                    
            return await asyncio.gather(*(get(d, p) for d, p in zip(devices, params)))

    def get_errors(self, warnings=True):
        if warnings:
            fun = lambda k, v: ('warn' in k.lower() or 'err' in k.lower()) and v and 'not_used' not in k and 'limit' not in k and 'stowpins' not in k and v
            return {k:v for k, v in self.getc(as_dict=True).items() if fun(k, v)}

        return {k:v for k, v in self.getc(as_dict=True).items() if 'err' in k.lower() and v}
    
    def scu_get_concurrent(self, devices, params):
        loop = asyncio.get_event_loop()
        return loop.run_until_complete(self.scu_get_async(devices, params))
            
        
    def scu_put(self, device, payload = {}, params = {}, data=''):
        '''This is a generic PUT command into http: scu port + folder 
        with json=payload'''
        URL = 'http://' + self.ip + ':' + self.port + device
        self.call_log[datetime.datetime.utcnow()] = 'PUT | ' + device
        if self.debug == True:
            log(f'request.put(**{dict(url = URL, json = payload, params = params, data = data)}):')

        r = requests.put(url = URL, json = payload, params = params, data = data)
        self._feedback(r)
        r.raise_for_status()
        if r.status_code != 200:
            log(f'Statuscode != 200. Returnded: {r.status_code}: {r.reason}', color=colors.WARNING)

        if self.post_put_delay > 0:
            self.wait_duration(self.post_put_delay, no_stout=True)

        return r

    def scu_delete(self, device, payload = {}, params = {}):
        '''This is a generic DELETE command into http: scu port + folder 
        with params=payload'''
        URL = 'http://' + self.ip + ':' + self.port + device
        self.call_log[datetime.datetime.utcnow()] = 'DEL | ' + device
        if self.debug == True:
            log(f'request.delete(**{dict(url = URL, json = payload, params = params)}):')

        r = requests.delete(url = URL, json = payload, params = params)
        self._feedback(r)
        r.raise_for_status()

        if r.status_code != 200:
            log(f'Statuscode != 200. Returnded: {r.status_code}: {r.reason}', color=colors.WARNING)
        
        if self.post_put_delay > 0:
            self.wait_duration(self.post_put_delay, no_stout=True)

        return r


    def determine_dish_type(self):
        """will set and return the dish type by checking a status value 

        Returns:
            string: either 'skampi' or 'mke'
        """
        chans = self.scu_get("/devices/statusPaths").json()
        if 'acu.general_management_and_controller.ds_software_version' in chans:
            self.dish_type = 'skampi'
        else:
            self.dish_type == 'mke'
        return self.dish_type
            

    def get_all_channels(self):
        return self.get_channel_list(with_values=True, with_timestamps=True)
    
    def __getitem__(self, key):
        return self.getc(key)
    
    
    def get_events(self, nlast=1000, as_pandas=True):
        records = self.scu_get('/events/lastn', params=dict(n=nlast)).json()
        if as_pandas:
            df = pd.DataFrame(records)
            df['time'] = pd.to_datetime(df['timestamp'], utc=True)
            df.set_index('time', inplace=True, drop=True)
            return df
        else:
            return records
        
    #SIMPLE PUTS

    #commands to DMC state - dish management controller
    def interlock_acknowledge_dmc(self):
        """Send an interlock acknowledge command to the digital motion controller in case
        of trying to acknowledge errors etc.
        """
        log('Acknowledge interlock...')
        self.scu_put('/devices/command',
            {'path': 'acu.dish_management_controller.interlock_acknowledge'})

    def reset_dmc(self):
        """reset the digital motion controller in case of errors"""
        log('reset dmc...')
        self.scu_put('/devices/command', 
            {'path': 'acu.dish_management_controller.reset'})

    def activate_dmc(self):
        """activate the digital motion controller"""
        log('activate dmc...')
        self.scu_put('/devices/command',
            {'path': 'acu.dish_management_controller.activate'})

    def deactivate_dmc(self):
        """deactivate the digital motion controller"""
        log('deactivate dmc')
        self.scu_put('/devices/command', 
            {'path': 'acu.dish_management_controller.deactivate'})
        
    def move_to_band(self, position):
        """move the feed indexer to a predefined band position
        options str: "Band 1", "Band 2", "Band 3", "Band 5a", "Band 5b"
        "Band 5c"
        Args:
            position (str or int): Either "Band 1"..."Band 5c" or 1...7
        """

        log('move to band:' + str(position))
        if not(isinstance(position, str)):
            self.scu_put('/devices/command',
            {'path': 'acu.dish_management_controller.move_to_band',
            'params': {'action': position}})
        else:
            self.scu_put('/devices/command',
            {'path': 'acu.dish_management_controller.move_to_band',
            'params': {'action': bands_dc[position]}})

    def get_azel(self, concurrent=False):
        """gets the current az and el values in degree and returns them as tuple
        """
        if concurrent:
            az_dc, el_dc = self.get_device_status_value_async(['acu.azimuth.p_act', 'acu.elevation.p_act'])
            return az_dc['value'], el_dc['value']
        else:
            return tuple(self.getc(['acu.azimuth.p_act', 'acu.elevation.p_act']))
    

    def get_state(self, path='acu.general_management_and_controller.state'):
        alias_dc = {    
            'gmc': 'acu.general_management_and_controller.state',
            'az': 'acu.azimuth.state',
            'el': 'acu.elevation.state',
            'fi': 'acu.feed_indexer.state',
        }
        
        if path in alias_dc:
            path = alias_dc[path]

        return self.status_finalValue(path)
    
    def move_to_azel(self, az_angle, el_angle, az_vel=None, el_vel=None):
        """synonym for abs_azel. Moves to an absolute az el position
        with a preset slew rate
        Args:
            az_angle (-270 <= az_angle <= 270): abs AZ angle in degree.
            el_angle (15 <= el_angle <= 90): abs EL angle in degree.
            az_vel (0 < az_vel <= 3.0): AZ angular slew rate in degree/s. None for as fast as possible.
            el_vel (0 < el_vel <= 1.35): EL angular slew rate in degree/s. None for as fast as possible.
        """
        if az_vel is None and el_vel is None: 
            self.abs_azel(az_angle, el_angle)
        else:
            assert all([v is not None for v in [az_angle, el_angle, az_vel, el_vel]]), 'inputs can not be None'
            log('abs az: {:.4f} el: {:.4f} (vels: ({:.4f}, {:.4f})'.format(az_angle, el_angle, az_vel, el_vel))
            assert (az_vel is None) == (el_vel is None), 'either both velocities must be None, or neither'

            az_angle, el_angle, az_vel, el_vel = self._limit_motion(az_angle, el_angle, az_vel, el_vel)

            self.scu_put('/devices/command',
                {'path': 'acu.azimuth.slew_to_abs_pos',
                'params': {'new_axis_absolute_position_set_point': az_angle,
                'new_axis_speed_set_point_for_this_move': az_vel}})    

            self.scu_put('/devices/command',
                {'path': 'acu.elevation.slew_to_abs_pos',
                'params': {'new_axis_absolute_position_set_point': el_angle,
                'new_axis_speed_set_point_for_this_move': el_vel}}) 



    def abs_azel(self, az_angle, el_angle):
        """move to a given absolut position on both axes

        Args:
            az_angle (-270 <= az_angle <= 270): abs AZ angle in degree.
            el_angle (15 <= el_angle <= 90): abs EL angle in degree.
        """
        log('abs az: {:.4f} el: {:.4f}'.format(az_angle, el_angle))

        az_angle, el_angle, _, _ = self._limit_motion(az_angle, el_angle)

        self.scu_put('/devices/command',
            {'path': 'acu.dish_management_controller.slew_to_abs_pos',
            'params': {'new_azimuth_absolute_position_set_point': az_angle,
                'new_elevation_absolute_position_set_point': el_angle}})

    def wait_track_start(self, timeout=600, query_delay=.25):
        """wait for a tracing table to start by waiting for the two axis to change 
        to state 'TRACK'

        Args:
            timeout (int, optional): timeout time in seconds. Defaults to 600.
            query_delay (float, optional): period between two checks if status has changed in seconds. Defaults to .25.
        """
        self.wait_state("acu.azimuth.state", "TRACK", timeout, query_delay)
        self.wait_state("acu.elevation.state", "TRACK", timeout, query_delay)

    # def wait_track_end(self, timeout=600, query_delay=1.):
    #     # This is to allow logging to continue until the track is completed
    #     log('Waiting for track to finish...')

    #     self.wait_duration(10.0, no_stout=True)  

    #     def tester():
    #         a = self.status_Value("acu.tracking.act_pt_end_index_a")
    #         b = self.status_Value("acu.tracking.act_pt_act_index_a")
    #         return (int(a) - int(b)) > 0
        

    #     self.wait_by_testfun(tester, timeout, query_delay)
    #     log('   -> done')

    def wait_settle(self, axis='all', timeout=600, query_delay=.25, tolerance=0.01, wait_by_pos=True, initial_delay=2.0):
        """
        alias for waitForStatusValue but mapping 'AZ', 'EL', 'FI' to 'acu.azimuth.p_act'
        'acu.elevation.p_act' and 'acu.feed_indexer.p_act'

        Periodically queries a device status 'path' until a specific value is reached.

        Args:
            path:       path of the SCU device status
            Value:      value to be reached
            timeout:    Raise TimeoutError after this duration
            query_delay: Period in seconds to wait between two queries.
        """
        
        if initial_delay > 0:
            # assures setpoint has actually been send to acu!
            self.wait_duration(initial_delay, no_stout=True)  

        dc1 = {
            'az': 'azimuth', 'el': 'elevation', 'fi': 'feed_indexer', 
            'azimuth':'azimuth', 'elevation': 'elevation', 'feed_indexer': 'feed_indexer' }
        
        if axis == 'all':
                    
            self.wait_settle('az', initial_delay=0.0)
            self.wait_settle('el', initial_delay=0.0)
            self.wait_settle('fi', initial_delay=0.0)
            self.wait_duration(0.5)

            return

        key = dc1[axis.lower()]


        if key == 'feed_indexer':
            path = 'acu.feed_indexer.state'
            self.wait_state(path, "SIP", timeout, query_delay, operator = '<=')
        else:
            
            if timeout is None:
                p_set = self.getc(f'acu.{key}.p_set')
                p_act = self.getc(f'acu.{key}.p_act')
                # print(p_set, p_act)
                p_diff = float(p_act) - float(p_set)

                if key == 'azimuth':
                    # these numbers are from a stepping test done on the MKE Simulator mid August 2023
                    p = [0.3335957 , 8.95694938] # s/° + s
                elif key == 'elevation':
                    # these numbers are from a stepping test done on the MKE Simulator mid August 2023
                    p = [0.7274147 , 5.90958871] # s/° + s

                dt = p_diff * p[0] + p[1]
                timeout = dt + 10
        
        
            if wait_by_pos:
                value = self.getc(f'acu.{key}.p_set')
                self.wait_for_pos(key, value, timeout, query_delay, tolerance)
            else:
                path = f'acu.{key}.state'
                self.wait_state(path, ['Standby', "Parked", "SIP", "TRACK"], timeout, query_delay, operator = 'IN')


    def wait_for_pos(self, axis, value, timeout=600, query_delay=.25, tolerance=None):
        """
        alias for waitForStatusValue but mapping 'AZ', 'EL', 'FI' to 'acu.azimuth.p_act'
        'acu.elevation.p_act' and 'acu.feed_indexer.p_act'

        Periodically queries a device status 'path' until a specific value is reached.

        Args:
            path:       path of the SCU device status
            Value:      value to be reached
            timeout:    Raise TimeoutError after this duration
            query_delay: Period in seconds to wait between two queries.
        """
        dc1 = {
            'az': 'azimuth', 'el': 'elevation', 'fi': 'feed_indexer', 
            'azimuth':'azimuth', 'elevation': 'elevation', 'feed_indexer': 'feed_indexer' }
        
        key = dc1[axis.lower()]
    
        if tolerance is not None:
            path = f'acu.{key}.p_shape'
            tester = lambda v: abs(v - value) < abs(tolerance)
        else:
            path = f'acu.{key}.p_shape'
            tester = lambda v: v == value

        self.wait_for_status(path, value, tester, timeout, query_delay, tolerance)


    def wait_state(self, path, value=None, timeout=600, query_delay=.5, operator = '=='):    
        """Wait for a given status at a given path using a given operator

        Args:
            path (str): the device path. Example
            value (str or int): the value to wait for
            timeout (int, optional): time in seconds after which to raise an timeout. Defaults to 600.
            query_delay (float, optional): re-query period for checking status change. Defaults to .25.
            operator (str, optional): optional operator to give '==', '!=', '<', '>' '>=', '<='. Defaults to '=='.
        """
        if value is None and path.upper() in state_dc_inv:
            value = path.upper()
            path = 'acu.general_management_and_controller.state'

        if isinstance(value, (int, str)):
            val = state_dc_inv.get(value, value)
        elif hasattr(value, '__len__'):
            val = [state_dc_inv.get(v.upper() if isinstance(v, str) else v, v) for v in value]
            errs = [v for v in val if not isinstance(v, int)]
            assert not any(errs), f'wait_state got unrecognized state: {errs}'
        else:
            val = value

        def tester(v):
            if  isinstance(v, int) or v.isnumeric():
                vv = int(v)    
            else:
                vv = state_dc_inv[v.upper()]

            if operator == '==':   return vv == val
            elif operator == '!=': return vv != val
            elif operator == '<':  return vv <  val
            elif operator == '<=': return vv <= val
            elif operator == '>':  return vv >  val
            elif operator == '>=': return vv >= val
            elif operator == 'IN': return vv in val
            else: raise Exception(str(operator) + ' is not recognized as a valid operator. Allowed are only ==, !=, <=, >=, <, >')

        self.wait_for_status(path, val, tester, timeout, query_delay)


    def waitForStatusValue(self, path, value, timeout=600, query_delay=.25, tolerance=None):
        """
        alias for wait_for_status
        queries a device status 'path' until a specific value is reached.

        Args:
            path:       path of the SCU device status
            Value:      value to be reached
            timeout:    Raise TimeoutError after this duration
            query_delay: Period in seconds to wait between two queries.
        """
        self.wait_for_status(path, value, None, timeout=timeout, query_delay=query_delay, tolerance=tolerance)


    def wait_by_testfun(self, tester, timeout=600, query_delay=1.0, no_stout=False):
        """
        Periodically queries a device status 'path' until a specific value is reached.

        Args:
            path:       path of the SCU device status
            tester:     any test function that returns true, when reached
            timeout:    Raise TimeoutError after this duration
            query_delay: Period in seconds to wait between two queries.
        """
        starttime = time.time()

        self.wait_duration(0.5, no_stout=True)    

        is_first = True
        while time.time() - starttime < timeout:            
            if tester():
                if not no_stout and not is_first:
                    log('  -> done', color=colors.OKBLUE)
                return True
            
            if not no_stout and is_first:
                log('waiting for tester to return true...', color=colors.OKBLUE)
                is_first = False
            self.wait_duration(query_delay, no_stout=True)  

        err = "Sensor: tester() not true after {}s".format(timeout)
        log(err, color=colors.FAIL)
        raise TimeoutError(err)


    def wait_for_status(self, path, value, tester = None, timeout=600, query_delay=.25, tolerance=None, no_stout=False):
        """
        Periodically queries a device status 'path' until a specific value is reached.

        Args:
            path:       path of the SCU device status
            Value:      value to be reached
            timeout:    Raise TimeoutError after this duration
            query_delay: Period in seconds to wait between two queries.
        """
        starttime = time.time()

        self.wait_duration(0.5, no_stout=True)    

        is_first = True
        v = 'UNKNOWN'
        
        while time.time() - starttime < timeout:
            v = self.getc(path)
            
            if tester is not None and tester(v):
                if not no_stout and not is_first:
                    log('  -> done', color=colors.OKBLUE)
                return True
            elif tester is None and tolerance is not None and abs(v - value) < abs(tolerance): 
                if not no_stout and not is_first:
                    log('  -> done', color=colors.OKBLUE)
                return True
            elif tester is None and v == value:
                if not no_stout and not is_first:
                    log('  -> done', color=colors.OKBLUE)
                return True

            if not no_stout and is_first:
                if isinstance(value, float):
                    log('wait for {}: {:.3f} (currently at: {:.3f})'.format(path, value, v), color=colors.OKBLUE)
                elif 'state' in path:
                    if not isinstance(value, str) and hasattr(value, '__len__'):
                        vv = [state_dc.get(vvv, vvv) for vvv in value]
                    else:
                        vv = state_dc.get(value, value)
                    log('wait for {}: {} (currently at: {})'.format(path, vv, state_dc.get(v, v)), color=colors.OKBLUE)
                else:
                    log('wait for {}: {} (currently at: {})'.format(path, value, v), color=colors.OKBLUE)

                is_first = False

            self.wait_duration(query_delay, no_stout=True)

        if 'state' in path:
            v = self.getc(path)
            v = (state_dc.get(v, v), v)
            if not isinstance(value, str) and hasattr(value, '__len__'):
                vv = [(state_dc.get(vvv, vvv), vvv) for vvv in value]
            else:
                vv = (state_dc.get(value, value), value)
            err = "Sensor: {} not equal to {} after {}s. Current value: {}".format(path, vv, timeout, v)
        else:
            err = "Sensor: {} not equal to {} after {}s. Current value: {}".format(path, value, timeout, v)

        log(err, color=colors.FAIL)
        raise TimeoutError(err)

    def get_device_status_value_async(self, pathes):
        params = [{"path": path} for path in pathes]
        devices = ["/devices/statusValue"] * len(params)
        return self.scu_get_concurrent(devices, params)


    def _get_device_status_value(self, path):
        """
        Gets one or many device status values (status now)

        Args:
            path:       path of the SCU device status as string, or a list of strings for many
        returns:
            either the value directly or a list of values in case of a list of pathes
        """

        if not isinstance(path, str):
            fun = lambda p: self.scu_get("/devices/statusValue", {"path": p}).json()['value']
            return [fun(p) for p in path]
        else:
            return self.scu_get("/devices/statusValue", {"path": path}).json()['value']
            
    def get_device_status_message(self, path, value_only=False):
        """
        Gets one or many device status fields (which is the value and additional information)

        Args:
            path:       path of the SCU device status as string, or a list of strings for many
        returns:
            either the status message dict directly or a list of dicts in case of a list of pathes
        """
        if not isinstance(path, str):
            if value_only:
                fun = lambda p: self.scu_get("/devices/statusMessageField", {"path": p}).json()['lastFinalValue']
            else:
                fun = lambda p: self.scu_get("/devices/statusMessageField", {"path": p}).json()
            return [fun(p) for p in path]
        else:
            if value_only:
                return self.scu_get("/devices/statusMessageField", {"path": path}).json()['lastFinalValue']
            else:
                return self.scu_get("/devices/statusMessageField", {"path": path}).json()
        
    def get_channel_list(self, with_values=False, with_timestamps=False):
        if self.dish_type == 'mke':
            lst = self.scu_get("/devices/getAllDeviceStatusValues", {"device":"acu"}).json()
            
            if with_values and with_timestamps:
                fun = lambda v: ('acu.' + v['path'], v['values'][0]['timestamp'], v['values'][0]['lastValue'])
            elif not with_values and with_timestamps:
                fun = lambda v: ('acu.' + v['path'], v['values'][0]['timestamp'])
            elif with_values and not with_timestamps:
                fun = lambda v: ('acu.' + v['path'], v['values'][0]['lastValue'])
            else:
                fun = lambda v: 'acu.' + v['path']
            
            return [fun(v) for v in lst if v]
        else:
            if with_values or with_timestamps:
                raise ValueError('skampi type dish can not be queried for values, only for channel names')
            return self.scu_get("/devices/statusPaths").json()
        
    #commands to ACU
    def deactivate_lowpowermode(self):
        return self.scu_put("/devices/command", {"path":"acu.dish_management_controller.set_power_mode","params":{"action":"0"}}).text
    
    def activate_lowpowermode(self):
        return self.scu_put("/devices/command", {"path":"acu.dish_management_controller.set_power_mode","params":{"action":"1"}}).text

    def stow(self, pre_move=True, nowait=False):
        """stow the antenna on pos 1 (both axes) and wait for stowing to be completed
        """
        log('Stowing...')


        if self.get_state().upper() != 'STOWED':

            if pre_move:
                self.abs_azel(*self.stow_pos)
                self.wait_duration(0.5)
                self.wait_settle()
                self.wait_duration(1.0)

            if self.dish_type == 'mke':
                self.scu_put("/devices/command", {"path": "acu.dish_management_controller.stow", "params": {"az_stow": "1", "el_stow": "1"}})
            else:
                self.scu_put("/devices/command", {"path": "acu.dish_management_controller.stow", "params": {"action": "1"}})

        if not nowait:
            self.wait_state("acu.stow_pin_controller.azimuth_status", "DEPLOYED", operator='==')
            self.wait_state("acu.stow_pin_controller.elevation_status", "DEPLOYED", operator='==')
            self.wait_duration(1, no_stout=True)  
        
    def unstow(self, nowait=False):
        """
        Unstow both axes
        """
        log('Unstowing...')
        self.scu_put("/devices/command", {"path": "acu.dish_management_controller.unstow"})


        if not nowait:
            self.wait_duration(3, no_stout=True)      
            self.wait_state("acu.stow_pin_controller.azimuth_status", "RETRACTED", operator='==')
            self.wait_state("acu.stow_pin_controller.elevation_status", "RETRACTED", operator='==')
            self.wait_duration(1, no_stout=True)  


    def activate_axes(self):
        """
        Activate axes
        """
        self.scu_put("/devices/command", {"path": "acu.azimuth.activate"})
        self.scu_put("/devices/command", {"path": "acu.elevation.activate"})

        self.wait_duration(1, no_stout=True)
        self.waitForStatusValue("acu.azimuth.axis_bit_status.abs_active", True, timeout=15)
        self.waitForStatusValue("acu.elevation.axis_bit_status.abs_active", True, timeout=15)


    def deactivate_axes(self):
        """
        Activate axes
        """
        self.scu_put("/devices/command", {"path": "acu.azimuth.deactivate"})
        self.scu_put("/devices/command", {"path": "acu.elevation.deactivate"})

        self.wait_duration(1, no_stout=True)        
        self.waitForStatusValue("acu.azimuth.axis_bit_status.abs_active", False, timeout=15)
        self.waitForStatusValue("acu.elevation.axis_bit_status.abs_active", False, timeout=15)


    def release_command_authority(self):
        """
        Release command authority.
        """
        log('Releasing Command Authority...')
        self._command_authority('Release')
        self.wait_duration(5)
    
    def get_command_authority(self):
        """
        get command authority.
        """
        log('Getting Command Authority...')


        self._command_authority('Get')
        # # ICD Version 2.4 says 4, but actual behavior of 21.07.2021 is value 3 == SCU
        # self.wait_for_status("acu.command_arbiter.act_authority", "3", timeout=10)
        self.wait_duration(5)
        
    #command authority
    def _command_authority(self, action):
        #1 get #2 release
        
        authority={'Get': 1, 'Release': 2}
        self.scu_put('/devices/command', 
            {'path': 'acu.command_arbiter.authority',
            'params': {'action': authority[action]}})
        
    def activate_az(self):
        """activate azimuth axis (controller)"""
        log('act azimuth')
        self.scu_put('/devices/command', 
            {'path': 'acu.elevation.activate'})

    def activate_el(self):
        """activate elevation axis (controller)"""
        log('activate elevation')
        self.scu_put('/devices/command', 
            {'path': 'acu.elevation.activate'})

    def deactivate_el(self):
        """deactivate elevation axis (controller)"""
        log('deactivate elevation')
        self.scu_put('/devices/command', 
            {'path': 'acu.elevation.deactivate'})

    def abs_azimuth(self, az_angle, az_vel):
        """Moves to an absolute az position
        with a preset slew rate
        Args:
            az_angle (-270 <= az_angle <= 270): abs AZ angle in degree.
            az_vel (0 <= az_vel <= 3.0): AZ angular slew rate in degree/s.
        """
        
        log('abs az: {:.4f} vel: {:.4f}'.format(az_angle, az_vel))
        az_vel = self.lims_az[-1] if az_vel is None else az_vel
        az_angle, _, az_vel, _ = self._limit_motion(az_angle, None, az_vel, None)
        self.scu_put('/devices/command',
            {'path': 'acu.azimuth.slew_to_abs_pos',
            'params': {'new_axis_absolute_position_set_point': az_angle,
            'new_axis_speed_set_point_for_this_move': az_vel}})    

    def abs_elevation(self, el_angle, el_vel):
        """Moves to an absolute el position
        with a preset slew rate
        Args:
            az_vel (0 <= az_vel <= 3.0): AZ angular slew rate in degree/s.
            el_vel (0 <= el_vel <= 3.0): EL angular slew rate in degree/s.
        """

        log('abs el: {:.4f} vel: {:.4f}'.format(el_angle, el_vel))

        el_vel = self.lims_el[-1] if el_vel is None else el_vel
        _, el_angle, _, el_vel = self._limit_motion(None, el_angle, None, el_vel)

        self.scu_put('/devices/command',
            {'path': 'acu.elevation.slew_to_abs_pos',
            'params': {'new_axis_absolute_position_set_point': el_angle,
            'new_axis_speed_set_point_for_this_move': el_vel}}) 

    def load_static_offset(self, az_offset, el_offset):
        """loads a static offset to the tracking controller

        Args:
            az_offset (float): the AZ offset to load. Unit Unclear!
            el_offset (float): the EL offset to load. Unit Unclear!
        """
        log('offset az: {:.4f} el: {:.4f}'.format(az_offset, el_offset))
        self.scu_put('/devices/command',
            {'path': 'acu.tracking_controller.load_static_tracking_offsets.',
            'params': {'azimuth_tracking_offset': az_offset,
                        'elevation_tracking_offset': el_offset}})     #Track table commands


    
    def load_program_track(self, load_type, entries, t=[0]*50, az=[0]*50, el=[0]*50):
        """WARNING DEPRECATED! please use upload_track_table instead
        load a program track table to the ACU of exactly 50 entries

        Args:
            load_type (str): either 'LOAD_NEW', 'LOAD_ADD', or 'LOAD_RESET'
            entries (int): number of entries in track table
            t (list of float, optional): time values to upload. Defaults to [0]*50.
            az (list of float, optional): az values to upload. Defaults to [0]*50.
            el (list of float, optional): el values to upload. Defaults to [0]*50.
        """

        warnings.warn(
            "Legacy Function: This is not maintained and might not work as intended\nWARNING DEPRECATED! please use upload_track_table instead",
            DeprecationWarning
        )

        log(load_type)    
        LOAD_TYPES = {
            'LOAD_NEW' : 1, 
            'LOAD_ADD' : 2, 
            'LOAD_RESET' : 3}
        
        #table selector - to tidy for future use
        ptrackA = 11
        
        TABLE_SELECTOR =  {
            'pTrackA' : 11,
            'pTrackB' : 12,
            'oTrackA' : 21,
            'oTrackB' : 22}
        
        #funny thing is SCU wants 50 entries, even for LOAD RESET! or if you send less then you have to pad the table
    
        if entries != 50:
            padding = 50 - entries
            t  += [0] * padding
            az += [0] * padding
            el += [0] * padding

        self.scu_put('/devices/command',
                    {'path': 'acu.dish_management_controller.load_program_track',
                    'params': {'table_selector': ptrackA,
                                'load_mode': LOAD_TYPES[load_type],
                                'number_of_transmitted_program_track_table_entries': entries,
                                'time_0': t[0], 'time_1': t[1], 'time_2': t[2], 'time_3': t[3], 'time_4': t[4], 'time_5': t[5], 'time_6': t[6], 'time_7': t[7], 'time_8': t[8], 'time_9': t[9], 'time_10': t[10], 'time_11': t[11], 'time_12': t[12], 'time_13': t[13], 'time_14': t[14], 'time_15': t[15], 'time_16': t[16], 'time_17': t[17], 'time_18': t[18], 'time_19': t[19], 'time_20': t[20], 'time_21': t[21], 'time_22': t[22], 'time_23': t[23], 'time_24': t[24], 'time_25': t[25], 'time_26': t[26], 'time_27': t[27], 'time_28': t[28], 'time_29': t[29], 'time_30': t[30], 'time_31': t[31], 'time_32': t[32], 'time_33': t[33], 'time_34': t[34], 'time_35': t[35], 'time_36': t[36], 'time_37': t[37], 'time_38': t[38], 'time_39': t[39], 'time_40': t[40], 'time_41': t[41], 'time_42': t[42], 'time_43': t[43], 'time_44': t[44], 'time_45': t[45], 'time_46': t[46], 'time_47': t[47], 'time_48': t[48], 'time_49': t[49],
                                'azimuth_position_0': az[0], 'azimuth_position_1': az[1], 'azimuth_position_2': az[2], 'azimuth_position_3': az[3], 'azimuth_position_4': az[4], 'azimuth_position_5': az[5], 'azimuth_position_6': az[6], 'azimuth_position_7': az[7], 'azimuth_position_8': az[8], 'azimuth_position_9': az[9], 'azimuth_position_10': az[10], 'azimuth_position_11': az[11], 'azimuth_position_12': az[12], 'azimuth_position_13': az[13], 'azimuth_position_14': az[14], 'azimuth_position_15': az[15], 'azimuth_position_16': az[16], 'azimuth_position_17': az[17], 'azimuth_position_18': az[18], 'azimuth_position_19': az[19], 'azimuth_position_20': az[20], 'azimuth_position_21': az[21], 'azimuth_position_22': az[22], 'azimuth_position_23': az[23], 'azimuth_position_24': az[24], 'azimuth_position_25': az[25], 'azimuth_position_26': az[26], 'azimuth_position_27': az[27], 'azimuth_position_28': az[28], 'azimuth_position_29': az[29], 'azimuth_position_30': az[30], 'azimuth_position_31': az[31], 'azimuth_position_32': az[32], 'azimuth_position_33': az[33], 'azimuth_position_34': az[34], 'azimuth_position_35': az[35], 'azimuth_position_36': az[36], 'azimuth_position_37': az[37], 'azimuth_position_38': az[38], 'azimuth_position_39': az[39], 'azimuth_position_40': az[40], 'azimuth_position_41': az[41], 'azimuth_position_42': az[42], 'azimuth_position_43': az[43], 'azimuth_position_44': az[44], 'azimuth_position_45': az[45], 'azimuth_position_46': az[46], 'azimuth_position_47': az[47], 'azimuth_position_48': az[48], 'azimuth_position_49': az[49],
                                'elevation_position_0': el[0], 'elevation_position_1': el[1], 'elevation_position_2': el[2], 'elevation_position_3': el[3], 'elevation_position_4': el[4], 'elevation_position_5': el[5], 'elevation_position_6': el[6], 'elevation_position_7': el[7], 'elevation_position_8': el[8], 'elevation_position_9': el[9], 'elevation_position_10': el[10], 'elevation_position_11': el[11], 'elevation_position_12': el[12], 'elevation_position_13': el[13], 'elevation_position_14': el[14], 'elevation_position_15': el[15], 'elevation_position_16': el[16], 'elevation_position_17': el[17], 'elevation_position_18': el[18], 'elevation_position_19': el[19], 'elevation_position_20': el[20], 'elevation_position_21': el[21], 'elevation_position_22': el[22], 'elevation_position_23': el[23], 'elevation_position_24': el[24], 'elevation_position_25': el[25], 'elevation_position_26': el[26], 'elevation_position_27': el[27], 'elevation_position_28': el[28], 'elevation_position_29': el[29], 'elevation_position_30': el[30], 'elevation_position_31': el[31], 'elevation_position_32': el[32], 'elevation_position_33': el[33], 'elevation_position_34': el[34], 'elevation_position_35': el[35], 'elevation_position_36': el[36], 'elevation_position_37': el[37], 'elevation_position_38': el[38], 'elevation_position_39': el[39], 'elevation_position_40': el[40], 'elevation_position_41': el[41], 'elevation_position_42': el[42], 'elevation_position_43': el[43], 'elevation_position_44': el[44], 'elevation_position_45': el[45], 'elevation_position_46': el[46], 'elevation_position_47': el[47], 'elevation_position_48': el[48], 'elevation_position_49': el[49]}})



    # def start_program_track(self, start_time):
    #     """Start a previously loaded tracking table (Table A) using SPLINE interpolation and AZ EL tracking Mode

    #     Args:
    #         start_time (float): start time as MJD
    #     """
    #     ptrackA = 11
    #     #interpol_modes
    #     NEWTON = 0
    #     SPLINE = 1
    #     #start_track_modes
    #     AZ_EL = 1
    #     RA_DEC = 2
    #     RA_DEC_SC = 3  #shortcut
    #     self.scu_put('/devices/command',
    #                 {'path': 'acu.dish_management_controller.start_program_track',
    #                 'params' : {'table_selector': ptrackA,
    #                             'start_time_mjd': start_time,
    #                             'interpol_mode': SPLINE,
    #                             'track_mode': AZ_EL }})


    def with_tt(self, t=None, az=None, el=None, df=None, columns=['time', 'az', 'el'], activate_logging=True, config_name='full_configuration'):
        """run a tracking tabel using a context manager like this: 

        with scu.with_tt(**tt) as track:
            print(f'{track.get_t_remaining()=} {scu.azel=}')

        carries out the steps:
            stop_program_track()
            start_logger() only if needed
            upload_trac_table()
            t0 = time.time()
            wait_duration(2)
            while dt_i < dt:
                dt_i = time.time() - t0
                yield dt_i
            wait_for_pos(az[-1])
            wait_for_pos(el[-1])
            stop_logger() only if needed
            stop_program_track()
            wait_duration(5)
         

        Args:
            t (numpy array Nx1, optional): time vector for tracking table in mjd. Defaults to None.
            az (numpy array Nx1, optional): azimuth values for tracking table. Defaults to None.
            el (numpy array N x1, optional): elevation values for tracking table. Defaults to None.
            df (pd.DataFrame, optional): optional instead of az, el, t to pass a dataframe and use the columns defined in colums. Defaults to None.
            columns (list, optional): columns to use in the dataframe for t, az, el. Defaults to ['time', 'az', 'el'].
            activate_logging (bool, optional): whether or not to activate logging during running of this table. Defaults to True.
            config_name (str, optional): the logging configuration to use. Defaults to 'full_configuration'.
        """


        assert not (isinstance(t, pd.DataFrame) and az is None and el is None), 'Must pass a dataframe with the keyword argument df=...'
        if t is None and az is None and el is None and df is not None and columns is not None:
            t, az, el = [df[c].values for c in columns]
        
        return TrackHandler(self, t, az, el, activate_logging, config_name)

    def run_track_table(self, t=None, az=None, el=None, df=None, columns=['time', 'az', 'el'], activate_logging=True, config_name='full_configuration', wait_start=True, verb=False):
        """run a tracking tabel and block until complete
        stop_program_track()
        start_logger() only if needed
        upload_trac_table()
        wait_duration(dt)
        wait_for_pos(az[-1])
        wait_for_pos(el[-1])
        stop_logger() only if needed
        stop_program_track()
        wait_duration(5)
        return 

        Args:
            t (numpy array Nx1, optional): time vector for tracking table in mjd. Defaults to None.
            az (numpy array Nx1, optional): azimuth values for tracking table. Defaults to None.
            el (numpy array N x1, optional): elevation values for tracking table. Defaults to None.
            df (pd.DataFrame, optional): optional instead of az, el, t to pass a dataframe and use the columns defined in colums. Defaults to None.
            columns (list, optional): columns to use in the dataframe for t, az, el. Defaults to ['time', 'az', 'el'].
            activate_logging (bool, optional): whether or not to activate logging during running of this table. Defaults to True.
            config_name (str, optional): the logging configuration to use. Defaults to 'full_configuration'.
        """
        assert not (isinstance(t, pd.DataFrame) and az is None and el is None), 'Must pass a dataframe with the keyword argument df=...'
        if t is None and az is None and el is None and df is not None and columns is not None:
            t, az, el = [df[c].values for c in columns]
        
        with TrackHandler(self, t, az, el, activate_logging, config_name, wait_start, verb) as track:
            log('Tracking...', color=colors.OKGREEN)



    def upload_track_table(self, t=None, az=None, el=None, df=None, columns=['time', 'az', 'el'], wait_for_start=False):
        """convenience funtion to wrap 
                scu.acu_ska_track(scu.format_body(t, az, el))
            in one call
        Args:
            Either:
                t (iterable of float): time as mjd
                az (iterable of float): azimuth in degree
                el (iterable of float): elevation (alt) in degree
            Or:
                df (pandas.DataFrame): with at least three columns giving time [mjd], az [deg], el[el]
                a (iterable of str): the columns to use in the dataframe. Default: ['time', 'az', 'el']
            wait_for_start (bool, optional): Whether or not to wait for the tracking table to start after upload. Defaults to False.
        """
        assert not (isinstance(t, pd.DataFrame) and az is None and el is None), 'Must pass a dataframe with the keyword argument df=...'
        if t is None and az is None and el is None and df is not None and columns is not None:
            t, az, el = [df[c].values for c in columns]
        

        is_mon_increasing = np.all(np.diff(t) > 0)
        if not is_mon_increasing:
            raise Exception('Time vector in tracking table is not monotonically increasing!')
        
        t_astro = Time(np.round(t, 12), format='mjd')
        t_end = t_astro[-1]
        dt = np.min(np.diff(t_astro.unix))
        assert dt >= 0.2, "ERROR, The ACU will only accept tracking tables with dt > 0.2s, but given was: dt_min={}".format(dt)
        log('Running Table ({}, {:5.3f}, {:5.3f}) --> ({}, {:5.3f}, {:5.3f}) T={:.2f} N={}, dt_min={:.12f}'.format(Time(t[0], format='mjd').datetime, az[0], el[0], Time(t[-1], format='mjd').datetime, az[-1], el[-1],  np.ptp(t_astro.unix), len(az), dt), color=colors.BOLD)
        
        # hack: Workaround. The SCU ignores tracking tables with <= 50 entries
        if len(t) <= 50:
            log(f'WARNING!: The given tracking table has only {len(t)} entries.', color=colors.WARNING)
            log(f'The SCU has a bug where tracking tables with <=50 samples are silently ignored.', color=colors.WARNING)
            log(f'appending until the tracking table has at least 51 entries', color=colors.WARNING)
            
            t, az, el = list(t), list(az), list(el)
            while len(t) <= 50:
                t.append(t[-1]+dt)
                az.append(az[-1])
                el.append(el[-1])
            t, az, el = np.array(t), np.array(az), np.array(el)


        rows_table = ["{:6.12f} {:3.8f} {:3.8f}".format(ti, azi, eli) for ti, azi, eli in zip(t, az, el)]
        self.acu_ska_track("\n".join(rows_table))
        if wait_for_start:
            self.wait_track_start(timeout=30)

        return t_end.unix - self.t_internal.unix


    def stop_program_track(self, no_stdout=False):
        """
        Stop loading program track table - presumably, stops a programmed track
        """

        log("Requesting stop program track")
        self.acu_ska_track_stoploadingtable()
        self.wait_duration(1, no_stdout)
        
        # See E-Mail from Arne, 2021/11/05
        self.scu_put("/devices/command", payload={"path": "acu.tracking_controller.reset_program_track",
                    "params": {"table_selector": "11", "load_mode": "3", "number_of_transmitted_program_track_table_entries": "0"}})

        self.wait_duration(1, no_stdout)


    def acu_ska_track(self, BODY):
        """Low level function. Use 'upload_track_table()' instead to upload a tracking table.
            This function uploads a programTrack (tracking table) in the internal SCU specific format, which is
                "{:6.12f} {:3.8f} {:3.8f}\n" for time[mjd] az[deg] el[deg]
            per row.
        

        Args:
            BODY (str): the string holing the tracking table in the internal SCU specific format.

        Returns:
            request.model.Response: response object from this call (useless for anything but ID and returncode checking)
        """
        log(f'uploading acu-ska-track with {len(BODY)} char size...')
        return self.scu_put('/acuska/programTrack', data = BODY)
        
    def acu_ska_track_stoploadingtable(self):
        """Low level function. Use 'stop_program_track()' instead 

        Returns:
            request.model.Response: response object from this call (useless for anything but ID and returncode checking)
        """
        log('acu ska track stop loading table')
        return self.scu_put('/acuska/stopLoadingTable')
        
    def format_tt_line(self, t, az,  el, capture_flag = 1, parallactic_angle = 0.0):
        """Low Level Function to format a single line within a tracking table in SCU native format.
        assumption is capture flag and parallactic angle will not be used.

        Args:
            t (float): time in MJD
            az (-270 <= float <= 270): azimuth position in deg
            el (15 <= float <= 90): elevation position in deg
            capture_flag (int, optional): ???. Defaults to 1.
            parallactic_angle (float, optional): ???. Defaults to 0.0.
        """
        f_str = '{:.12f} {:.6f} {:.6f} {:.0f} {:.6f}\n'.format(float(t), float(az), float(el), capture_flag, float(parallactic_angle))
        return(f_str)

    def format_body(self, t, az, el):
        """Low Level function to format a list of tracking table entries in SCU native format.

        Args:
            t (list of float): time in MJD
            az (list of float, -270 <= float <= 270): azimuth position in deg
            el (lost of float, 15 <= float <= 90): elevation position in deg
        Returns:
            str: The tracking table in SCU native format.
        """

        body = ''
        for i in range(len(t)):
            body += self.format_tt_line(t[i], az[i], el[i])
        return(body)        

    #status get functions goes here
    
    def status_Value(self, sensor):
        """Low Level function to get the 'value' field from the 
        status message fields a of given device.

        Args:
            sensor (str): path to the sensor to get
        """

        r = self.scu_get('/devices/statusValue', 
            {'path': sensor})
        data = r.json()['value']
        #log('value: ', data)
        return(data)

    def status_finalValue(self, sensor):
        """Low Level function to get the 'finalValue' field from the 
        status message fields a of given device.

        Args:
            sensor (str): path to the sensor to get
        """
        #log('get status finalValue: ', sensor)
        r = self.scu_get('/devices/statusValue', 
            {'path': sensor})
        data = r.json()['finalValue']
        #log('finalValue: ', data)
        return(data)



    def commandMessageFields(self, commandPath):
        """Low Level function to get complete list of all commands message 
        fields from a device given by device name or a single command by given path.

        Use the responses .json() method to access the returned data as a dictionary.

        Args:
            commandPath (str): The path of the command to get

        Returns:
            request.model.Response: response object from this call.
        """
        r = self.scu_get('/devices/commandMessageFields', 
            {'path': commandPath})
        return r

    def statusMessageField(self, statusPath):
        """Low Level function to get a complete list of all 
        status message fields (including the last known value) of given device.

        Use the responses .json() method to access the returned data as a dictionary.

        Args:
            statusPath (str): The path of the status to get

        Returns:
            request.model.Response: response object from this call.
        """
        r = self.scu_get('/devices/statusMessageFields', 
            {'deviceName': statusPath})
        return r
    
    #ppak added 1/10/2020 as debug for onsite SCU version
    #but only info about sensor, value itself is murky?
    def field(self, sensor):
        """Low Level function to get a specific status field.

        Args:
            sensor (str): path to the sensor to get
        """
        #old field method still used on site
        r = self.scu_get('/devices/field', 
            {'path': sensor})
        #data = r.json()['value']
        data = r.json()
        return(data)
    
    #logger functions goes here

    def create_logger(self, config_name, sensor_list):
        '''
        PUT create a config for logging
        Usage:
        create_logger('HN_INDEX_TEST', hn_feed_indexer_sensors)
        or 
        create_logger('HN_TILT_TEST', hn_tilt_sensors)
        '''
        log('create logger')
        r = self.scu_put('/datalogging/config', 
            {'name': config_name,
            'paths': sensor_list})
        return r

    '''unusual does not take json but params'''
    def start_logger(self, config_name='full_configuration', stop_if_need=True):
        """start logging with a given logging config.
        The logging config must have been registered prior. (see self.start() method)

        Args:
            config_name (str, optional): Config to use for logging. Defaults to 'normal'.
            stop_if_need (bool, optional): _description_. Defaults to True.
        """
            # Start data recording
        if stop_if_need and self.logger_state() != 'STOPPED':
            log('WARNING, logger already recording - attempting to stop and start a fresh logger...', color=colors.WARNING)
            self.stop_logger()  
            self.wait_duration(5)

        if self.logger_state() == 'STOPPED':
            log('Starting logger with config: {} ...'.format(config_name))
            r = self.scu_put('/datalogging/start', params='configName=' + config_name)
        else:
            raise Exception(f'Can not start logging, since logging state != "STOPPED" (actual state: "{self.logger_state()}"')

        return r

    def stop_logger(self):
        """stop logging, will raise HTTPError: 412 if logging is not running

        Returns:
            request.model.Response: response object from this call (useless for anything but ID and returncode checking)
        """
        log('stop logger')
        r = self.scu_put('/datalogging/stop')
        return r

    def logger_state(self):
        """get current logger state

        Returns:
            str: "RUNNING" or "STOPPED"
        """
#        log('logger state ')
        r = self.scu_get('/datalogging/currentState')
        #log(r.json()['state'])
        return(r.json()['state'])

    def logger_configs(self):
        """GET all config names
        """
        log('logger configs ')
        r = self.scu_get('/datalogging/configs')
        return(r.json())

    def last_session(self):
        '''
        GET last session
        '''
        log('Last sessions ')
        r = self.scu_get('/datalogging/lastSession')
        session = (r.json()['uuid'])
        return(session)
    
    def logger_sessions(self):
        '''
        GET all sessions
        '''
        warnings.warn(
            "Legacy Function: This is not maintained and might not work as intended",
            DeprecationWarning
        )
        log('logger sessions ')
        r = self.scu_get('/datalogging/sessions')
        return r.json()
    

    def session_query(self, id):
        '''
        GET specific session only - specified by id number
        Usage:
        session_query('16')
        '''
        warnings.warn(
            "Legacy Function: This is not maintained and might not work as intended",
            DeprecationWarning
        )

        log(f'logger session query id "{id}"')
        r = self.scu_get('/datalogging/session',
            {'id': id})
        return r.json()



    def export_session(self, id = 'last', interval_ms=100):
        '''
        LEGACY function: Do not use!

        EXPORT specific session - by id and with interval
        output r.text could be directed to be saved to file 
        Usage: 
        export_session('16',1000)
        or export_session('16',1000).text 
        '''
        log('export session ')
        if interval_ms is None and not hasattr(self, 'telescope'):
            interval_ms = 100

        if id == 'last':
            id = self.last_session()

        r = self.scu_get('/datalogging/exportSession',
            params = {'id': id, 
                'interval_ms' : interval_ms})
        return r

    #sorted_sessions not working yet

    def sorted_sessions(self, isDescending = 'True', startValue = '1', endValue = '25', sortBy = 'Name', filterType='indexSpan'):
        log('sorted sessions')
        r = self.scu_get('/datalogging/sortedSessions',
            {'isDescending': isDescending,
            'startValue': startValue,
            'endValue': endValue,
            'filterType': filterType, #STRING - indexSpan|timeSpan,
            'sortBy': sortBy})
        return r
    
    def get_session_as_text(self, interval_ms=100, session = 'last'):
        """Download and return a session log in original text form

        Args:
            interval_ms (int, optional): sampling interval in milliseconds. Defaults to 1000.
            session (str, optional): session id to save either string with number or 'last'. Defaults to 'last'.

        Returns:
            str: the raw text as downloaded from the SCU
        """

        log('Attempt export of session: "{}" at rate {} ms'.format(session, interval_ms))
        if session == 'last':
            #get all logger sessions, may be many
            # r = self.logger_sessions()
            #[-1] for end of list, and ['uuid'] to get id of last session in list
            session = self.last_session()
        log('Session id: {} '.format(session))
        file_txt = self.export_session(session, interval_ms).text
        return file_txt

    def get_session_as_df(self, interval_ms=100, session = 'last'):
        """Download and return a session log as pandas dataframe

        Args:
            interval_ms (int, optional): sampling interval in milliseconds. Defaults to 1000.
            session (str, optional): session id to save either string with number or 'last'. Defaults to 'last'.
        Raises:
            Exception: on unknown format returned by the SCU for the log file it will raise an generic exception

        Returns:
            pandas.DataFrame: a dataframe with the time column (datetime UTC) as index.
        """
        
        file_txt = self.get_session_as_text(interval_ms=interval_ms, session = session)

        buf = StringIO(file_txt)
        columns = None

        for i in range(100):
            linestart = buf.tell()
            s = buf.readline()
            if s.strip().startswith(';acu.') or s.strip().startswith('Date/Time;acu.'):
                columns = s
                buf.seek(linestart)
                break
        
        if columns is None: 
            raise Exception("The return format of the acu was not recognized. Here is the first 1000 chars:" + file_txt[:1000]) 

        df = pd.read_csv(buf, sep=';', index_col=0)

        if 'Unnamed: 0' in df:
            df = df.set_index('Unnamed: 0')

        df.index = pd.to_datetime(df.index, errors='coerce')

        return df

                        
    # def save_session(self, path_to_save, interval_ms=1000, session = 'last'):
    #     """Download and save a session to the filesystem

    #     Args:
    #         path_to_save (str): path on local filesys to save the session to
    #         interval_ms (int, optional): sampling interval in milliseconds. Defaults to 1000.
    #         session (str, optional): session id to save either string with number or 'last'. Defaults to 'last'.
    #     """
        
        
    #     file_txt = self.get_session_as_text(interval_ms, session)

    #     folder = os.path.dirname(path_to_save)
    #     if os.path.exists(folder) == 0:
    #         log(folder + " does not exist. making new dir", color=colors.WARNING)
    #         os.mkdir(folder)
            
    #     log(f'Saving Session as log file to: "{path_to_save}"', color=colors.BOLD)    
    #     with open(path_to_save, 'a+') as f:
    #         f.write(file_txt)
        
        
    #Simplified one line commands particular to test section being peformed 


    #wait seconds, wait value, wait finalValue
    def wait_until(self, T, no_stout=False):
        """wait until a certain time has been reached

        Args:
            T (astropy.time.Time object): timestamp until which to wait
            no_stout (bool, optional): whether or not to give feedback in stdout. Defaults to False.
        """
        tnow = self.t_internal
        
        if T < tnow:
            log('T={} is in the past! returning without waiting!'.format(T.iso), color=colors.WARNING)

        else:
            # fun = lambda : self.t_internal >= T
            dt_wait = (T - tnow).to(u.s).value                
            return self.wait_duration(dt_wait, no_stout=no_stout)
        
            


    #wait seconds, wait value, wait finalValue
    def wait_duration(self, seconds, no_stout=False):
        """wait for a given amount of seconds

        Args:
            seconds (int): number of seconds to wait for
            no_stout (bool, optional): whether or not to give feedback in stdout. Defaults to False.
        """
        if not no_stout:
            log('wait for {:.1f}s'.format(seconds), color=colors.OKBLUE)
        time.sleep(seconds)
        # if not no_stout:
        #     log('  -> done', color=colors.OKBLUE)

    # def set_time_source(self, select='2'):
    #     self.scu_put("/devices/command", payload={"path": "time_controller.time_source",
    #                 "params": {"time_controller.time_source.choose_time_source": str(select)}})
        
    #Simplified track table functions
        
    def __point_toggle(self, action, select):
        self.scu_put("/devices/command", payload={"path": "acu.pointing_controller.pointing_correction_toggle",
                    "params": {"action": str(action), "select": str(select)}})

        
    def time_test(self):
        try:
            tworld = requests.get('http://worldtimeapi.org/api/timezone/utc').json()['utc_datetime']
        except Exception as err:
            tworld = 'Error while fetching: ' + str(err)
        
        dct = {'local_ntp': Time.now().iso, 
            'worldtimeapi': tworld,
            'D119_ptp_direct' : Time(self['acu.time.external_ptp'], format='mjd').iso,
            'D119A_internal': self.t_internal.iso, 
            }
        return dct
    



    def point_all_ON(self):
        self.__point_toggle(1, '1')

    def point_all_OFF(self):
        self.__point_toggle(0, '1')
    
    def point_tilt_ON(self):
        self.__point_toggle(1, '10')

    def point_tilt_OFF(self):
        self.__point_toggle(0, '10')

    def point_AT_ON(self):
        self.__point_toggle(1, '11')
    
    def point_AT_OFF(self):
        self.__point_toggle(0, '11')
    
    def point_refr_ON(self):
        self.__point_toggle(1, '12')
    
    def point_refr_OFF(self):
        self.__point_toggle(0, '12')
    
    def point_spem_ON(self):
        self.__point_toggle(1, '20')
    
    def point_spem_OFF(self):
        self.__point_toggle(0, '20')
    

    def point_spem_set(self, params:dict, band = 'all', activate = True, set_rest_zero=True, wait_for_set=False, no_stdout=False):
        """set new pointing model parameters to a specific band in ArcSec

        Args:
            params (list): list of parameters (must be of length 9 only contain numbers and in ArcSeconds)
            band (int, optional): the band to set these values to (1...7). Defaults to 1.
            activate (bool, optional): whether or not ton directly activate after setting (there seems to be an error currently). Defaults to False.
        """

        n_timouts = 0
        ready = False
        while not ready:
            try:
                fi_pos = self.get_state('acu.general_management_and_controller.feed_indexer_pos')
                params_old = self.point_spem_get()

                if band == 'all':
                    bands = bands_dc.values()
                else:
                    bands = [band]

                for i, b in enumerate(bands):
                    is_last = i+1 == len(bands)

                    if isinstance(b, str):
                        assert b in bands_dc, f'ERROR: Band "{b}" not in allowed bands: ' + str(bands_dc)
                        bandi = bands_dc[b]
                    else:
                        assert b in bands_dc_inv, f'ERROR: Band {b} not in allowed bands: ' + str(bands_dc_inv)
                        bandi = b

                    assert isinstance(params, dict), 'this version of the SCU lib does only accept dict parameters'
                    assert np.all([isinstance(k, str) for k in params.keys()]), 'dict keys must be strings in the form "P1"..."P9", but given was: ' + ', '.join(params.keys())
                    
                    params = {k.upper():v for k, v in params.items()}
                    param_name_map = self.spem_param_name_map

                    assert np.all([p in param_name_map for p in params.keys()]), 'only "P1 to P22 is allowed, but given was: ' + ', '.join(params.keys())

                    d = {param_name_map[k]:v for k, v in params.items()}
                    if set_rest_zero:
                        d = {**{k:0 for k in self.spem_keys}, **d}
                    else:
                        if band != 'all' and bands_dc_inv[bandi] != str(fi_pos) and len(d) != 18:
                            raise ValueError("can not set a partial pointing model for a feed indexer position which is currently not in focus! Need to position the Feed indexer first and then set the Pointing model or provide a full model!")
                        elif len(d) != 18:
                            params = {**params_old, **params}
                            dold = {param_name_map[k]:v for k, v in params_old.items()}
                            d = {**dold, **d}

                    
                        
                    d['band_information'] = str(bandi)

                    path = 'acu.pointing_controller.set_static_pointing_model_parameters'
                    payload = {'path': path, "params": d}
                    if self.debug:
                        log('setting pointing model with payload: ' + json.dumps(payload))

                    if is_last and not no_stdout:
                        log('setting pointing model to band "{}". Params: {}'.format(band, params), color=colors.OKBLUE)

                    self.scu_put('/devices/command', payload=payload)
                    self.wait_duration(0.5)
                
                if wait_for_set:
                    if band != 'all' and bands_dc_inv[bandi] != str(fi_pos):
                        log('Can not wait for pointing model parameters to be set, because the feed indexer is in the wrong position. Will return without waiting! band_is = {} vs band_set = {}'.format(fi_pos, bandi), color=colors.WARNING)
                    else:
                        
                        for k, v in params.items():
                            delay = wait_for_set if not isinstance(wait_for_set, bool) and wait_for_set > 0 else 0.5
                            self.wait_for_status('acu.pointing.' + k.lower(), v, query_delay=delay, timeout=5, no_stout=no_stdout)
                        if not no_stdout:
                            log(f'Pointing model {params} has been set successfully!', color=colors.OKGREEN)

                if activate:
                    self.point_spem_ON()

                ready = True
            except TimeoutError as terr:
                n_timouts += 1
                if n_timouts > 3:
                    log('ERROR: ' + str(terr), colors.FAIL)
                    raise
            

    def point_spem_get(self):
        """get channels for the pointing model parameters"""
        param_keys = list(self.spem_param_name_map.keys())
        pathes = {k:f'acu.pointing.' + k.lower() for k in param_keys}
        # return dict(zip(pathes.keys(), self.getc(list(pathes.values()))))
        dc = self.getc(list(pathes.values()), as_dict=True)
        return {k:dc[p] for k, p in pathes.items()}
        

    def point_AT_set(self, p_at_azel, band = 'all', activate=False, wait_for_set=True):
        """Set new values to the ambient temperature correction values in arcseconds for a specific band

        Args:
            p_at_azel (ambient_temperature_factor_az, ambient_temperature_factor_el) (float): factors for temp correction in ArcSec/deg-C
            band (int|str, optional): the band to set these values to (1...7). Defaults to 0.
            activate (bool, optional): whether or not ton directly activate after setting (there seems to be an error currently). Defaults to False.
        """
        # TG 2023-09-07 seems to work with MKE
        # log('WARNING! setting an ambient temp correction model currently has errors in SCU and may not work as intended!', color=colors.WARNING)
        
        if band == 'all':
            bands = self.bands_possible.values()
        else:
            bands = [band]

        for bandi in bands:
            if isinstance(bandi, str):
                assert bandi in bands_dc, f'ERROR: Band "{bandi}" not in allowed bands: ' + str(bands_dc)
                bandi = bands_dc[bandi]
            else:
                assert bandi in bands_dc_inv, f'ERROR: Band {bandi} not in allowed bands: ' + str(bands_dc_inv)
                bandi = bandi

            assert int(bandi) in bands_dc_inv, "The given band was not pound in the possible bands for this scu. See scu.bands_possible for further info"

            d = {   'ambient_temperature_factor_az': p_at_azel[0],
                    'ambient_temperature_factor_el': p_at_azel[1],
                    'band_information': str(bandi)}
            
            path = 'acu.pointing_controller.ambient_temperature_correction_setup_values'
            self.scu_put('/devices/command', payload={'path': path, "params": d})
            
            if wait_for_set:
                fun = lambda: self.point_AT_get(band=bandi) == tuple(p_at_azel)
                delay = wait_for_set if not isinstance(wait_for_set, bool) and wait_for_set > 0 else 0.5
                self.wait_by_testfun(fun, query_delay=delay, timeout=10)
        
        if wait_for_set:
            log(f'AT correction model {p_at_azel} for band {band} has been set successfully!', color=colors.OKGREEN)

        if activate:
            self.point_AT_ON()

    def point_AT_get(self, band = 'current'):
        """get the ambient temperature correction coefficients for the band in question

        Args:
            band (str, optional): either 'all' or current, or a specific band name, sich as 'Band 5c'. Defaults to 'current'.

        Returns:
            if band == 'all' a dictionary of all coefficients, else a tuple with the two coefficients for az and el
        """

        channels = [
            'acu.ambient_temp_correction_config.ambient_factor_az_band_1',
            'acu.ambient_temp_correction_config.ambient_factor_az_band_2',
            'acu.ambient_temp_correction_config.ambient_factor_az_band_3',
            'acu.ambient_temp_correction_config.ambient_factor_az_band_4',
            'acu.ambient_temp_correction_config.ambient_factor_az_band_5a',
            'acu.ambient_temp_correction_config.ambient_factor_az_band_5b',
            'acu.ambient_temp_correction_config.ambient_factor_az_band_5c',
            'acu.ambient_temp_correction_config.ambient_factor_az_band_8',
            'acu.ambient_temp_correction_config.ambient_factor_az_band_9',
            'acu.ambient_temp_correction_config.ambient_factor_az_band_10',
            'acu.ambient_temp_correction_config.ambient_factor_el_band_1',
            'acu.ambient_temp_correction_config.ambient_factor_el_band_2',
            'acu.ambient_temp_correction_config.ambient_factor_el_band_3',
            'acu.ambient_temp_correction_config.ambient_factor_el_band_4',
            'acu.ambient_temp_correction_config.ambient_factor_el_band_5a',
            'acu.ambient_temp_correction_config.ambient_factor_el_band_5b',
            'acu.ambient_temp_correction_config.ambient_factor_el_band_5c',
            'acu.ambient_temp_correction_config.ambient_factor_el_band_8',
            'acu.ambient_temp_correction_config.ambient_factor_el_band_9',
            'acu.ambient_temp_correction_config.ambient_factor_el_band_10'
            # 'acu.pointing.amb_temp_corr_val_az',
            # 'acu.pointing.amb_temp_corr_val_el',
            # 'acu.pointing.amb_temp_corr_filter_constant'
        ]

        vals = self.getc(channels, as_dict=True)
        if band == 'all':
            return vals
        elif band == 'current':
            b = self.band_in_focus
            assert b, 'Current Band is None or empty. This should not be for getting AT with the current band!'
            b = b.lower().replace(' ', '_')
            v = tuple([v for p, v in vals.items() if p.endswith(b)])
            return v[0], v[1]
        else:
            b = bands_dc_inv[band] if band in bands_dc_inv else band
            b = b.lower().replace(' ', '_')
            v = tuple([v for p, v in vals.items() if p.endswith(b)])
            return v[0], v[1]
        

        


    def get_from_scu_influx(self, tstart, tend, token=None, org = 'OHBDC', timeout=360_000, channels = None):
        if 'InfluxDBClient' not in locals():
            from influxdb_client import InfluxDBClient
        
        if token is None and hasattr(self, 'influx_token') and self.influx_token:
            token = self.influx_token

        base_query= """
                from(bucket: "SCU")
                |> range(start: {}, stop: {})
                |> filter(fn: (r) => r["_measurement"] == "device_status")
                |> filter(fn: (r) => r["device"] == "acu")
                |> pivot(rowKey: ["_time"], columnKey: ["_field"], valueColumn: "_value")
                |> keep(columns: ["_time", {}])
                """
        mk_query = lambda tstart, tend, chans: base_query.format(tstart, tend, ', '.join(['"_time"'] + [f'"{s}"' for s in chans]))

        if isinstance(tstart, Time):
            tstart = make_zulustr(tstart.datetime, remove_ms=False)
        elif isinstance(tstart, datetime.datetime):
            tstart = make_zulustr(tstart, remove_ms=False)

        if isinstance(tend, Time):
            tend = make_zulustr(tend.datetime, remove_ms=False)
        elif isinstance(tstart, datetime.datetime):
            tend = make_zulustr(tend, remove_ms=False)

        url = f'http://{self.ip}:8086'
        client = InfluxDBClient(url, token, org=org, debug = False, timeout=timeout)

        if not channels:
            channels = self.get_channel_list()

        query = mk_query(tstart, tend, channels)
        
        df = client.query_api().query_data_frame(query)
        df = df.drop(columns=['result', 'table'])
        df = df.rename(columns={'_time':'time'})
        df = df.set_index('time')

        return df



        

    def start(self, az_start=None, el_start=None, band_start=None, az_speed=None, el_speed=None, send_default_configs=False, reset_after_start=True):
        """getting command authority, unstow, activate and start the antenna for usage

        Args:
            az_start (-270 <= az_start <= 270, optional): start position for AZ axis in degree. Defaults to None.
            el_start (15 <= el_start <= 90, optional): start position for EL axis in degree. Defaults to None.
            band_start (str or int, optional): start position ('Band 1'... 'Band 5c' or 1...7) for the Feed Indexer Axis to move to. Defaults to None.
            az_speed (0 < az_speed <= 3.0, optional): azimuth speed to use for movement to inital position. None means as fast as possible. Defaults to None.
            el_speed (0 < el_speed <= 1.0, optional): elevation speed to use for movement to inital position None means as fast as possible. Defaults to None.
            send_default_configs (bool, optional): Whether or not to generate the default logging configs on the SCU on startup. Defaults to True.
            reset_after_start(bool, optional): Whether or not to generate send another "reset" after the whole startup routine. Defaults to True.
        """
        log('=== INITIATING STARTUP ROUTINE ===', color=colors.BOLD)
        self.t_start = Time.now()
        self.get_command_authority()
        self.wait_duration(0.1)
        if self.dish_type == 'mke':
            self.deactivate_lowpowermode()
            self.wait_duration(0.1)
        self.reset_dmc()
        self.wait_duration(3)

        if self.use_socket and not self.data_streamer.is_alive():
            self.data_streamer.start()

        if self.get_state().upper() == 'STOWED':
            self.unstow()

        if send_default_configs:
            configs_scu_dc = self.logger_configs()
            configs_scu = [c['name'] for c in configs_scu_dc]
            for k, v in configs_dc.items():
                if k not in configs_scu:
                    log(f'Creating Default Config: {k} with n={len(v)} channels')
                    self.create_logger(k, v)
        
        self.wait_duration(5)
        self.activate_dmc()
        self.waitForStatusValue("acu.azimuth.axis_bit_status.abs_active", True, timeout=20)
        self.waitForStatusValue("acu.elevation.axis_bit_status.abs_active", True, timeout=20)
        self.wait_duration(5)

        if band_start is not None:
            self.move_to_band(band_start)

        if az_start is not None:
            self.abs_azimuth(az_start, az_speed)
        if el_start is not None:
            self.abs_elevation(el_start, el_speed)

        if az_start is not None or el_start is not None or band_start is not None:
            self.wait_settle()
            self.wait_duration(3)
        log('=== STARTUP ROUTINE COMPLETED ===', color=colors.BOLD)

    def shutdown(self):
        """Stow, deactivate, and release command authority for antenna in order to finish before handing back the antenna
        """
        log('=== INITIATING SHUTDOWN ROUTINE ===', color=colors.BOLD)
        if self.get_state().upper() != 'STOWED':
            self.stow()
        self.wait_duration(10)
        self.release_command_authority()
        self.wait_duration(5)

        log('=== SHUTDOWN ROUTINE COMPLETED ===', color=colors.BOLD)



    def sock_stream(self, channels=None):
        """connect to the websocket of the acu, request some measurement channels and stream them continiously. 
        use with for loop. 

        for t, fields in obj.sock_stream():
           print(t.iso, fields)

        Args:
            channels (list of strings or None): None for all channels available, else list of channel names to request

        Yields:
            tuple: Time, dict[channel_name, val]
        """
        

        _log = logging.getLogger('sock_stream')
        host, port = self.ip, self.port

        url = f'ws://{host}:{port}/wsstatus'

        if not channels:
            channels = self.get_channel_list()
            _log.debug('Creating new connection...')

        with ws_sync_connect(url) as ws:

            out = json.dumps(channels)
            _log.info(f"{self} | requesting n={len(channels)} channels from ACU")
            _log.debug(f"{self} | requesting {out}")
            ws.send(out)
                
            is_first = True
            while True:
            # listener loop

                if is_first:
                    _log.info('Staring data stream...')

                reply = ws.recv()

                if is_first:
                    is_first = False
                    _log.info('Stream data OK. Continuing to stream data...')

                data = json.loads(reply)
                ts = Time(data['timestamp'])

                fields = {k:v[0] for k, v in data['fields'].items()}
                yield ts, fields


    async def sock_stream_async(self, channels=None):
        """connect to the websocket of the acu, request some measurement channels and stream the data. 
        use with an async for loop. 

        async for t, fields in obj.sock_stream_async():
           print(t.iso, fields)

        Args:
            channels (list of strings or None): None for all channels available, else list of channel names to request

        Yields:
            tuple: Time, dict[channel_name, val]
        """
        

        _log = logging.getLogger('sock_stream_async')
        host, port = self.ip, self.port

        url = f'ws://{host}:{port}/wsstatus'

        if not channels:
            channels = self.get_channel_list()

        _log.debug('Creating new connection...')

        async with websockets.connect(url) as ws:

            out = json.dumps(channels)
            _log.info(f"{self} | requesting n={len(channels)} channels from ACU")
            _log.debug(f"{self} | requesting {out}")
            await ws.send(out)
                
            is_first = True
            while True:
            # listener loop
                if is_first:
                    _log.info('Staring data stream...')

                reply = await ws.recv()

                if is_first:
                    is_first = False
                    _log.info('Stream data OK. Continuing to stream data...')

                data = json.loads(reply)

                ts = Time(data['timestamp'])

                fields = {k:v[0] for k, v in data['fields'].items()}
                yield ts, fields




    def sock_listen_forever(self, channels=None, ping_timeout=10, sleep_time=30):
        gen = self.sock_listen_forever_async(channels, ping_timeout=ping_timeout, sleep_time=sleep_time)

        loop = asyncio.get_event_loop()
        while 1:
            yield loop.run_until_complete(gen.__anext__())


    async def sock_listen_forever_async(self, channels=None, ping_timeout=10, sleep_time=30):
        """connect to the websocket of the acu, request some measurement channels and listen forver on it. 
        use with an async for loop. 

        async for t, fields in obj.sock_listen_forever():
           print(t.iso, fields)

        Args:
            channels (list of strings or None): None for all channels available, else list of channel names to request
            ping_timeout (int, optional): timeout for the ping to keep a connection alive. Defaults to 10.
            sleep_time (int, optional): timeout between reconnection attempts. Defaults to 30.

        Yields:
            tuple: Time, dict[channel_name, val]
        """
        

        _log = logging.getLogger('listen_forever')
        host, port = self.ip, self.port

        url = f'ws://{host}:{port}/wsstatus'

        if not channels:
            channels = self.get_channel_list()


        while True:
        # outer loop restarted every time the connection fails
            has_err = False
            _log.debug('Creating new connection...')
            try:
                async with websockets.connect(url) as ws:

                    out = json.dumps(channels)
                    _log.info(f"{self} | requesting n={len(channels)} channels from ACU")
                    _log.debug(f"{self} | requesting {out}")
                    await ws.send(out)
                        
                    is_first = True
                    while True:
                    # listener loop
                        try:
                            if is_first:
                                _log.info('Staring data stream...')

                            reply = await ws.recv()

                            if is_first:
                                is_first = False
                                _log.info('Stream data OK. Continuing to stream data...')

                            if has_err:
                                has_err = False
                                _log.info('Stream data OK, keeping connection alive...')

                        except (asyncio.TimeoutError, websockets.exceptions.ConnectionClosed):
                            try:
                                pong = await ws.ping()
                                await asyncio.wait_for(pong, timeout=ping_timeout)
                                _log.info('Ping OK, keeping connection alive...')
                                continue
                            except:
                                _log.warn(
                                    'Ping error - retrying connection in {} sec (Ctrl-C to quit)'.format(sleep_time))
                                await asyncio.sleep(sleep_time)
                                has_err = True
                                break

                        data = json.loads(reply)

                        ts = Time(data['timestamp'])

                        fields = {k:v[0] for k, v in data['fields'].items()}
                        yield ts, fields

            except (asyncio.TimeoutError, websockets.exceptions.ConnectionClosed):
                _log.warn(
                    'TimeoutError error - retrying connection in {} sec (Ctrl-C to quit)'.format(sleep_time))
                await asyncio.sleep(sleep_time)
                has_err = True
                continue
            except socket.gaierror:
                _log.warn(
                    'Socket error - retrying connection in {} sec (Ctrl-C to quit)'.format(sleep_time))
                await asyncio.sleep(sleep_time)
                has_err = True
                continue
            except ConnectionRefusedError:
                _log.warn('Nobody seems to listen to this endpoint. Please check the URL.')
                _log.warn('Retrying connection in {} sec (Ctrl-C to quit)'.format(sleep_time))
                await asyncio.sleep(sleep_time)
                continue
    
    def show_liveplot(self, channels:list=None, width=600, height=100):
        try:
            get_ipython().__class__.__name__
        except NameError as err:
            raise EnvironmentError('Not running in Ipython!') 
        from IPython.display import display, HTML
        from mke_sculib.js_helpers import make_livepos_page, make_liveplot_page
        if not channels:
            return HTML(make_livepos_page(self.ip, port = self.port, style=f"width:{width}px;height:{height}px;"))
        else:
            return HTML(make_liveplot_page(self.ip, channels, port = self.port, style=f"width:{width}px;height:{height}px;"))


def plot_tt(t, az, el, tint, do_show=True, is_simulation=False):
    if not 'plt' in locals():
        import matplotlib.pyplot as plt

    t_local = Time.now()
    f, (ax1, ax2) = plt.subplots(2,1, figsize=(12,6), sharex=True)
    ti = Time(t, format='mjd').datetime
    # print(t)
    # print(ti)
    
    ax1.plot(ti, az, 'b', label='AZ tracking curve')
    ax2.plot(ti, el, 'g', label='EL tracking curve')

    ax1.axvline(tint.datetime, label=f'Creation Time: {tint.datetime}', color='k')
    ax2.axvline(tint.datetime, label=f'Creation Time: {tint.datetime}', color='k')
    if not is_simulation:
        ax1.axvline(t_local.datetime, label=f'Local Time: {t_local.datetime}', color='r')
        ax2.axvline(t_local.datetime, label=f'Local Time: {t_local.datetime}', color='r')

    ax1.set_ylabel('AZ [deg]')
    ax2.set_ylabel('EL [deg]')
    ax2.set_xlabel('time')
    # ax1.set_title('Tracking Table')
    ax1.legend()
    ax2.legend()
    ax1.grid()
    ax2.grid()

    if do_show:
        plt.show()

    return f, (ax1, ax2)
        
        
# if __name__ == '__main__':
#     log("main")

#     # print(websockets.__version__)
#     # with websockets.sync.client.connect(f'ws://{"10.96.66.10"}:{8080}/wsstatus') as ws:
#     #     print('success!')

#     # from astropy.time import Time
#     # from astropy import units as u

#     d119 = scu('http://10.96.66.10:8080/', debug=False, use_socket=True)

#     d119.set_time_source()

#     d119.wait_duration(10)
    
#     print({k:v for k, v in d119.get_channel_list(with_values=True) if 'time' in k})

#     # stream = DataStreamHandler(d119)
#     # stream.start()
#     # while 1:
#     #     print(f'{stream.t_last_local=} | {stream.t_last_remote=}')
#     # try:



#     el = 80

#     az, el = 0, 15

#     d119.move_to_azel(az, el)
#     d119.wait_settle()
#     print(d119.azel)
#     d119.reset_dmc()

#     #t0 = datetime.datetime.now(tz=datetime.timezone.utc)
#     t0 = d119.t_internal.datetime
#     t0p = t0.replace(second=0, microsecond=0)



#     t0, t0p
#     if t0.second > 30:
#         t00 = Time(t0p) + 2*u.minute
#     else:
#         t00 = Time(t0p) + 1*u.minute

    
#     t = np.arange(0, 60, 0.5)
#     azs = np.linspace(az, az+10, len(t))
#     els = np.ones_like(t) * el
#     tmjd = (t00 + t * u.s).mjd

#     d119.run_track_table(tmjd, azs, els, wait_start=False, verb=True)

    # finally:
    #     d119.shutdown()
    # D119A.reset_dmc()
    # D119A.stow()

    # D119A.move_to_azel(45, 80)
    # D119A.wait_settle()
    # print(D119A.azel)
    # D119A.reset_dmc()
    # D119A.move_to_azel(40, 75)
    # print(D119A.azel)
    # D119A.shutdown()

    # channels = ['acu.azimuth.p_act', 'acu.elevation.p_act']
    
    # while 1:
    #     print(api.get_device_status_value_async(channels))

    # gen = api.sock_listen_forever(channels)
    # while 1:
    #     t, fields = next(gen)
    # # for t, fields in api.sock_listen_forever(channels):
    #     print(time.time(), t, list(fields.values()))

    # import asyncio

    # async def run():
        
    #     gen = api.sock_listen_forever_async(channels)

    #     while 1:
    #         t, fields = await gen.__anext__()
    #         print(time.time(), t, list(fields.values()))

    # loop = asyncio.get_event_loop()
    # loop.run_until_complete(run())
    # loop.close()

