# Mixed utilities
#
# Author: F. Mertens

import os

import itertools

import astropy.time as at
from casacore import tables


def alphanum(s):
    return "".join(filter(str.isalnum, s))


def all_same(l):
    return not l or l.count(l[0]) == len(l)


def all_in_other(l, other):
    return all(elem in other for elem in l)


def get_lst(obs_mjd, longitude=6.57):
    return at.Time(obs_mjd, scale='utc', format='mjd').sidereal_time('mean', longitude=longitude).value


def is_in_lst_bin(lst, lst_s, lst_e):
    if lst_s == lst_e:
        return False
    if lst_e > lst_s:
        return (lst >= lst_s) and (lst < lst_e)
    return (lst >= lst_s) or (lst < lst_e)


def pairwise(iterable):
    '''s -> (s0,s1), (s1,s2), (s2, s3), ...'''
    a, b = itertools.tee(iterable)
    next(b, None)
    return list(zip(a, b))

def is_number(i):
    '''Check if input is a string or an integer'''
    try:
        bool_i = i.isdigit()

    except AttributeError:
        bool_i = isinstance(i, int)

    return bool_i


def get_ms_freqs(ms_file):
    with tables.table(os.path.join(ms_file, 'SPECTRAL_WINDOW'), readonly=True, ack=False) as t_spec_win:
        freqs = t_spec_win.getcol('CHAN_FREQ').reshape(-1)
        chan_widths = t_spec_win.getcol('CHAN_WIDTH').reshape(-1)

    return freqs, chan_widths


def get_info_from_ms_files(files):
    t_start = []    
    t_end = []
    t_total = 0

    for file in files:
        with tables.table(file, ack=False) as t:
            start = t.getcell('TIME', 0)
            end = t.getcell('TIME', t.nrows() - 1)
            t_int = t.getcell('INTERVAL', 0)
        t_start.append(start)
        t_end.append(end)
        t_total += end - start
        freqs, chan_widths = get_ms_freqs(file)
        n_chan = len(freqs)
        chan_width = chan_widths.mean()

    t_start = min(t_start)
    t_end = max(t_end)

    return {'start_time': t_start, 'end_time': t_end, 'int_time': t_int,
            'total_time': t_total, 'n_channels': n_chan, 'chan_width': chan_width}

