
from __future__ import annotations
import re
from dataclasses import dataclass
from pathlib import Path
from typing import Optional
from string import Template

PYPROJECT_TEMPLATE = Template("""[build-system]
requires = ["setuptools>=68", "wheel"]
build-backend = "setuptools.build_meta"

[project]
name = "$name"
version = "$version"
description = "$description"
readme = "README.md"
requires-python = ">=3.8"
authors = [ { name = "$creator_name" } ]
license = { text = "MIT" }
keywords = ["analysis", "packaging", "import-parser"]
classifiers = [
  "Programming Language :: Python :: 3",
  "License :: OSI Approved :: MIT License",
  "Operating System :: OS Independent",
]

[project.urls]
Homepage = "$homepage"

[tool.setuptools]
package-dir = { "" : "." }
packages = ["$package_name"]
""")

def _fetch_pypi_json(project: str) -> Optional[dict]:
    import urllib.request, urllib.error, json
    url = f"https://pypi.org/pypi/{project}/json"
    try:
        with urllib.request.urlopen(url, timeout=3) as resp:  # nosec
            return json.load(resp)
    except urllib.error.HTTPError as e:
        if e.code == 404:
            return None
        return None
    except Exception:
        return None

def _bump_patch(version: str) -> str:
    import re as _re
    m = _re.match(r"^(\d+)\.(\d+)\.(\d+)$", version)
    if not m:
        return version
    major, minor, patch = map(int, m.groups())
    return f"{major}.{minor}.{patch+1}"

@dataclass
class PypiPlan:
    name: str
    version: str
    is_new_project: bool

def plan_pypi_version(name: str, version: Optional[str], new: bool) -> PypiPlan:
    if new:
        exists = _fetch_pypi_json(name) is not None
        if exists:
            raise ValueError(f"Project name '{name}' already exists on PyPI")
        use_ver = version or "0.1.0"
        return PypiPlan(name, use_ver, True)
    else:
        data = _fetch_pypi_json(name)
        if data is None:
            return PypiPlan(name, version or "0.1.0", True)
        latest = data.get("info", {}).get("version", "0.0.0")
        use_ver = version if (version and version > latest) else _bump_patch(latest)
        return PypiPlan(name, use_ver, False)

def create_pypi_package(project_root: str | Path, package_name: str, *,
                         version: Optional[str] = None, new: bool = True,
                         creator_name: str = "Unknown",
                         description: str = "Auto-generated package",
                         homepage: str = "") -> Path:
    project_root = Path(project_root)
    plan = plan_pypi_version(package_name, version, new)
    pyproject = PYPROJECT_TEMPLATE.substitute(
        name=plan.name,
        version=plan.version,
        description=description,
        creator_name=creator_name,
        homepage=homepage or "",
        package_name=package_name,
    )
    (project_root / "pyproject.toml").write_text(pyproject, encoding="utf-8")
    pkg_dir = project_root / package_name
    pkg_dir.mkdir(parents=True, exist_ok=True)
    (pkg_dir / "__init__.py").write_text("__all__ = []\n", encoding="utf-8")
    readme = project_root / "README.md"
    if not readme.exists():
        readme.write_text(f"# {package_name}\n\nGenerated by pyproj_inspector.\n", encoding="utf-8")
    return project_root / "pyproject.toml"

def create_debian_package(project_root: str | Path, package_name: str, *,
                           version: str = "0.1.0", creator_name: str = "Unknown",
                           entry: Optional[str] = None) -> Path:
    import tempfile, shutil, subprocess, os
    project_root = Path(project_root).resolve()
    build = Path(tempfile.mkdtemp(prefix="deb_build_"))
    try:
        base = build / package_name
        (base / "DEBIAN").mkdir(parents=True, exist_ok=True)
        (base / f"usr/local/lib/{package_name}").mkdir(parents=True, exist_ok=True)
        for p in project_root.rglob("*"):
            if p.is_file():
                rel = p.relative_to(project_root)
                dest = base / f"usr/local/lib/{package_name}" / rel
                dest.parent.mkdir(parents=True, exist_ok=True)
                dest.write_bytes(p.read_bytes())
        if entry:
            bin_dir = base / "usr/local/bin"
            bin_dir.mkdir(parents=True, exist_ok=True)
            (bin_dir / package_name).write_text(
                f"#!/usr/bin/env bash\npython3 /usr/local/lib/{package_name}/{entry} \"$@\"\n",
                encoding="utf-8",
            )
            (bin_dir / package_name).chmod(0o755)
        control = f"""Package: {package_name}
Version: {version}
Section: utils
Priority: optional
Architecture: all
Maintainer: {creator_name}
Description: Auto-generated package built by pyproj_inspector
"""
        (base / "DEBIAN/control").write_text(control, encoding="utf-8")
        deb_path = project_root / f"{package_name}_{version}.deb"
        subprocess.check_call(["dpkg-deb", "--build", str(base), str(deb_path)])
        return deb_path
    finally:
        shutil.rmtree(build, ignore_errors=True)
