"""Enhanced tree display components for YouTrack CLI."""

from datetime import datetime
from typing import Any, Dict, List, Optional

from rich.tree import Tree

from .console import get_console


def _get_assignee_name_from_issue(issue: Dict[str, Any]) -> str:
    """Get assignee name from either regular field or custom field."""
    # First try the regular assignee field
    assignee = issue.get("assignee")
    if assignee and isinstance(assignee, dict):
        if assignee.get("fullName"):
            return assignee["fullName"]
        if assignee.get("name"):
            return assignee["name"]
        if assignee.get("login"):
            return assignee["login"]

    # If not found, try the Assignee custom field
    custom_fields = issue.get("customFields", [])
    if isinstance(custom_fields, list):
        for field in custom_fields:
            if isinstance(field, dict) and field.get("name") == "Assignee":
                value = field.get("value")
                if value and isinstance(value, dict):
                    if value.get("fullName"):
                        return value["fullName"]
                    if value.get("name"):
                        return value["name"]
                    if value.get("login"):
                        return value["login"]

    return "Unassigned"


class EnhancedTreeBuilder:
    """Builder class for creating enhanced Rich tree displays."""

    def __init__(self, root_label: str, console=None):
        """Initialize the tree builder.

        Args:
            root_label: Label for the root node
            console: Optional console instance (uses global if not provided)
        """
        self.tree = Tree(root_label)
        self.console = console or get_console()

    def add_node_with_metadata(
        self,
        parent_node: Any,
        label: str,
        metadata: Dict[str, Any],
        style: str = "white",
        dim_metadata: bool = True,
    ) -> Any:
        """Add a node with metadata display.

        Args:
            parent_node: Parent tree node
            label: Primary label for the node
            metadata: Dictionary of metadata to display
            style: Rich style for the main label
            dim_metadata: Whether to dim the metadata display

        Returns:
            The created tree node
        """
        metadata_parts = []
        for key, value in metadata.items():
            if value is not None:
                metadata_parts.append(f"{key}: {value}")

        metadata_text = " | ".join(metadata_parts)

        if metadata_text:
            if dim_metadata:
                node_text = f"[{style}]{label}[/{style}] [dim]({metadata_text})[/dim]"
            else:
                node_text = f"[{style}]{label}[/{style}] ({metadata_text})"
        else:
            node_text = f"[{style}]{label}[/{style}]"

        return parent_node.add(node_text)

    def add_status_node(
        self,
        parent_node: Any,
        label: str,
        status: str,
        status_color: str = "green",
        additional_info: Optional[Dict[str, Any]] = None,
    ) -> Any:
        """Add a node with status indicator.

        Args:
            parent_node: Parent tree node
            label: Primary label for the node
            status: Status text
            status_color: Color for status indicator
            additional_info: Optional additional metadata

        Returns:
            The created tree node
        """
        status_text = f"[{status_color}]●[/{status_color}] {status}"

        if additional_info:
            metadata_parts = []
            for key, value in additional_info.items():
                if value is not None:
                    metadata_parts.append(f"{key}: {value}")

            if metadata_parts:
                metadata_text = " | ".join(metadata_parts)
                node_text = f"{label} {status_text} [dim]({metadata_text})[/dim]"
            else:
                node_text = f"{label} {status_text}"
        else:
            node_text = f"{label} {status_text}"

        return parent_node.add(node_text)

    def display(self) -> None:
        """Display the tree."""
        self.console.print(self.tree)

    def get_tree(self) -> Tree:
        """Get the tree object.

        Returns:
            The Rich Tree object
        """
        return self.tree


def create_issue_dependencies_tree(
    issue: Dict[str, Any],
    dependencies: List[Dict[str, Any]],
    show_status: bool = True,
) -> Tree:
    """Create a tree view of issue dependencies (legacy function - kept for compatibility).

    Args:
        issue: Main issue data
        dependencies: List of related/dependent issues
        show_status: Whether to show status indicators

    Returns:
        Rich Tree object with dependencies
    """
    issue_id = issue.get("idReadable") or issue.get("id", "Unknown")
    issue_summary = issue.get("summary", "No summary")

    builder = EnhancedTreeBuilder(f"🎯 Dependencies for {issue_id}")
    root_node = builder.tree

    # Add main issue node
    main_status = issue.get("state", {}).get("name", "Unknown")
    status_color = _get_status_color(main_status)

    main_metadata = {
        "Priority": issue.get("priority", {}).get("name"),
        "Assignee": _get_assignee_name_from_issue(issue),
    }

    if show_status:
        main_node = builder.add_status_node(
            root_node,
            f"{issue_id}: {issue_summary}",
            main_status,
            status_color,
            main_metadata,
        )
    else:
        main_node = builder.add_node_with_metadata(
            root_node,
            f"{issue_id}: {issue_summary}",
            main_metadata,
        )

    # Group dependencies by type
    depends_on = []
    dependents = []
    related = []

    for dep in dependencies:
        link_type = dep.get("linkType", {}).get("name", "").lower()
        if "depend" in link_type:
            if "outward" in dep.get("direction", "").lower():
                depends_on.append(dep)
            else:
                dependents.append(dep)
        else:
            related.append(dep)

    # Add depends on section
    if depends_on:
        depends_node = main_node.add("[red]⬆ Depends On[/red]")
        for dep in depends_on:
            _add_dependency_node(builder, depends_node, dep, show_status)

    # Add dependents section
    if dependents:
        dependents_node = main_node.add("[blue]⬇ Dependents[/blue]")
        for dep in dependents:
            _add_dependency_node(builder, dependents_node, dep, show_status)

    # Add related issues section
    if related:
        related_node = main_node.add("[yellow]🔗 Related[/yellow]")
        for dep in related:
            _add_dependency_node(builder, related_node, dep, show_status)

    return builder.tree


def create_issue_relationships_tree(
    issue: Dict[str, Any],
    links: List[Dict[str, Any]],
    link_types: List[Dict[str, Any]],
    show_status: bool = True,
) -> Tree:
    """Create a tree view of all issue relationships dynamically.

    Args:
        issue: Main issue data
        links: List of issue links from the API
        link_types: List of available link types from the API
        show_status: Whether to show status indicators

    Returns:
        Rich Tree object with all relationships
    """
    from collections import defaultdict

    issue_id = issue.get("idReadable") or issue.get("id", "Unknown")
    issue_summary = issue.get("summary", "No summary")

    builder = EnhancedTreeBuilder(f"🔗 Relationships for {issue_id}: {issue_summary}")
    root_node = builder.tree

    if not links:
        root_node.add("[dim]No relationships found[/dim]")
        return builder.get_tree()

    # Create a map of link type IDs to their metadata
    link_type_map = {}
    for link_type in link_types:
        link_type_map[link_type.get("id")] = link_type

    # Group relationships by type and direction
    relationships_by_type = defaultdict(lambda: {"outward": [], "inward": []})

    for link in links:
        direction = link.get("direction", "")
        link_type = link.get("linkType", {})
        link_type_id = link_type.get("id")
        link_type_name = link_type.get("name", "Unknown")

        # Get the appropriate display name based on direction
        display_name = link_type_name
        if link_type_id in link_type_map:
            type_info = link_type_map[link_type_id]
            if direction == "OUTWARD" and type_info.get("sourceToTarget"):
                display_name = type_info.get("sourceToTarget")
            elif direction == "INWARD" and type_info.get("targetToSource"):
                display_name = type_info.get("targetToSource")

        # Extract linked issues
        linked_issues = []
        if "issues" in link:
            linked_issues = link["issues"]
        elif "issue" in link:
            linked_issues = [link["issue"]]

        # Add to appropriate direction group
        direction_key = "outward" if direction == "OUTWARD" else "inward"
        for linked_issue in linked_issues:
            issue_data = {
                "id": linked_issue.get("idReadable") or linked_issue.get("id", "Unknown"),
                "summary": linked_issue.get("summary", "No summary"),
                "issue_data": linked_issue,
                "display_name": display_name,
                "type_name": link_type_name,
            }
            relationships_by_type[link_type_name][direction_key].append(issue_data)

    # Build the tree structure grouped by relationship types
    for relationship_type, directions in relationships_by_type.items():
        # Add relationship type node
        type_node = root_node.add(f"[bold cyan]{relationship_type}[/bold cyan]")

        # Add outward relationships
        if directions["outward"]:
            for rel_issue in directions["outward"]:
                display_name = rel_issue["display_name"]
                issue_metadata = {}

                if show_status:
                    # Try to get status from the linked issue
                    linked_issue_data = rel_issue["issue_data"]
                    status = _extract_status_from_issue(linked_issue_data)
                    if status:
                        issue_metadata["Status"] = status

                    assignee = _get_assignee_name_from_issue(linked_issue_data)
                    if assignee and assignee != "Unassigned":
                        issue_metadata["Assignee"] = assignee

                outward_node = type_node.add(f"[green]→ {display_name}[/green]")
                builder.add_node_with_metadata(
                    outward_node,
                    f"{rel_issue['id']}: {rel_issue['summary'][:60]}{'...' if len(rel_issue['summary']) > 60 else ''}",
                    issue_metadata,
                    style="blue",
                )

        # Add inward relationships
        if directions["inward"]:
            for rel_issue in directions["inward"]:
                display_name = rel_issue["display_name"]
                issue_metadata = {}

                if show_status:
                    # Try to get status from the linked issue
                    linked_issue_data = rel_issue["issue_data"]
                    status = _extract_status_from_issue(linked_issue_data)
                    if status:
                        issue_metadata["Status"] = status

                    assignee = _get_assignee_name_from_issue(linked_issue_data)
                    if assignee and assignee != "Unassigned":
                        issue_metadata["Assignee"] = assignee

                inward_node = type_node.add(f"[yellow]← {display_name}[/yellow]")
                builder.add_node_with_metadata(
                    inward_node,
                    f"{rel_issue['id']}: {rel_issue['summary'][:60]}{'...' if len(rel_issue['summary']) > 60 else ''}",
                    issue_metadata,
                    style="blue",
                )

    return builder.get_tree()


def _extract_status_from_issue(issue: Dict[str, Any]) -> Optional[str]:
    """Extract status from issue data, checking multiple possible locations."""
    # Try state field first
    if "state" in issue and isinstance(issue["state"], dict):
        return issue["state"].get("name")

    # Try custom fields
    custom_fields = issue.get("customFields", [])
    if isinstance(custom_fields, list):
        for field in custom_fields:
            if isinstance(field, dict) and field.get("name") in ["State", "Status", "Stage"]:
                value = field.get("value")
                if value and isinstance(value, dict):
                    return value.get("name")
                elif isinstance(value, str):
                    return value

    return None


def create_project_hierarchy_tree(projects: List[Dict[str, Any]]) -> Tree:
    """Create a tree view of project hierarchy.

    Args:
        projects: List of project data

    Returns:
        Rich Tree object with project hierarchy
    """
    builder = EnhancedTreeBuilder("🏗️ Project Hierarchy")

    # Group projects by parent
    root_projects = []
    child_projects: Dict[str, List[Dict[str, Any]]] = {}

    for project in projects:
        parent_id = project.get("parent", {}).get("id")
        if parent_id:
            if parent_id not in child_projects:
                child_projects[parent_id] = []
            child_projects[parent_id].append(project)
        else:
            root_projects.append(project)

    def add_project_to_tree(parent_node: Any, project: Dict[str, Any]) -> None:
        project_id = project.get("shortName", project.get("id", "unknown"))
        project_name = project.get("name", "Unnamed Project")

        metadata = {
            "Lead": project.get("leader", {}).get("fullName"),
            "Issues": project.get("issuesCount", 0),
        }

        # Determine project status
        archived = project.get("archived", False)
        if archived:
            style = "dim red"
            metadata["Status"] = "Archived"
        else:
            style = "green"
            metadata["Status"] = "Active"

        child_node = builder.add_node_with_metadata(
            parent_node,
            f"{project_name} ({project_id})",
            metadata,
            style=style,
        )

        # Add children if any
        if project_id in child_projects:
            for child in child_projects[project_id]:
                add_project_to_tree(child_node, child)

    # Add root projects
    for project in root_projects:
        add_project_to_tree(builder.tree, project)

    return builder.tree


def create_enhanced_articles_tree(
    articles: List[Dict[str, Any]],
    show_metadata: bool = True,
    expand_all: bool = False,
) -> Tree:
    """Create an enhanced tree view of articles with metadata.

    Args:
        articles: List of article data
        show_metadata: Whether to show metadata (author, dates, etc.)
        expand_all: Whether to expand all nodes (placeholder for future feature)

    Returns:
        Rich Tree object with enhanced article display
    """
    builder = EnhancedTreeBuilder("📚 Articles")

    # Group articles by parent
    root_articles = []
    child_articles: Dict[str, List[Dict[str, Any]]] = {}

    for article in articles:
        parent_article = article.get("parentArticle")
        parent_id = parent_article.get("id") if parent_article else None
        if parent_id:
            if parent_id not in child_articles:
                child_articles[parent_id] = []
            child_articles[parent_id].append(article)
        else:
            root_articles.append(article)

    def add_article_to_tree(parent_node: Any, article: Dict[str, Any]) -> None:
        article_id = str(article.get("idReadable", article.get("id", "unknown")))
        title = str(article.get("summary", "Untitled"))

        metadata = {}

        if show_metadata:
            # Add author information
            author = article.get("reporter", {}).get("fullName")
            if author:
                metadata["Author"] = author

            # Add creation date
            created = article.get("created")
            if created:
                try:
                    # Format date nicely
                    date_obj = datetime.fromisoformat(created.replace("Z", "+00:00"))
                    metadata["Created"] = date_obj.strftime("%Y-%m-%d")
                except (ValueError, AttributeError):
                    metadata["Created"] = created

        # Determine visibility status and style
        visibility_info = article.get("visibility", {})
        visibility_type = visibility_info.get("$type", "")

        if visibility_type == "UnlimitedVisibility":
            style = "green"
            metadata["Visibility"] = "Public"
        elif visibility_type == "LimitedVisibility":
            style = "yellow"
            metadata["Visibility"] = "Limited"
        else:
            style = "dim white"
            metadata["Visibility"] = "Unknown"

        child_node = builder.add_node_with_metadata(
            parent_node,
            f"{title} ({article_id})",
            metadata,
            style=style,
        )

        # Add children if any
        if article_id in child_articles:
            for child in child_articles[article_id]:
                add_article_to_tree(child_node, child)

    # Add root articles
    for article in root_articles:
        add_article_to_tree(builder.tree, article)

    return builder.tree


def _add_dependency_node(
    builder: EnhancedTreeBuilder,
    parent_node: Any,
    dependency: Dict[str, Any],
    show_status: bool,
) -> None:
    """Helper function to add a dependency node to the tree."""
    # Handle different link data structures
    target_issue = None

    # Check for "issues" field (array of issues)
    if "issues" in dependency and dependency["issues"]:
        target_issue = dependency["issues"][0]  # Take first issue if multiple
    # Check for "issue" field (single issue)
    elif "issue" in dependency:
        target_issue = dependency["issue"]
    # Fallback to "target" field for backward compatibility
    else:
        target_issue = dependency.get("target", {})

    if not target_issue:
        # If no target issue found, skip this dependency
        return

    issue_id = target_issue.get("idReadable", target_issue.get("id", "Unknown"))
    issue_summary = target_issue.get("summary", "No summary")

    metadata = {
        "Type": target_issue.get("type", {}).get("name"),
        "Priority": target_issue.get("priority", {}).get("name"),
    }

    if show_status:
        status = target_issue.get("state", {}).get("name", "Unknown")
        status_color = _get_status_color(status)
        builder.add_status_node(
            parent_node,
            f"{issue_id}: {issue_summary}",
            status,
            status_color,
            metadata,
        )
    else:
        builder.add_node_with_metadata(
            parent_node,
            f"{issue_id}: {issue_summary}",
            metadata,
        )


def _get_status_color(status: str) -> str:
    """Get color for issue status.

    Args:
        status: Issue status name

    Returns:
        Rich color string
    """
    status_lower = status.lower()

    if any(word in status_lower for word in ["open", "new", "to do", "backlog"]):
        return "blue"
    if any(word in status_lower for word in ["in progress", "working", "active"]):
        return "yellow"
    if any(word in status_lower for word in ["done", "resolved", "closed", "fixed"]):
        return "green"
    if any(word in status_lower for word in ["blocked", "waiting", "on hold"]):
        return "red"
    return "white"
