"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsIsoServiceAutoscaler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const constructs_1 = require("constructs");
/**
 * Creates a EcsIsoServiceAutoscaler construct. This construct allows you to scale an ECS service in an ISO
 * region where classic ECS Autoscaling may not be available.
 *
 * @param scope The parent creating construct (usually `this`).
 * @param id The construct's name.
 * @param this A `EcsIsoServiceAutoscalerthis` interface.
 */
class EcsIsoServiceAutoscaler extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { minimumTaskCount = 1, maximumTaskCount = 10, scaleOutIncrement = 1, scaleInIncrement = 1, scaleOutCooldown = aws_cdk_lib_1.Duration.seconds(60), scaleInCooldown = aws_cdk_lib_1.Duration.seconds(60), } = props;
        this.ecsScalingManagerFunction = new aws_lambda_1.Function(this, `${id}-EcsServiceScalingManager`, {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '../../../resources/constructs/ecsIsoServiceAutoscaler')),
            handler: 'ecs_scaling_manager.handler',
            runtime: aws_lambda_1.Runtime.PYTHON_3_11,
            role: props.role || undefined,
            environment: {
                ECS_CLUSTER_NAME: props.ecsCluster.clusterName,
                ECS_SERVICE_NAME: props.ecsService.serviceName,
                MINIMUM_TASK_COUNT: minimumTaskCount.toString(),
                MAXIMUM_TASK_COUNT: maximumTaskCount.toString(),
                SCALE_ALARM_NAME: props.scaleAlarm.alarmName,
                SCALE_OUT_INCREMENT: scaleOutIncrement.toString(),
                SCALE_OUT_COOLDOWN: scaleOutCooldown.toSeconds().toString(),
                SCALE_IN_INCREMENT: scaleInIncrement.toString(),
                SCALE_IN_COOLDOWN: scaleInCooldown.toSeconds().toString(),
            },
        });
        new aws_events_1.Rule(this, `${id}-EcsScalingManagerSchedule`, {
            description: `Kicks off Lambda to adjust ECS scaling for service: ${props.ecsService.serviceName}`,
            enabled: true,
            schedule: aws_events_1.Schedule.rate(aws_cdk_lib_1.Duration.minutes(1)),
            targets: [new aws_events_targets_1.LambdaFunction(this.ecsScalingManagerFunction)],
        });
        if (!props.role) {
            // Set permissions for ecsScalingManagerFunction role
            this.ecsScalingManagerFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
                actions: ['cloudwatch:DescribeAlarms'],
                effect: aws_iam_1.Effect.ALLOW,
                // Required to be * and can not scope down due to composite alarm role constraints listed here:
                // https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/permissions-reference-cw.html
                // "cloudwatch:DescribeAlarms permission must have a * scope. You can't return information about
                // composite alarms if your cloudwatch:DescribeAlarms permission has a narrower scope."
                resources: ['*'],
            }));
            this.ecsScalingManagerFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
                actions: ['ecs:DescribeServices', 'ecs:UpdateService'],
                effect: aws_iam_1.Effect.ALLOW,
                resources: [props.ecsService.serviceArn],
                conditions: {
                    StringLike: {
                        'ecs:cluster': props.ecsCluster.clusterArn,
                    },
                },
            }));
        }
    }
}
exports.EcsIsoServiceAutoscaler = EcsIsoServiceAutoscaler;
_a = JSII_RTTI_SYMBOL_1;
EcsIsoServiceAutoscaler[_a] = { fqn: "@cdklabs/cdk-enterprise-iac.EcsIsoServiceAutoscaler", version: "0.0.411" };
//# sourceMappingURL=data:application/json;base64,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