"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnStore = void 0;
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const cp = require("child_process");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const flattener_1 = require("./flattener");
const resourceExtractor_1 = require("./resourceExtractor");
class CfnStore {
    constructor(props) {
        this.templates = {};
        this.extractedStackExports = {};
        this.fnJoins = {};
        /** Save CloudFormation templates for future lookup */
        for (const item of props.stackArtifacts) {
            this.templates[item.stackName] = item.template;
        }
        this.flatTemplates = flattener_1.Flattener.flattenObject(this.templates);
        /** Save Value Map if we are using the `API_LOOKUP` sharing method */
        if (props.valueShareMethod == resourceExtractor_1.ResourceExtractorShareMethod.API_LOOKUP) {
            const stack = this.describeStack(props.extractedStackName, props.region);
            if (stack) {
                this.extractedStackExports = this.createExportMap(stack);
            }
        }
    }
    /**
     * Determine what the export value should be, for example if it should be a
     * `Ref` or `Fn::GetAtt`
     *
     * @param resource
     * @param flattendKey
     * @returns
     */
    determineExportValue(resource, flattendKey) {
        const splitKey = flattendKey.split('.');
        if (splitKey.slice(-1)[0] == 'Ref') {
            return resource.ref;
        }
        else if (splitKey.slice(-2)[0] == 'Fn::GetAtt') {
            const attToGet = flattener_1.Flattener.getValueByPath(this.templates, splitKey.slice(0, -1).concat(['1']).join('.'));
            return resource.getAtt(attToGet);
        }
        else if (splitKey.slice(-2)[0] == 'DependsOn') {
            return false;
        }
        else {
            throw new Error(`Can't determine export value for ${flattendKey}`);
        }
    }
    /**
     * Retrieve a Stack Name from a given Logical ID
     *
     * @param logicalId the Logical ID of the Resource to find
     * @returns the Stack Name that the Logical ID is found to be in
     */
    getStackNameFromLogicalId(logicalId) {
        const stackTypeKey = Object.keys(this.flatTemplates).filter((key) => key.indexOf(`${logicalId}.Type`) > -1)[0];
        const stackName = stackTypeKey.split('.')[0];
        return stackName;
    }
    /**
     * Retrieve a Resource Type from a given Logical ID
     *
     * @param logicalId the logical ID of the Resource to find
     * @returns the Resource Type of the provided Logical ID
     */
    getResourceTypeFromLogicalId(logicalId) {
        const typeKey = Object.keys(this.flatTemplates).filter((key) => key.indexOf(`${logicalId}.Type`) > -1)[0];
        const resourceType = this.flatTemplates[typeKey];
        return resourceType;
    }
    /**
     *
     * @param stackName Stack name
     * @param logicalId the logical ID of the Resource to find
     * @returns Json object of the Cloudformation resource properties
     */
    getResourcePropertiesFromLogicalId(stackName, logicalId) {
        return this.templates[stackName].Resources[logicalId].Properties;
    }
    /**
     * Performs a Describe Stack API call with the AWS SDK to determine what
     * the CloudFormation Exports are for a given Stack Name.
     *
     * @param stackName the CloudFormation stack name to query against
     * @param region the AWS region to target
     * @returns CloudFormation Stack object
     */
    describeStack(stackName, region) {
        let stack;
        try {
            const output = cp.spawnSync('node', [
                '-e',
                `
            const sdk = require('aws-sdk');
            const cfn = new sdk.CloudFormation({
              apiVersion: '2016-11-15',
              region: '${region}'
            });
            cfn.describeStacks({StackName: '${stackName}'})
                .promise()
                .then((data) => {
                  const j = JSON.stringify(data);
                  console.log(j);
                }
            );
          `,
            ], { encoding: 'utf8' });
            const response = JSON.parse(output.stdout);
            const stacks = response.Stacks;
            stack = stacks[0];
        }
        catch { }
        return stack;
    }
    /**
     * Builds an Export lookup table from the provided AWS SDK CloudFormation
     * stack object. This will be in the form of `{ 'MyExport': 'foobar' }`.
     *
     * @param stack
     * @returns
     */
    createExportMap(stack) {
        const exports = {};
        const outputs = stack.Outputs ?? [];
        for (const output of outputs) {
            if (output.ExportName && output.OutputValue) {
                exports[output.ExportName] = output.OutputValue;
            }
        }
        return exports;
    }
    /**
     * Rebuilds an `Fn::Join` statement from the Flat Json path. This is
     * necessary because CDK does not correctly inject values back into
     * `Fn::Join` blocks. The rebuilt object is injected back into the spot where
     * the original `Fn::Join` was at. By using this method, we can get around
     * the problem where `addPropertyOverride` does not work for Fn::Join lists.
     *
     * @param fnJoinFlatJsonPath
     * @returns string of Fn::Join (may be string or Json string depending on
     * if CloudFormation needs to reference values or if the entire string can
     * be hardcoded back)
     */
    rebuildFnJoin(fnJoinFlatJsonPath) {
        // Find the items that this fnJoin references
        const items = Object.keys(this.flatTemplates).filter((key) => key.includes(`${fnJoinFlatJsonPath}.1`));
        // Sort the items so that they are rebuilt in the right order
        items.sort();
        // Get the actual values from the flat templates map
        const listOfValues = [];
        items.forEach((item) => {
            if (Object.keys(this.fnJoins).includes(item)) {
                listOfValues.push(this.fnJoins[item]);
            }
            else if (item.split('.').slice(-1)[0] == 'Ref') {
                listOfValues.push(aws_cdk_lib_1.Fn.ref(this.flatTemplates[item]));
            }
            else if (!item.includes('Fn::GetAtt.1')) {
                listOfValues.push(this.flatTemplates[item]);
            }
        });
        // Return the rebuilt Join statement
        return aws_cdk_lib_1.Fn.join(this.flatTemplates[`${fnJoinFlatJsonPath}.0`], listOfValues);
    }
}
exports.CfnStore = CfnStore;
//# sourceMappingURL=data:application/json;base64,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