"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Flattener = void 0;
/**
 * Contains static helper functions to deal with flattening objects,
 * retrieving values from flat objects and setting values on flat objects.
 */
class Flattener {
}
exports.Flattener = Flattener;
/**
 * Flattens a provided Json object, moving all nested objects to the top
 * level and separating with dots, example:
 * ```
 * {
 *   "Stack.Resources.Foo.Properties.Tags.0.Value": "bar"
 * }
 * ```
 *
 * @param obj complex object to flatten
 * @returns key-value pairs of object separated by dots, and the value
 */
Flattener.flattenObject = (obj) => {
    var toReturn = {};
    for (let i in obj) {
        if (!obj.hasOwnProperty(i))
            continue;
        if (typeof obj[i] == 'object' && obj[i] !== null) {
            let flatObject = Flattener.flattenObject(obj[i]);
            for (let x in flatObject) {
                if (!flatObject.hasOwnProperty(x))
                    continue;
                toReturn[i + '.' + x] = flatObject[x];
            }
        }
        else {
            toReturn[i] = obj[i];
        }
    }
    return toReturn;
};
/**
 * Retrieves a value from a provided Json object by using the provided flat
 * path to look it up
 *
 * @param obj object to search through
 * @param path string of where in the object to get, separated by dots
 * @returns found value
 */
Flattener.getValueByPath = (obj, path) => path.split('.').reduce((acc, part) => (acc ? acc[part] : undefined), obj);
/**
 * Updates a value in a provided Json object by using the provided flat
 * path to look it up and the provided value to set
 *
 * @param obj object to update
 * @param path string of path to key to update (separated by dots)
 * @param value value to set in object path
 */
Flattener.setToValue = (obj, path, value) => {
    let i;
    const pathSplit = path.split('.');
    for (i = 0; i < pathSplit.length - 1; i++)
        obj = obj[pathSplit[i]];
    obj[pathSplit[i]] = value;
};
//# sourceMappingURL=data:application/json;base64,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