import { IAspect, Stack } from 'aws-cdk-lib';
import { CloudFormationStackArtifact } from 'aws-cdk-lib/cx-api';
import { IConstruct } from 'constructs';
/**
 * The available value sharing methods to pass values from the extracted stack
 * onto the original stack(s).
 */
export declare enum ResourceExtractorShareMethod {
    'CFN_OUTPUT' = "CFN_OUTPUT",
    'SSM_PARAMETER' = "SSM_PARAMETER",
    'API_LOOKUP' = "API_LOOKUP"
}
export interface ResourceExtractorProps {
    /** Stack to move found extracted resources into. */
    readonly extractDestinationStack: Stack;
    /** Synthed stack artifacts from your CDK app. */
    readonly stackArtifacts: CloudFormationStackArtifact[];
    /** The sharing method to use when passing exported resources from the "Extracted Stack" into the original stack(s). */
    readonly valueShareMethod?: ResourceExtractorShareMethod;
    /** List of resource types to extract, ex: `AWS::IAM::Role`. */
    readonly resourceTypesToExtract: string[];
    /** Additional resource transformations. */
    readonly additionalTransforms?: {
        [key: string]: string;
    };
}
/**
 * This Aspect takes a CDK application, all synthesized CloudFormationStackArtifact,
 * a value share method, and a list of Cloudformation resources that should be
 * pulled out of the main CDK application, which should be synthesized to a
 * cloudformation template that an external team (e.g. security team) to deploy,
 * and adjusting the CDK application to reference pre-created resources already pulled out
 *
 * @example
    const app = App()
    const stack = new Stack(app, 'MyStack');
    extractedStack = new Stack(app, 'ExtractedStack');
    const synthedApp = app.synth();

    Aspects.of(app).add(new ResourceExtractor({
      extractDestinationStack: extractedStack,
      stackArtifacts: synthedApp.stacks,
      valueShareMethod: ResourceExtractorShareMethod.CFN_OUTPUT,
      resourceTypesToExtract: [
        'AWS::IAM::Role',
        'AWS::IAM::Policy',
        'AWS::IAM::ManagedPolicy',
        'AWS::IAM::InstanceProfile',
      ],
    });
    app.synth({ force: true });
 */
export declare class ResourceExtractor implements IAspect {
    private readonly extractDestinationStack;
    private readonly resourceTypesToExtract;
    private readonly valueShareMethod;
    private readonly logicalIdsNotGettingExtracted;
    private readonly cfn;
    private readonly resourceTransformer;
    constructor(props: ResourceExtractorProps);
    /** Entrypoint */
    visit(node: IConstruct): void;
    /**
     * Takes in a CfnResource object and processes it by recreating it in
     * the Security Stack and deleting it from the stack in which it came
     * from.
     *
     * @param node the CfnResource to process
     */
    private processCfnResource;
    /**
     * This will adjust properties in extracted resources that have
     * CloudFormation intrinsic functions referencing resources in the original
     * stack.
     *
     * Most commonly found in IAM policies that scope IAM actions to specific
     * resources in the Stack the policy is being extracted from. In this case
     * it will take a look at the resource type and transform into a partial ARN.
     *
     * @param props modifyExtractedResourcePropertiesProps
     * @returns Json of modified properties, including partial wildcard matchers
     */
    private modifyExtractedResourceProperties;
    /**
     * This takes the flattened key of the resource that's referencing an
     * extracted resource via Cfn intrinsic function, the CfnResource itself
     * (e.g. CfnFunction) as well as the import value (from
     * determineExportValue), and adds a Property override.
     *
     * @param props overrideFoundRefWithImportValueProps
     */
    private overrideFoundRefWithImportValue;
    /**
     * Remove a `DependsOn` reference when not needed
     *
     * @param props
     */
    private deletionOverrideDependsOn;
    /**
     * Finds the CDK resource for a given flattened key by looking it up in the
     * context of the provided node's stack.
     *
     * @param node
     * @param flattendKey
     * @returns CfnResource
     */
    private getResourceFromFoundRef;
    /**
     * Given a flattened key, determine if the resouce is also getting
     * extracted so we know whether or not to adjust CloudFormation Intrinsic
     * functions.
     *
     * @param flattendKey
     * @returns boolean
     */
    private isRefAlsoGettingExtracted;
    /**
     * Get a list of logical ids of resources _not_ getting exported.
     *
     * @returns flat json in the form of `{ LogicalId: 'ResourceType' }`
     */
    private getLogicalIdsNotGettingExtracted;
    /**
     * Exports a value using a consistent standard for the different value
     * share methods.
     *
     * @param stack - the CDK `Stack` object for the resource to import
     * @param name - the name of the export, normally the LogicalId of the
     * resource being exported
     * @param value - the value to export
     */
    private exportValue;
}
