"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResourceTransformer = exports.ResourceTransform = exports.MissingTransformError = void 0;
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const aws_dynamodb_1 = require("aws-cdk-lib/aws-dynamodb");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_elasticsearch_1 = require("aws-cdk-lib/aws-elasticsearch");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_opensearchservice_1 = require("aws-cdk-lib/aws-opensearchservice");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_sns_1 = require("aws-cdk-lib/aws-sns");
const aws_sqs_1 = require("aws-cdk-lib/aws-sqs");
const aws_ssm_1 = require("aws-cdk-lib/aws-ssm");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
class MissingTransformError extends Error {
    constructor(resourceType) {
        super(`
    Unable to find transform for resource type: ${resourceType}.
    You can provide an additional transformation to the Aspect to resolve
    this and/or open a GitHub Issue to get the resource transformation
    added at https://github.com/cdklabs/cdk-enterprise-iac/issues.
    `);
    }
}
exports.MissingTransformError = MissingTransformError;
var ResourceTransform;
(function (ResourceTransform) {
    ResourceTransform["STACK_NAME"] = "ResourceTransform.STACK_NAME";
    ResourceTransform["LOGICAL_ID"] = "ResourceTransform.LOGICAL_ID";
})(ResourceTransform || (exports.ResourceTransform = ResourceTransform = {}));
class ResourceTransformer {
    constructor(props) {
        this.cfn = props.cfnStore;
        this.defaultTransforms = this.createDefaultTransforms();
        this.additionalTransforms = props.additionalTransforms;
    }
    /**
     * Helper function that generates the beginning portion of an AWS Arn.
     */
    generateArnPreamble(service, includeRegion = true, includeAccount = true) {
        const region = includeRegion ? `${aws_cdk_lib_1.Aws.REGION}` : '';
        const account = includeAccount ? `${aws_cdk_lib_1.Aws.ACCOUNT_ID}` : '';
        return `arn:${aws_cdk_lib_1.Aws.PARTITION}:${service}:${region}:${account}`;
    }
    /**
     * Creates the Default Transformation function table that contains transform
     * functions for each CloudFormation Resource Type.
     *
     * This should be updated to support additional resource types.
     * In addition, it may need to be fixed if some of the transformations are
     * found to be incorrect or inconsistent.
     *
     * @returns object in the form of { 'AWS::S3::Bucket': transform function }
     */
    createDefaultTransforms() {
        return {
            /** Standard */
            [aws_dynamodb_1.CfnTable.CFN_RESOURCE_TYPE_NAME]: (stackName, logicalId, _resourceProperties) => {
                const partial = `${stackName}-${logicalId}*`;
                const preamble = this.generateArnPreamble('dynamodb');
                return `${preamble}:table/${partial}`;
            },
            [aws_elasticsearch_1.CfnDomain.CFN_RESOURCE_TYPE_NAME]: (_, logicalId, _resourceProperties) => {
                const partial = `${logicalId.substring(0, 15).toLowerCase()}*`;
                const preamble = this.generateArnPreamble('es');
                return `${preamble}:domain/${partial}`;
            },
            [aws_opensearchservice_1.CfnDomain.CFN_RESOURCE_TYPE_NAME]: (_, logicalId, _resourceProperties) => {
                const partial = `${logicalId.substring(0, 15).toLowerCase()}*`;
                const preamble = this.generateArnPreamble('es');
                return `${preamble}:domain/${partial}`;
            },
            [aws_ecs_1.CfnTaskDefinition.CFN_RESOURCE_TYPE_NAME]: (stackName, logicalId, _resourceProperties) => {
                const partial = `${stackName}${logicalId}*:*`.replace('-', '');
                const preamble = this.generateArnPreamble('ecs');
                return `${preamble}:task-definition/${partial}`;
            },
            [aws_ecs_1.CfnCluster.CFN_RESOURCE_TYPE_NAME]: (stackName, logicalId, _resourceProperties) => {
                const partial = `${stackName}-${logicalId}*`;
                const preamble = this.generateArnPreamble('ecs');
                return `${preamble}:cluster/${partial}`;
            },
            [aws_ecs_1.CfnService.CFN_RESOURCE_TYPE_NAME]: (stackName, logicalId, _resourceProperties) => {
                const partial = `*/${stackName}-${logicalId}*`;
                const preamble = this.generateArnPreamble('ecs');
                return `${preamble}:service/${partial}`;
            },
            /** Colon-resource name grouping */
            [aws_logs_1.CfnLogGroup.CFN_RESOURCE_TYPE_NAME]: (stackName, logicalId, _resourceProperties) => {
                const partial = `${stackName}-${logicalId}*:*`;
                const preamble = this.generateArnPreamble('logs');
                return `${preamble}:log-group:${partial}`;
            },
            [aws_lambda_1.CfnFunction.CFN_RESOURCE_TYPE_NAME]: (stackName, logicalId, _resourceProperties) => {
                const partial = `${stackName}-${logicalId}`.substring(0, 50) + '*';
                const preamble = this.generateArnPreamble('lambda');
                return `${preamble}:function:${partial}`;
            },
            [aws_stepfunctions_1.CfnStateMachine.CFN_RESOURCE_TYPE_NAME]: (_, logicalId, _resourceProperties) => {
                const partial = `${logicalId}*`;
                const preamble = this.generateArnPreamble('states');
                return `${preamble}:stateMachine:${partial}`;
            },
            [aws_cloudwatch_1.CfnAlarm.CFN_RESOURCE_TYPE_NAME]: (stackName, logicalId, _resourceProperties) => {
                const partial = `${stackName}-${logicalId}*`;
                const preamble = this.generateArnPreamble('cloudwatch');
                return `${preamble}:alarm:${partial}`;
            },
            /** No resource name grouping */
            [aws_sqs_1.CfnQueue.CFN_RESOURCE_TYPE_NAME]: (stackName, logicalId, _resourceProperties) => {
                const partial = `${stackName}-${logicalId}*`;
                const preamble = this.generateArnPreamble('sqs');
                return `${preamble}:${partial}`;
            },
            [aws_s3_1.CfnBucket.CFN_RESOURCE_TYPE_NAME]: (stackName, logicalId, resourceProperties) => {
                let partial;
                if (resourceProperties !== undefined &&
                    'BucketName' in resourceProperties) {
                    partial = `${resourceProperties.BucketName}*`;
                }
                else {
                    partial = `${stackName}-${logicalId}*`.toLowerCase();
                }
                const preamble = this.generateArnPreamble('s3', false, false);
                return `${preamble}:${partial}`;
            },
            [aws_sns_1.CfnTopic.CFN_RESOURCE_TYPE_NAME]: (stackName, logicalId, _resourceProperties) => {
                const partial = `${stackName}-${logicalId}*`;
                const preamble = this.generateArnPreamble('sns');
                return `${preamble}:${partial}`;
            },
            /** Non-standard */
            [aws_ssm_1.CfnParameter.CFN_RESOURCE_TYPE_NAME]: (_, logicalId, resourceProperties) => {
                if ('Name' in resourceProperties) {
                    return resourceProperties.Name;
                }
                const partial = `CFN-${logicalId}*`;
                return partial;
            },
            [aws_apigateway_1.CfnApiKey.CFN_RESOURCE_TYPE_NAME]: (stackName, logicalId, _resourceProperties) => {
                const partial = `${stackName.substring(0, 6)}-${logicalId.substring(0, 5)}*`;
                return partial;
            },
        };
    }
    /**
     * Transforms resource names to partial values (primarily ARNs) using
     * wildcards.
     *
     * Takes in a generated resource name from CDK and transforms it to a
     * partial value that is used to replace resource references that use
     * `Fn::GetAtt` from the generated resource name. This is mainly used to
     * avoid cyclical dependencies within CDK and ensure that Policies can be
     * correctly created without knowing the future value of a resource. This
     * relies on the assumption that the developer does NOT input the `name`
     * of the resource they are creating. In other words, you must let CDK
     * generate the resource name.
     *
     * @param logicalId the Logical ID of the Resource generated by CDK
     * @returns string of partial match value to use in IAM Policies
     */
    toPartial(logicalId) {
        const stackName = this.cfn.getStackNameFromLogicalId(logicalId);
        const resourceType = this.cfn.getResourceTypeFromLogicalId(logicalId);
        const resourceProperties = this.cfn.getResourcePropertiesFromLogicalId(stackName, logicalId);
        if (this.additionalTransforms &&
            resourceType in this.additionalTransforms) {
            return this.additionalTransforms[resourceType]
                .replace(ResourceTransform.STACK_NAME, stackName)
                .replace(ResourceTransform.LOGICAL_ID, logicalId);
        }
        else if (resourceType in this.defaultTransforms) {
            const transformResourceToPartial = this.defaultTransforms[resourceType];
            return transformResourceToPartial(stackName, logicalId, resourceProperties);
        }
        else {
            throw new MissingTransformError(resourceType);
        }
    }
}
exports.ResourceTransformer = ResourceTransformer;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVzb3VyY2VUcmFuc2Zvcm1lci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9wYXRjaGVzL3Jlc291cmNlLWV4dHJhY3Rvci9yZXNvdXJjZVRyYW5zZm9ybWVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBOzs7RUFHRTtBQUNGLDZDQUFrQztBQUNsQywrREFBdUQ7QUFDdkQsK0RBQXNEO0FBQ3RELDJEQUFvRDtBQUNwRCxpREFBZ0Y7QUFDaEYscUVBQTBFO0FBQzFFLHVEQUFxRDtBQUNyRCxtREFBbUQ7QUFDbkQsNkVBQThFO0FBQzlFLCtDQUErQztBQUMvQyxpREFBK0M7QUFDL0MsaURBQStDO0FBQy9DLGlEQUFtRDtBQUNuRCxxRUFBZ0U7QUFVaEUsTUFBYSxxQkFBc0IsU0FBUSxLQUFLO0lBQzlDLFlBQVksWUFBb0I7UUFDOUIsS0FBSyxDQUFDO2tEQUN3QyxZQUFZOzs7O0tBSXpELENBQUMsQ0FBQztJQUNMLENBQUM7Q0FDRjtBQVRELHNEQVNDO0FBRUQsSUFBWSxpQkFHWDtBQUhELFdBQVksaUJBQWlCO0lBQzNCLGdFQUEyQyxDQUFBO0lBQzNDLGdFQUEyQyxDQUFBO0FBQzdDLENBQUMsRUFIVyxpQkFBaUIsaUNBQWpCLGlCQUFpQixRQUc1QjtBQU9ELE1BQWEsbUJBQW1CO0lBSzlCLFlBQVksS0FBK0I7UUFDekMsSUFBSSxDQUFDLEdBQUcsR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDO1FBQzFCLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxJQUFJLENBQUMsdUJBQXVCLEVBQUUsQ0FBQztRQUN4RCxJQUFJLENBQUMsb0JBQW9CLEdBQUcsS0FBSyxDQUFDLG9CQUFvQixDQUFDO0lBQ3pELENBQUM7SUFFRDs7T0FFRztJQUNLLG1CQUFtQixDQUN6QixPQUFlLEVBQ2YsZ0JBQXlCLElBQUksRUFDN0IsaUJBQTBCLElBQUk7UUFFOUIsTUFBTSxNQUFNLEdBQUcsYUFBYSxDQUFDLENBQUMsQ0FBQyxHQUFHLGlCQUFHLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUNwRCxNQUFNLE9BQU8sR0FBRyxjQUFjLENBQUMsQ0FBQyxDQUFDLEdBQUcsaUJBQUcsQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO1FBRTFELE9BQU8sT0FBTyxpQkFBRyxDQUFDLFNBQVMsSUFBSSxPQUFPLElBQUksTUFBTSxJQUFJLE9BQU8sRUFBRSxDQUFDO0lBQ2hFLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSyx1QkFBdUI7UUFDN0IsT0FBTztZQUNMLGVBQWU7WUFDZixDQUFDLHVCQUFRLENBQUMsc0JBQXNCLENBQUMsRUFBRSxDQUNqQyxTQUFTLEVBQ1QsU0FBUyxFQUNULG1CQUFtQixFQUNuQixFQUFFO2dCQUNGLE1BQU0sT0FBTyxHQUFHLEdBQUcsU0FBUyxJQUFJLFNBQVMsR0FBRyxDQUFDO2dCQUM3QyxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsVUFBVSxDQUFDLENBQUM7Z0JBQ3RELE9BQU8sR0FBRyxRQUFRLFVBQVUsT0FBTyxFQUFFLENBQUM7WUFDeEMsQ0FBQztZQUNELENBQUMsNkJBQVksQ0FBQyxzQkFBc0IsQ0FBQyxFQUFFLENBQ3JDLENBQUMsRUFDRCxTQUFTLEVBQ1QsbUJBQW1CLEVBQ25CLEVBQUU7Z0JBQ0YsTUFBTSxPQUFPLEdBQUcsR0FBRyxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxXQUFXLEVBQUUsR0FBRyxDQUFDO2dCQUMvRCxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ2hELE9BQU8sR0FBRyxRQUFRLFdBQVcsT0FBTyxFQUFFLENBQUM7WUFDekMsQ0FBQztZQUNELENBQUMsaUNBQVksQ0FBQyxzQkFBc0IsQ0FBQyxFQUFFLENBQ3JDLENBQUMsRUFDRCxTQUFTLEVBQ1QsbUJBQW1CLEVBQ25CLEVBQUU7Z0JBQ0YsTUFBTSxPQUFPLEdBQUcsR0FBRyxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxXQUFXLEVBQUUsR0FBRyxDQUFDO2dCQUMvRCxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ2hELE9BQU8sR0FBRyxRQUFRLFdBQVcsT0FBTyxFQUFFLENBQUM7WUFDekMsQ0FBQztZQUNELENBQUMsMkJBQWlCLENBQUMsc0JBQXNCLENBQUMsRUFBRSxDQUMxQyxTQUFTLEVBQ1QsU0FBUyxFQUNULG1CQUFtQixFQUNuQixFQUFFO2dCQUNGLE1BQU0sT0FBTyxHQUFHLEdBQUcsU0FBUyxHQUFHLFNBQVMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDLENBQUM7Z0JBQy9ELE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDakQsT0FBTyxHQUFHLFFBQVEsb0JBQW9CLE9BQU8sRUFBRSxDQUFDO1lBQ2xELENBQUM7WUFDRCxDQUFDLG9CQUFVLENBQUMsc0JBQXNCLENBQUMsRUFBRSxDQUNuQyxTQUFTLEVBQ1QsU0FBUyxFQUNULG1CQUFtQixFQUNuQixFQUFFO2dCQUNGLE1BQU0sT0FBTyxHQUFHLEdBQUcsU0FBUyxJQUFJLFNBQVMsR0FBRyxDQUFDO2dCQUM3QyxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsS0FBSyxDQUFDLENBQUM7Z0JBQ2pELE9BQU8sR0FBRyxRQUFRLFlBQVksT0FBTyxFQUFFLENBQUM7WUFDMUMsQ0FBQztZQUNELENBQUMsb0JBQVUsQ0FBQyxzQkFBc0IsQ0FBQyxFQUFFLENBQ25DLFNBQVMsRUFDVCxTQUFTLEVBQ1QsbUJBQW1CLEVBQ25CLEVBQUU7Z0JBQ0YsTUFBTSxPQUFPLEdBQUcsS0FBSyxTQUFTLElBQUksU0FBUyxHQUFHLENBQUM7Z0JBQy9DLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDakQsT0FBTyxHQUFHLFFBQVEsWUFBWSxPQUFPLEVBQUUsQ0FBQztZQUMxQyxDQUFDO1lBQ0QsbUNBQW1DO1lBQ25DLENBQUMsc0JBQVcsQ0FBQyxzQkFBc0IsQ0FBQyxFQUFFLENBQ3BDLFNBQVMsRUFDVCxTQUFTLEVBQ1QsbUJBQW1CLEVBQ25CLEVBQUU7Z0JBQ0YsTUFBTSxPQUFPLEdBQUcsR0FBRyxTQUFTLElBQUksU0FBUyxLQUFLLENBQUM7Z0JBQy9DLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxNQUFNLENBQUMsQ0FBQztnQkFDbEQsT0FBTyxHQUFHLFFBQVEsY0FBYyxPQUFPLEVBQUUsQ0FBQztZQUM1QyxDQUFDO1lBQ0QsQ0FBQyx3QkFBVyxDQUFDLHNCQUFzQixDQUFDLEVBQUUsQ0FDcEMsU0FBUyxFQUNULFNBQVMsRUFDVCxtQkFBbUIsRUFDbkIsRUFBRTtnQkFDRixNQUFNLE9BQU8sR0FBRyxHQUFHLFNBQVMsSUFBSSxTQUFTLEVBQUUsQ0FBQyxTQUFTLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxHQUFHLEdBQUcsQ0FBQztnQkFDbkUsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixDQUFDLFFBQVEsQ0FBQyxDQUFDO2dCQUNwRCxPQUFPLEdBQUcsUUFBUSxhQUFhLE9BQU8sRUFBRSxDQUFDO1lBQzNDLENBQUM7WUFDRCxDQUFDLG1DQUFlLENBQUMsc0JBQXNCLENBQUMsRUFBRSxDQUN4QyxDQUFDLEVBQ0QsU0FBUyxFQUNULG1CQUFtQixFQUNuQixFQUFFO2dCQUNGLE1BQU0sT0FBTyxHQUFHLEdBQUcsU0FBUyxHQUFHLENBQUM7Z0JBQ2hDLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxRQUFRLENBQUMsQ0FBQztnQkFDcEQsT0FBTyxHQUFHLFFBQVEsaUJBQWlCLE9BQU8sRUFBRSxDQUFDO1lBQy9DLENBQUM7WUFDRCxDQUFDLHlCQUFRLENBQUMsc0JBQXNCLENBQUMsRUFBRSxDQUNqQyxTQUFTLEVBQ1QsU0FBUyxFQUNULG1CQUFtQixFQUNuQixFQUFFO2dCQUNGLE1BQU0sT0FBTyxHQUFHLEdBQUcsU0FBUyxJQUFJLFNBQVMsR0FBRyxDQUFDO2dCQUM3QyxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsWUFBWSxDQUFDLENBQUM7Z0JBQ3hELE9BQU8sR0FBRyxRQUFRLFVBQVUsT0FBTyxFQUFFLENBQUM7WUFDeEMsQ0FBQztZQUNELGdDQUFnQztZQUNoQyxDQUFDLGtCQUFRLENBQUMsc0JBQXNCLENBQUMsRUFBRSxDQUNqQyxTQUFTLEVBQ1QsU0FBUyxFQUNULG1CQUFtQixFQUNuQixFQUFFO2dCQUNGLE1BQU0sT0FBTyxHQUFHLEdBQUcsU0FBUyxJQUFJLFNBQVMsR0FBRyxDQUFDO2dCQUM3QyxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsS0FBSyxDQUFDLENBQUM7Z0JBQ2pELE9BQU8sR0FBRyxRQUFRLElBQUksT0FBTyxFQUFFLENBQUM7WUFDbEMsQ0FBQztZQUNELENBQUMsa0JBQVMsQ0FBQyxzQkFBc0IsQ0FBQyxFQUFFLENBQ2xDLFNBQVMsRUFDVCxTQUFTLEVBQ1Qsa0JBQWtCLEVBQ2xCLEVBQUU7Z0JBQ0YsSUFBSSxPQUFlLENBQUM7Z0JBQ3BCLElBQ0Usa0JBQWtCLEtBQUssU0FBUztvQkFDaEMsWUFBWSxJQUFJLGtCQUFrQixFQUNsQztvQkFDQSxPQUFPLEdBQUcsR0FBRyxrQkFBa0IsQ0FBQyxVQUFVLEdBQUcsQ0FBQztpQkFDL0M7cUJBQU07b0JBQ0wsT0FBTyxHQUFHLEdBQUcsU0FBUyxJQUFJLFNBQVMsR0FBRyxDQUFDLFdBQVcsRUFBRSxDQUFDO2lCQUN0RDtnQkFDRCxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsSUFBSSxFQUFFLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztnQkFDOUQsT0FBTyxHQUFHLFFBQVEsSUFBSSxPQUFPLEVBQUUsQ0FBQztZQUNsQyxDQUFDO1lBQ0QsQ0FBQyxrQkFBUSxDQUFDLHNCQUFzQixDQUFDLEVBQUUsQ0FDakMsU0FBUyxFQUNULFNBQVMsRUFDVCxtQkFBbUIsRUFDbkIsRUFBRTtnQkFDRixNQUFNLE9BQU8sR0FBRyxHQUFHLFNBQVMsSUFBSSxTQUFTLEdBQUcsQ0FBQztnQkFDN0MsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEtBQUssQ0FBQyxDQUFDO2dCQUNqRCxPQUFPLEdBQUcsUUFBUSxJQUFJLE9BQU8sRUFBRSxDQUFDO1lBQ2xDLENBQUM7WUFDRCxtQkFBbUI7WUFDbkIsQ0FBQyxzQkFBWSxDQUFDLHNCQUFzQixDQUFDLEVBQUUsQ0FDckMsQ0FBQyxFQUNELFNBQVMsRUFDVCxrQkFBa0IsRUFDbEIsRUFBRTtnQkFDRixJQUFJLE1BQU0sSUFBSSxrQkFBa0IsRUFBRTtvQkFDaEMsT0FBTyxrQkFBa0IsQ0FBQyxJQUFJLENBQUM7aUJBQ2hDO2dCQUNELE1BQU0sT0FBTyxHQUFHLE9BQU8sU0FBUyxHQUFHLENBQUM7Z0JBQ3BDLE9BQU8sT0FBTyxDQUFDO1lBQ2pCLENBQUM7WUFDRCxDQUFDLDBCQUFTLENBQUMsc0JBQXNCLENBQUMsRUFBRSxDQUNsQyxTQUFTLEVBQ1QsU0FBUyxFQUNULG1CQUFtQixFQUNuQixFQUFFO2dCQUNGLE1BQU0sT0FBTyxHQUFHLEdBQUcsU0FBUyxDQUFDLFNBQVMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLElBQUksU0FBUyxDQUFDLFNBQVMsQ0FDakUsQ0FBQyxFQUNELENBQUMsQ0FDRixHQUFHLENBQUM7Z0JBQ0wsT0FBTyxPQUFPLENBQUM7WUFDakIsQ0FBQztTQUNGLENBQUM7SUFDSixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0ksU0FBUyxDQUFDLFNBQWlCO1FBQ2hDLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDaEUsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUN0RSxNQUFNLGtCQUFrQixHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsa0NBQWtDLENBQ3BFLFNBQVMsRUFDVCxTQUFTLENBQ1YsQ0FBQztRQUVGLElBQ0UsSUFBSSxDQUFDLG9CQUFvQjtZQUN6QixZQUFZLElBQUksSUFBSSxDQUFDLG9CQUFvQixFQUN6QztZQUNBLE9BQU8sSUFBSSxDQUFDLG9CQUFvQixDQUFDLFlBQVksQ0FBQztpQkFDM0MsT0FBTyxDQUFDLGlCQUFpQixDQUFDLFVBQVUsRUFBRSxTQUFTLENBQUM7aUJBQ2hELE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxVQUFVLEVBQUUsU0FBUyxDQUFDLENBQUM7U0FDckQ7YUFBTSxJQUFJLFlBQVksSUFBSSxJQUFJLENBQUMsaUJBQWlCLEVBQUU7WUFDakQsTUFBTSwwQkFBMEIsR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsWUFBWSxDQUFDLENBQUM7WUFDeEUsT0FBTywwQkFBMEIsQ0FDL0IsU0FBUyxFQUNULFNBQVMsRUFDVCxrQkFBa0IsQ0FDbkIsQ0FBQztTQUNIO2FBQU07WUFDTCxNQUFNLElBQUkscUJBQXFCLENBQUMsWUFBWSxDQUFDLENBQUM7U0FDL0M7SUFDSCxDQUFDO0NBQ0Y7QUF6T0Qsa0RBeU9DIiwic291cmNlc0NvbnRlbnQiOlsiLypcbkNvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiovXG5pbXBvcnQgeyBBd3MgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBDZm5BcGlLZXkgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtYXBpZ2F0ZXdheSc7XG5pbXBvcnQgeyBDZm5BbGFybSB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1jbG91ZHdhdGNoJztcbmltcG9ydCB7IENmblRhYmxlIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWR5bmFtb2RiJztcbmltcG9ydCB7IENmbkNsdXN0ZXIsIENmblNlcnZpY2UsIENmblRhc2tEZWZpbml0aW9uIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWVjcyc7XG5pbXBvcnQgeyBDZm5Eb21haW4gYXMgQ2ZuRG9tYWluRXNzIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWVsYXN0aWNzZWFyY2gnO1xuaW1wb3J0IHsgQ2ZuRnVuY3Rpb24gfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbGFtYmRhJztcbmltcG9ydCB7IENmbkxvZ0dyb3VwIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWxvZ3MnO1xuaW1wb3J0IHsgQ2ZuRG9tYWluIGFzIENmbkRvbWFpbk9zcyB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1vcGVuc2VhcmNoc2VydmljZSc7XG5pbXBvcnQgeyBDZm5CdWNrZXQgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtczMnO1xuaW1wb3J0IHsgQ2ZuVG9waWMgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc25zJztcbmltcG9ydCB7IENmblF1ZXVlIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLXNxcyc7XG5pbXBvcnQgeyBDZm5QYXJhbWV0ZXIgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc3NtJztcbmltcG9ydCB7IENmblN0YXRlTWFjaGluZSB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1zdGVwZnVuY3Rpb25zJztcbmltcG9ydCB7IENmblN0b3JlIH0gZnJvbSAnLi9jZm5TdG9yZSc7XG5pbXBvcnQgeyBGbGF0SnNvbiwgSnNvbiB9IGZyb20gJy4vdHlwZXMnO1xuXG50eXBlIFRyYW5zZm9ybWVyID0gKFxuICBzdGFja05hbWU6IHN0cmluZyxcbiAgbG9naWNhbElkOiBzdHJpbmcsXG4gIHJlc291cmNlUHJvcGVydGllczogSnNvblxuKSA9PiBzdHJpbmc7XG5cbmV4cG9ydCBjbGFzcyBNaXNzaW5nVHJhbnNmb3JtRXJyb3IgZXh0ZW5kcyBFcnJvciB7XG4gIGNvbnN0cnVjdG9yKHJlc291cmNlVHlwZTogc3RyaW5nKSB7XG4gICAgc3VwZXIoYFxuICAgIFVuYWJsZSB0byBmaW5kIHRyYW5zZm9ybSBmb3IgcmVzb3VyY2UgdHlwZTogJHtyZXNvdXJjZVR5cGV9LlxuICAgIFlvdSBjYW4gcHJvdmlkZSBhbiBhZGRpdGlvbmFsIHRyYW5zZm9ybWF0aW9uIHRvIHRoZSBBc3BlY3QgdG8gcmVzb2x2ZVxuICAgIHRoaXMgYW5kL29yIG9wZW4gYSBHaXRIdWIgSXNzdWUgdG8gZ2V0IHRoZSByZXNvdXJjZSB0cmFuc2Zvcm1hdGlvblxuICAgIGFkZGVkIGF0IGh0dHBzOi8vZ2l0aHViLmNvbS9jZGtsYWJzL2Nkay1lbnRlcnByaXNlLWlhYy9pc3N1ZXMuXG4gICAgYCk7XG4gIH1cbn1cblxuZXhwb3J0IGVudW0gUmVzb3VyY2VUcmFuc2Zvcm0ge1xuICBTVEFDS19OQU1FID0gJ1Jlc291cmNlVHJhbnNmb3JtLlNUQUNLX05BTUUnLFxuICBMT0dJQ0FMX0lEID0gJ1Jlc291cmNlVHJhbnNmb3JtLkxPR0lDQUxfSUQnLFxufVxuXG5leHBvcnQgaW50ZXJmYWNlIFJlc291cmNlVHJhbnNmb3JtZXJQcm9wcyB7XG4gIHJlYWRvbmx5IGNmblN0b3JlOiBDZm5TdG9yZTtcbiAgcmVhZG9ubHkgYWRkaXRpb25hbFRyYW5zZm9ybXM/OiBGbGF0SnNvbjtcbn1cblxuZXhwb3J0IGNsYXNzIFJlc291cmNlVHJhbnNmb3JtZXIge1xuICBwcml2YXRlIHJlYWRvbmx5IGNmbjogQ2ZuU3RvcmU7XG4gIHByaXZhdGUgcmVhZG9ubHkgZGVmYXVsdFRyYW5zZm9ybXM6IHsgW2tleTogc3RyaW5nXTogVHJhbnNmb3JtZXIgfTtcbiAgcHJpdmF0ZSByZWFkb25seSBhZGRpdGlvbmFsVHJhbnNmb3Jtcz86IEZsYXRKc29uO1xuXG4gIGNvbnN0cnVjdG9yKHByb3BzOiBSZXNvdXJjZVRyYW5zZm9ybWVyUHJvcHMpIHtcbiAgICB0aGlzLmNmbiA9IHByb3BzLmNmblN0b3JlO1xuICAgIHRoaXMuZGVmYXVsdFRyYW5zZm9ybXMgPSB0aGlzLmNyZWF0ZURlZmF1bHRUcmFuc2Zvcm1zKCk7XG4gICAgdGhpcy5hZGRpdGlvbmFsVHJhbnNmb3JtcyA9IHByb3BzLmFkZGl0aW9uYWxUcmFuc2Zvcm1zO1xuICB9XG5cbiAgLyoqXG4gICAqIEhlbHBlciBmdW5jdGlvbiB0aGF0IGdlbmVyYXRlcyB0aGUgYmVnaW5uaW5nIHBvcnRpb24gb2YgYW4gQVdTIEFybi5cbiAgICovXG4gIHByaXZhdGUgZ2VuZXJhdGVBcm5QcmVhbWJsZShcbiAgICBzZXJ2aWNlOiBzdHJpbmcsXG4gICAgaW5jbHVkZVJlZ2lvbjogYm9vbGVhbiA9IHRydWUsXG4gICAgaW5jbHVkZUFjY291bnQ6IGJvb2xlYW4gPSB0cnVlXG4gICk6IHN0cmluZyB7XG4gICAgY29uc3QgcmVnaW9uID0gaW5jbHVkZVJlZ2lvbiA/IGAke0F3cy5SRUdJT059YCA6ICcnO1xuICAgIGNvbnN0IGFjY291bnQgPSBpbmNsdWRlQWNjb3VudCA/IGAke0F3cy5BQ0NPVU5UX0lEfWAgOiAnJztcblxuICAgIHJldHVybiBgYXJuOiR7QXdzLlBBUlRJVElPTn06JHtzZXJ2aWNlfToke3JlZ2lvbn06JHthY2NvdW50fWA7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyB0aGUgRGVmYXVsdCBUcmFuc2Zvcm1hdGlvbiBmdW5jdGlvbiB0YWJsZSB0aGF0IGNvbnRhaW5zIHRyYW5zZm9ybVxuICAgKiBmdW5jdGlvbnMgZm9yIGVhY2ggQ2xvdWRGb3JtYXRpb24gUmVzb3VyY2UgVHlwZS5cbiAgICpcbiAgICogVGhpcyBzaG91bGQgYmUgdXBkYXRlZCB0byBzdXBwb3J0IGFkZGl0aW9uYWwgcmVzb3VyY2UgdHlwZXMuXG4gICAqIEluIGFkZGl0aW9uLCBpdCBtYXkgbmVlZCB0byBiZSBmaXhlZCBpZiBzb21lIG9mIHRoZSB0cmFuc2Zvcm1hdGlvbnMgYXJlXG4gICAqIGZvdW5kIHRvIGJlIGluY29ycmVjdCBvciBpbmNvbnNpc3RlbnQuXG4gICAqXG4gICAqIEByZXR1cm5zIG9iamVjdCBpbiB0aGUgZm9ybSBvZiB7ICdBV1M6OlMzOjpCdWNrZXQnOiB0cmFuc2Zvcm0gZnVuY3Rpb24gfVxuICAgKi9cbiAgcHJpdmF0ZSBjcmVhdGVEZWZhdWx0VHJhbnNmb3JtcygpOiB7IFtrZXk6IHN0cmluZ106IFRyYW5zZm9ybWVyIH0ge1xuICAgIHJldHVybiB7XG4gICAgICAvKiogU3RhbmRhcmQgKi9cbiAgICAgIFtDZm5UYWJsZS5DRk5fUkVTT1VSQ0VfVFlQRV9OQU1FXTogKFxuICAgICAgICBzdGFja05hbWUsXG4gICAgICAgIGxvZ2ljYWxJZCxcbiAgICAgICAgX3Jlc291cmNlUHJvcGVydGllc1xuICAgICAgKSA9PiB7XG4gICAgICAgIGNvbnN0IHBhcnRpYWwgPSBgJHtzdGFja05hbWV9LSR7bG9naWNhbElkfSpgO1xuICAgICAgICBjb25zdCBwcmVhbWJsZSA9IHRoaXMuZ2VuZXJhdGVBcm5QcmVhbWJsZSgnZHluYW1vZGInKTtcbiAgICAgICAgcmV0dXJuIGAke3ByZWFtYmxlfTp0YWJsZS8ke3BhcnRpYWx9YDtcbiAgICAgIH0sXG4gICAgICBbQ2ZuRG9tYWluRXNzLkNGTl9SRVNPVVJDRV9UWVBFX05BTUVdOiAoXG4gICAgICAgIF8sXG4gICAgICAgIGxvZ2ljYWxJZCxcbiAgICAgICAgX3Jlc291cmNlUHJvcGVydGllc1xuICAgICAgKSA9PiB7XG4gICAgICAgIGNvbnN0IHBhcnRpYWwgPSBgJHtsb2dpY2FsSWQuc3Vic3RyaW5nKDAsIDE1KS50b0xvd2VyQ2FzZSgpfSpgO1xuICAgICAgICBjb25zdCBwcmVhbWJsZSA9IHRoaXMuZ2VuZXJhdGVBcm5QcmVhbWJsZSgnZXMnKTtcbiAgICAgICAgcmV0dXJuIGAke3ByZWFtYmxlfTpkb21haW4vJHtwYXJ0aWFsfWA7XG4gICAgICB9LFxuICAgICAgW0NmbkRvbWFpbk9zcy5DRk5fUkVTT1VSQ0VfVFlQRV9OQU1FXTogKFxuICAgICAgICBfLFxuICAgICAgICBsb2dpY2FsSWQsXG4gICAgICAgIF9yZXNvdXJjZVByb3BlcnRpZXNcbiAgICAgICkgPT4ge1xuICAgICAgICBjb25zdCBwYXJ0aWFsID0gYCR7bG9naWNhbElkLnN1YnN0cmluZygwLCAxNSkudG9Mb3dlckNhc2UoKX0qYDtcbiAgICAgICAgY29uc3QgcHJlYW1ibGUgPSB0aGlzLmdlbmVyYXRlQXJuUHJlYW1ibGUoJ2VzJyk7XG4gICAgICAgIHJldHVybiBgJHtwcmVhbWJsZX06ZG9tYWluLyR7cGFydGlhbH1gO1xuICAgICAgfSxcbiAgICAgIFtDZm5UYXNrRGVmaW5pdGlvbi5DRk5fUkVTT1VSQ0VfVFlQRV9OQU1FXTogKFxuICAgICAgICBzdGFja05hbWUsXG4gICAgICAgIGxvZ2ljYWxJZCxcbiAgICAgICAgX3Jlc291cmNlUHJvcGVydGllc1xuICAgICAgKSA9PiB7XG4gICAgICAgIGNvbnN0IHBhcnRpYWwgPSBgJHtzdGFja05hbWV9JHtsb2dpY2FsSWR9KjoqYC5yZXBsYWNlKCctJywgJycpO1xuICAgICAgICBjb25zdCBwcmVhbWJsZSA9IHRoaXMuZ2VuZXJhdGVBcm5QcmVhbWJsZSgnZWNzJyk7XG4gICAgICAgIHJldHVybiBgJHtwcmVhbWJsZX06dGFzay1kZWZpbml0aW9uLyR7cGFydGlhbH1gO1xuICAgICAgfSxcbiAgICAgIFtDZm5DbHVzdGVyLkNGTl9SRVNPVVJDRV9UWVBFX05BTUVdOiAoXG4gICAgICAgIHN0YWNrTmFtZSxcbiAgICAgICAgbG9naWNhbElkLFxuICAgICAgICBfcmVzb3VyY2VQcm9wZXJ0aWVzXG4gICAgICApID0+IHtcbiAgICAgICAgY29uc3QgcGFydGlhbCA9IGAke3N0YWNrTmFtZX0tJHtsb2dpY2FsSWR9KmA7XG4gICAgICAgIGNvbnN0IHByZWFtYmxlID0gdGhpcy5nZW5lcmF0ZUFyblByZWFtYmxlKCdlY3MnKTtcbiAgICAgICAgcmV0dXJuIGAke3ByZWFtYmxlfTpjbHVzdGVyLyR7cGFydGlhbH1gO1xuICAgICAgfSxcbiAgICAgIFtDZm5TZXJ2aWNlLkNGTl9SRVNPVVJDRV9UWVBFX05BTUVdOiAoXG4gICAgICAgIHN0YWNrTmFtZSxcbiAgICAgICAgbG9naWNhbElkLFxuICAgICAgICBfcmVzb3VyY2VQcm9wZXJ0aWVzXG4gICAgICApID0+IHtcbiAgICAgICAgY29uc3QgcGFydGlhbCA9IGAqLyR7c3RhY2tOYW1lfS0ke2xvZ2ljYWxJZH0qYDtcbiAgICAgICAgY29uc3QgcHJlYW1ibGUgPSB0aGlzLmdlbmVyYXRlQXJuUHJlYW1ibGUoJ2VjcycpO1xuICAgICAgICByZXR1cm4gYCR7cHJlYW1ibGV9OnNlcnZpY2UvJHtwYXJ0aWFsfWA7XG4gICAgICB9LFxuICAgICAgLyoqIENvbG9uLXJlc291cmNlIG5hbWUgZ3JvdXBpbmcgKi9cbiAgICAgIFtDZm5Mb2dHcm91cC5DRk5fUkVTT1VSQ0VfVFlQRV9OQU1FXTogKFxuICAgICAgICBzdGFja05hbWUsXG4gICAgICAgIGxvZ2ljYWxJZCxcbiAgICAgICAgX3Jlc291cmNlUHJvcGVydGllc1xuICAgICAgKSA9PiB7XG4gICAgICAgIGNvbnN0IHBhcnRpYWwgPSBgJHtzdGFja05hbWV9LSR7bG9naWNhbElkfSo6KmA7XG4gICAgICAgIGNvbnN0IHByZWFtYmxlID0gdGhpcy5nZW5lcmF0ZUFyblByZWFtYmxlKCdsb2dzJyk7XG4gICAgICAgIHJldHVybiBgJHtwcmVhbWJsZX06bG9nLWdyb3VwOiR7cGFydGlhbH1gO1xuICAgICAgfSxcbiAgICAgIFtDZm5GdW5jdGlvbi5DRk5fUkVTT1VSQ0VfVFlQRV9OQU1FXTogKFxuICAgICAgICBzdGFja05hbWUsXG4gICAgICAgIGxvZ2ljYWxJZCxcbiAgICAgICAgX3Jlc291cmNlUHJvcGVydGllc1xuICAgICAgKSA9PiB7XG4gICAgICAgIGNvbnN0IHBhcnRpYWwgPSBgJHtzdGFja05hbWV9LSR7bG9naWNhbElkfWAuc3Vic3RyaW5nKDAsIDUwKSArICcqJztcbiAgICAgICAgY29uc3QgcHJlYW1ibGUgPSB0aGlzLmdlbmVyYXRlQXJuUHJlYW1ibGUoJ2xhbWJkYScpO1xuICAgICAgICByZXR1cm4gYCR7cHJlYW1ibGV9OmZ1bmN0aW9uOiR7cGFydGlhbH1gO1xuICAgICAgfSxcbiAgICAgIFtDZm5TdGF0ZU1hY2hpbmUuQ0ZOX1JFU09VUkNFX1RZUEVfTkFNRV06IChcbiAgICAgICAgXyxcbiAgICAgICAgbG9naWNhbElkLFxuICAgICAgICBfcmVzb3VyY2VQcm9wZXJ0aWVzXG4gICAgICApID0+IHtcbiAgICAgICAgY29uc3QgcGFydGlhbCA9IGAke2xvZ2ljYWxJZH0qYDtcbiAgICAgICAgY29uc3QgcHJlYW1ibGUgPSB0aGlzLmdlbmVyYXRlQXJuUHJlYW1ibGUoJ3N0YXRlcycpO1xuICAgICAgICByZXR1cm4gYCR7cHJlYW1ibGV9OnN0YXRlTWFjaGluZToke3BhcnRpYWx9YDtcbiAgICAgIH0sXG4gICAgICBbQ2ZuQWxhcm0uQ0ZOX1JFU09VUkNFX1RZUEVfTkFNRV06IChcbiAgICAgICAgc3RhY2tOYW1lLFxuICAgICAgICBsb2dpY2FsSWQsXG4gICAgICAgIF9yZXNvdXJjZVByb3BlcnRpZXNcbiAgICAgICkgPT4ge1xuICAgICAgICBjb25zdCBwYXJ0aWFsID0gYCR7c3RhY2tOYW1lfS0ke2xvZ2ljYWxJZH0qYDtcbiAgICAgICAgY29uc3QgcHJlYW1ibGUgPSB0aGlzLmdlbmVyYXRlQXJuUHJlYW1ibGUoJ2Nsb3Vkd2F0Y2gnKTtcbiAgICAgICAgcmV0dXJuIGAke3ByZWFtYmxlfTphbGFybToke3BhcnRpYWx9YDtcbiAgICAgIH0sXG4gICAgICAvKiogTm8gcmVzb3VyY2UgbmFtZSBncm91cGluZyAqL1xuICAgICAgW0NmblF1ZXVlLkNGTl9SRVNPVVJDRV9UWVBFX05BTUVdOiAoXG4gICAgICAgIHN0YWNrTmFtZSxcbiAgICAgICAgbG9naWNhbElkLFxuICAgICAgICBfcmVzb3VyY2VQcm9wZXJ0aWVzXG4gICAgICApID0+IHtcbiAgICAgICAgY29uc3QgcGFydGlhbCA9IGAke3N0YWNrTmFtZX0tJHtsb2dpY2FsSWR9KmA7XG4gICAgICAgIGNvbnN0IHByZWFtYmxlID0gdGhpcy5nZW5lcmF0ZUFyblByZWFtYmxlKCdzcXMnKTtcbiAgICAgICAgcmV0dXJuIGAke3ByZWFtYmxlfToke3BhcnRpYWx9YDtcbiAgICAgIH0sXG4gICAgICBbQ2ZuQnVja2V0LkNGTl9SRVNPVVJDRV9UWVBFX05BTUVdOiAoXG4gICAgICAgIHN0YWNrTmFtZSxcbiAgICAgICAgbG9naWNhbElkLFxuICAgICAgICByZXNvdXJjZVByb3BlcnRpZXNcbiAgICAgICkgPT4ge1xuICAgICAgICBsZXQgcGFydGlhbDogc3RyaW5nO1xuICAgICAgICBpZiAoXG4gICAgICAgICAgcmVzb3VyY2VQcm9wZXJ0aWVzICE9PSB1bmRlZmluZWQgJiZcbiAgICAgICAgICAnQnVja2V0TmFtZScgaW4gcmVzb3VyY2VQcm9wZXJ0aWVzXG4gICAgICAgICkge1xuICAgICAgICAgIHBhcnRpYWwgPSBgJHtyZXNvdXJjZVByb3BlcnRpZXMuQnVja2V0TmFtZX0qYDtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICBwYXJ0aWFsID0gYCR7c3RhY2tOYW1lfS0ke2xvZ2ljYWxJZH0qYC50b0xvd2VyQ2FzZSgpO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IHByZWFtYmxlID0gdGhpcy5nZW5lcmF0ZUFyblByZWFtYmxlKCdzMycsIGZhbHNlLCBmYWxzZSk7XG4gICAgICAgIHJldHVybiBgJHtwcmVhbWJsZX06JHtwYXJ0aWFsfWA7XG4gICAgICB9LFxuICAgICAgW0NmblRvcGljLkNGTl9SRVNPVVJDRV9UWVBFX05BTUVdOiAoXG4gICAgICAgIHN0YWNrTmFtZSxcbiAgICAgICAgbG9naWNhbElkLFxuICAgICAgICBfcmVzb3VyY2VQcm9wZXJ0aWVzXG4gICAgICApID0+IHtcbiAgICAgICAgY29uc3QgcGFydGlhbCA9IGAke3N0YWNrTmFtZX0tJHtsb2dpY2FsSWR9KmA7XG4gICAgICAgIGNvbnN0IHByZWFtYmxlID0gdGhpcy5nZW5lcmF0ZUFyblByZWFtYmxlKCdzbnMnKTtcbiAgICAgICAgcmV0dXJuIGAke3ByZWFtYmxlfToke3BhcnRpYWx9YDtcbiAgICAgIH0sXG4gICAgICAvKiogTm9uLXN0YW5kYXJkICovXG4gICAgICBbQ2ZuUGFyYW1ldGVyLkNGTl9SRVNPVVJDRV9UWVBFX05BTUVdOiAoXG4gICAgICAgIF8sXG4gICAgICAgIGxvZ2ljYWxJZCxcbiAgICAgICAgcmVzb3VyY2VQcm9wZXJ0aWVzXG4gICAgICApID0+IHtcbiAgICAgICAgaWYgKCdOYW1lJyBpbiByZXNvdXJjZVByb3BlcnRpZXMpIHtcbiAgICAgICAgICByZXR1cm4gcmVzb3VyY2VQcm9wZXJ0aWVzLk5hbWU7XG4gICAgICAgIH1cbiAgICAgICAgY29uc3QgcGFydGlhbCA9IGBDRk4tJHtsb2dpY2FsSWR9KmA7XG4gICAgICAgIHJldHVybiBwYXJ0aWFsO1xuICAgICAgfSxcbiAgICAgIFtDZm5BcGlLZXkuQ0ZOX1JFU09VUkNFX1RZUEVfTkFNRV06IChcbiAgICAgICAgc3RhY2tOYW1lLFxuICAgICAgICBsb2dpY2FsSWQsXG4gICAgICAgIF9yZXNvdXJjZVByb3BlcnRpZXNcbiAgICAgICkgPT4ge1xuICAgICAgICBjb25zdCBwYXJ0aWFsID0gYCR7c3RhY2tOYW1lLnN1YnN0cmluZygwLCA2KX0tJHtsb2dpY2FsSWQuc3Vic3RyaW5nKFxuICAgICAgICAgIDAsXG4gICAgICAgICAgNVxuICAgICAgICApfSpgO1xuICAgICAgICByZXR1cm4gcGFydGlhbDtcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUcmFuc2Zvcm1zIHJlc291cmNlIG5hbWVzIHRvIHBhcnRpYWwgdmFsdWVzIChwcmltYXJpbHkgQVJOcykgdXNpbmdcbiAgICogd2lsZGNhcmRzLlxuICAgKlxuICAgKiBUYWtlcyBpbiBhIGdlbmVyYXRlZCByZXNvdXJjZSBuYW1lIGZyb20gQ0RLIGFuZCB0cmFuc2Zvcm1zIGl0IHRvIGFcbiAgICogcGFydGlhbCB2YWx1ZSB0aGF0IGlzIHVzZWQgdG8gcmVwbGFjZSByZXNvdXJjZSByZWZlcmVuY2VzIHRoYXQgdXNlXG4gICAqIGBGbjo6R2V0QXR0YCBmcm9tIHRoZSBnZW5lcmF0ZWQgcmVzb3VyY2UgbmFtZS4gVGhpcyBpcyBtYWlubHkgdXNlZCB0b1xuICAgKiBhdm9pZCBjeWNsaWNhbCBkZXBlbmRlbmNpZXMgd2l0aGluIENESyBhbmQgZW5zdXJlIHRoYXQgUG9saWNpZXMgY2FuIGJlXG4gICAqIGNvcnJlY3RseSBjcmVhdGVkIHdpdGhvdXQga25vd2luZyB0aGUgZnV0dXJlIHZhbHVlIG9mIGEgcmVzb3VyY2UuIFRoaXNcbiAgICogcmVsaWVzIG9uIHRoZSBhc3N1bXB0aW9uIHRoYXQgdGhlIGRldmVsb3BlciBkb2VzIE5PVCBpbnB1dCB0aGUgYG5hbWVgXG4gICAqIG9mIHRoZSByZXNvdXJjZSB0aGV5IGFyZSBjcmVhdGluZy4gSW4gb3RoZXIgd29yZHMsIHlvdSBtdXN0IGxldCBDREtcbiAgICogZ2VuZXJhdGUgdGhlIHJlc291cmNlIG5hbWUuXG4gICAqXG4gICAqIEBwYXJhbSBsb2dpY2FsSWQgdGhlIExvZ2ljYWwgSUQgb2YgdGhlIFJlc291cmNlIGdlbmVyYXRlZCBieSBDREtcbiAgICogQHJldHVybnMgc3RyaW5nIG9mIHBhcnRpYWwgbWF0Y2ggdmFsdWUgdG8gdXNlIGluIElBTSBQb2xpY2llc1xuICAgKi9cbiAgcHVibGljIHRvUGFydGlhbChsb2dpY2FsSWQ6IHN0cmluZykge1xuICAgIGNvbnN0IHN0YWNrTmFtZSA9IHRoaXMuY2ZuLmdldFN0YWNrTmFtZUZyb21Mb2dpY2FsSWQobG9naWNhbElkKTtcbiAgICBjb25zdCByZXNvdXJjZVR5cGUgPSB0aGlzLmNmbi5nZXRSZXNvdXJjZVR5cGVGcm9tTG9naWNhbElkKGxvZ2ljYWxJZCk7XG4gICAgY29uc3QgcmVzb3VyY2VQcm9wZXJ0aWVzID0gdGhpcy5jZm4uZ2V0UmVzb3VyY2VQcm9wZXJ0aWVzRnJvbUxvZ2ljYWxJZChcbiAgICAgIHN0YWNrTmFtZSxcbiAgICAgIGxvZ2ljYWxJZFxuICAgICk7XG5cbiAgICBpZiAoXG4gICAgICB0aGlzLmFkZGl0aW9uYWxUcmFuc2Zvcm1zICYmXG4gICAgICByZXNvdXJjZVR5cGUgaW4gdGhpcy5hZGRpdGlvbmFsVHJhbnNmb3Jtc1xuICAgICkge1xuICAgICAgcmV0dXJuIHRoaXMuYWRkaXRpb25hbFRyYW5zZm9ybXNbcmVzb3VyY2VUeXBlXVxuICAgICAgICAucmVwbGFjZShSZXNvdXJjZVRyYW5zZm9ybS5TVEFDS19OQU1FLCBzdGFja05hbWUpXG4gICAgICAgIC5yZXBsYWNlKFJlc291cmNlVHJhbnNmb3JtLkxPR0lDQUxfSUQsIGxvZ2ljYWxJZCk7XG4gICAgfSBlbHNlIGlmIChyZXNvdXJjZVR5cGUgaW4gdGhpcy5kZWZhdWx0VHJhbnNmb3Jtcykge1xuICAgICAgY29uc3QgdHJhbnNmb3JtUmVzb3VyY2VUb1BhcnRpYWwgPSB0aGlzLmRlZmF1bHRUcmFuc2Zvcm1zW3Jlc291cmNlVHlwZV07XG4gICAgICByZXR1cm4gdHJhbnNmb3JtUmVzb3VyY2VUb1BhcnRpYWwoXG4gICAgICAgIHN0YWNrTmFtZSxcbiAgICAgICAgbG9naWNhbElkLFxuICAgICAgICByZXNvdXJjZVByb3BlcnRpZXNcbiAgICAgICk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRocm93IG5ldyBNaXNzaW5nVHJhbnNmb3JtRXJyb3IocmVzb3VyY2VUeXBlKTtcbiAgICB9XG4gIH1cbn1cbiJdfQ==