/*
   base64.cpp and base64.h

   Copyright (C) 2004-2008 René Nyffenegger

   This source code is provided 'as-is', without any express or implied
   warranty. In no event will the author be held liable for any damages
   arising from the use of this software.

   Permission is granted to anyone to use this software for any purpose,
   including commercial applications, and to alter it and redistribute it
   freely, subject to the following restrictions:

   1. The origin of this source code must not be misrepresented; you must not
   claim that you wrote the original source code. If you use this source code
   in a product, an acknowledgment in the product documentation would be
   appreciated but is not required.

   2. Altered source versions must be plainly marked as such, and must not be
   misrepresented as being the original source code.

   3. This notice may not be removed or altered from any source distribution.

   René Nyffenegger rene.nyffenegger@adp-gmbh.ch

*/

/* This source version has been altered by Klebert-Engineering. */

#include "base64.hpp"
#include <cstdlib>
#include <cctype>

/* Standart base64 */
static const std::string base64_chars = 
    "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
    "abcdefghijklmnopqrstuvwxyz"
    "0123456789+/";

static inline bool is_base64(unsigned char c)
{
    return (std::isalnum(c) || c == '+' || c == '/');
}

/* URL safe base64 */
static const std::string base64url_chars =
    "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
    "abcdefghijklmnopqrstuvwxyz"
    "0123456789-_";

static inline bool is_base64url(unsigned char c)
{
    return (std::isalnum(c) || c == '-' || c == '_');
}

static std::string encode(const std::string& alphabet,
                          unsigned char const* bytes_to_encode,
                          unsigned int in_len)
{
    std::string ret;
    ret.reserve((in_len * 4 + 3) / 3 + 1); /* Size estimation optimization */

    int i = 0;
    int j = 0;
    unsigned char char_array_3[3];
    unsigned char char_array_4[4];

    while (in_len--) {
        char_array_3[i++] = *(bytes_to_encode++);
        if (i == 3) {
            char_array_4[0] = (char_array_3[0] & 0xfc) >> 2;
            char_array_4[1] = ((char_array_3[0] & 0x03) << 4) + ((char_array_3[1] & 0xf0) >> 4);
            char_array_4[2] = ((char_array_3[1] & 0x0f) << 2) + ((char_array_3[2] & 0xc0) >> 6);
            char_array_4[3] = char_array_3[2] & 0x3f;

            for(i = 0; (i <4) ; i++)
                ret += alphabet[char_array_4[i]];
            i = 0;
        }
    }

    if (i)
    {
        for(j = i; j < 3; j++)
            char_array_3[j] = '\0';

        char_array_4[0] = (char_array_3[0] & 0xfc) >> 2;
        char_array_4[1] = ((char_array_3[0] & 0x03) << 4) + ((char_array_3[1] & 0xf0) >> 4);
        char_array_4[2] = ((char_array_3[1] & 0x0f) << 2) + ((char_array_3[2] & 0xc0) >> 6);
        char_array_4[3] = char_array_3[2] & 0x3f;

        for (j = 0; (j < i + 1); j++)
            ret += alphabet[char_array_4[j]];

        while((i++ < 3))
            ret += '=';

    }

    return ret;

}

static std::string decode(const std::string& alphabet,
                          bool (*is_base64_pred)(unsigned char),
                          std::string const& encoded_string)
{
    auto in_len = encoded_string.size();
    int i = 0;
    int j = 0;
    int in_ = 0;
    unsigned char char_array_4[4], char_array_3[3];
    std::string ret;

    while (in_len-- && ( encoded_string[in_] != '=') && is_base64_pred(encoded_string[in_])) {
        char_array_4[i++] = encoded_string[in_]; in_++;
        if (i ==4) {
            for (i = 0; i <4; i++)
                char_array_4[i] = alphabet.find(char_array_4[i]);

            char_array_3[0] = (char_array_4[0] << 2) + ((char_array_4[1] & 0x30) >> 4);
            char_array_3[1] = ((char_array_4[1] & 0xf) << 4) + ((char_array_4[2] & 0x3c) >> 2);
            char_array_3[2] = ((char_array_4[2] & 0x3) << 6) + char_array_4[3];

            for (i = 0; (i < 3); i++)
                ret += char_array_3[i];
            i = 0;
        }
    }

    if (i) {
        for (j = i; j <4; j++)
            char_array_4[j] = 0;

        for (j = 0; j <4; j++)
            char_array_4[j] = alphabet.find(char_array_4[j]);

        char_array_3[0] = (char_array_4[0] << 2) + ((char_array_4[1] & 0x30) >> 4);
        char_array_3[1] = ((char_array_4[1] & 0xf) << 4) + ((char_array_4[2] & 0x3c) >> 2);
        char_array_3[2] = ((char_array_4[2] & 0x3) << 6) + char_array_4[3];

        for (j = 0; (j < i - 1); j++) ret += char_array_3[j];
    }

    return ret;
}

namespace zswagcl
{

std::string base64_encode(unsigned char const* bytes_to_encode,
                          unsigned int in_len)
{
    return encode(base64_chars, bytes_to_encode, in_len);
}

std::string base64url_encode(unsigned char const* bytes_to_encode,
                          unsigned int in_len)
{
    auto str = encode(base64url_chars, bytes_to_encode, in_len);
    // Note: The spec would allow for truncating the padding. While
    // saving some bandwidth, this also requires cooperation from the server
    // to repair the padding before decoding, which is not a standard
    // operation. So we can add this as an option in the future, but for
    // now let's keep the padding.
    // while (str.back() == '=')
    //     str.erase(str.size() - 1);
    return str;
}

std::string base64_decode(std::string const& encoded_string)
{
    return decode(base64_chars, &is_base64, encoded_string);
}

std::string base64url_decode(std::string const& encoded_string)
{
    return decode(base64url_chars, &is_base64url, encoded_string);
}

}
