package utils

import "strings"

// ExtractRoleNameFromPrincipal extracts role name from principal ARN
// Handles both STS assumed role and IAM role formats
func ExtractRoleNameFromPrincipal(principal string) string {
	// Handle STS assumed role format: arn:seaweed:sts::assumed-role/RoleName/SessionName
	stsPrefix := "arn:seaweed:sts::assumed-role/"
	if strings.HasPrefix(principal, stsPrefix) {
		remainder := principal[len(stsPrefix):]
		// Split on first '/' to get role name
		if slashIndex := strings.Index(remainder, "/"); slashIndex != -1 {
			return remainder[:slashIndex]
		}
		// If no slash found, return the remainder (edge case)
		return remainder
	}

	// Handle IAM role format: arn:seaweed:iam::role/RoleName
	iamPrefix := "arn:seaweed:iam::role/"
	if strings.HasPrefix(principal, iamPrefix) {
		return principal[len(iamPrefix):]
	}

	// Return empty string to signal invalid ARN format
	// This allows callers to handle the error explicitly instead of masking it
	return ""
}

// ExtractRoleNameFromArn extracts role name from an IAM role ARN
// Specifically handles: arn:seaweed:iam::role/RoleName
func ExtractRoleNameFromArn(roleArn string) string {
	prefix := "arn:seaweed:iam::role/"
	if strings.HasPrefix(roleArn, prefix) && len(roleArn) > len(prefix) {
		return roleArn[len(prefix):]
	}
	return ""
}
