import { CfnResource } from 'aws-cdk-lib';
import { NagMessageLevel, NagRuleStates } from './nag-rules';
/**
 * Shared data for all INagLogger methods
 * @param nagPackName The name of the NagPack that the rule belongs to.
 * @param resource The resource the suppression is applied to.
 * @param ruleId Why the rule was triggered.
 * @param ruleInfo The id of the rule to ignore.
 * @param ruleExplanation Why the rule exists.
 * @param ruleLevel The severity level of the rule.
 */
export interface NagLoggerBaseData {
    readonly nagPackName: string;
    readonly resource: CfnResource;
    readonly ruleId: string;
    readonly ruleInfo: string;
    readonly ruleExplanation: string;
    readonly ruleLevel: NagMessageLevel;
}
/**
 * Data for onCompliance method of an INagLogger.
 */
export interface NagLoggerComplianceData extends NagLoggerBaseData {
}
/**
 * Data for onNonCompliance method of an INagLogger.
 * @param findingId The id of the finding that is being checked.
 */
export interface NagLoggerNonComplianceData extends NagLoggerBaseData {
    readonly findingId: string;
}
/**
 * Data for onSuppressed method of an INagLogger.
 * @param suppressionReason The reason given for the suppression.
 */
export interface NagLoggerSuppressedData extends NagLoggerNonComplianceData {
    readonly suppressionReason: string;
}
/**
 * Data for onError method of an INagLogger.
 * @param errorMessage: The error that was thrown.
 * @param shouldLogIgnored Whether or not the NagPack user has indicated that they want to log suppression details.
 */
export interface NagLoggerErrorData extends NagLoggerBaseData {
    readonly errorMessage: string;
}
/**
 * Data for onSuppressedError method of an INagLogger.
 * @param errorSuppressionReason The reason given for the validation error suppression.
 */
export interface NagLoggerSuppressedErrorData extends NagLoggerErrorData {
    readonly errorSuppressionReason: string;
}
/**
 * Data for onNotApplicable method of an INagLogger.
 */
export interface NagLoggerNotApplicableData extends NagLoggerBaseData {
}
/**
 * Interface for creating NagSuppression Ignores
 */
export interface INagLogger {
    /**
     * Called when a CfnResource passes the compliance check for a given rule.
     */
    onCompliance(data: NagLoggerComplianceData): void;
    /**
     * Called when a CfnResource does not pass the compliance check for a given rule and the the rule violation is not suppressed by the user.
     */
    onNonCompliance(data: NagLoggerNonComplianceData): void;
    /**
     * Called when a CfnResource does not pass the compliance check for a given rule and the rule violation is suppressed by the user.
     */
    onSuppressed(data: NagLoggerSuppressedData): void;
    /**
     * Called when a rule throws an error during while validating a CfnResource for compliance.
     */
    onError(data: NagLoggerErrorData): void;
    /**
     * Called when a rule throws an error during while validating a CfnResource for compliance and the error is suppressed.
     */
    onSuppressedError(data: NagLoggerSuppressedErrorData): void;
    /**
     * Called when a rule does not apply to the given CfnResource.
     */
    onNotApplicable(data: NagLoggerNotApplicableData): void;
}
/**
 * Props for the AnnotationLogger
 */
export interface AnnotationLoggerProps {
    /**
     * Whether or not to enable extended explanatory descriptions on warning, error, and logged ignore messages.
     */
    readonly verbose?: boolean;
    /**
     * Whether or not to log suppressed rule violations as informational messages (default: false).
     */
    readonly logIgnores?: boolean;
}
/**
 * A NagLogger that outputs to the CDK Annotations system.
 */
export declare class AnnotationLogger implements INagLogger {
    suppressionId: string;
    readonly verbose: boolean;
    readonly logIgnores: boolean;
    constructor(props?: AnnotationLoggerProps);
    onCompliance(_data: NagLoggerComplianceData): void;
    onNonCompliance(data: NagLoggerNonComplianceData): void;
    onSuppressed(data: NagLoggerSuppressedData): void;
    onError(data: NagLoggerErrorData): void;
    onSuppressedError(data: NagLoggerSuppressedErrorData): void;
    onNotApplicable(_data: NagLoggerNotApplicableData): void;
    protected createMessage(ruleId: string, findingId: string, ruleInfo: string, ruleExplanation: string, verbose: boolean): string;
}
export interface NagReportSchema {
    readonly lines: NagReportLine[];
}
export interface NagReportLine {
    readonly ruleId: string;
    readonly resourceId: string;
    readonly compliance: string;
    readonly exceptionReason: string;
    readonly ruleLevel: string;
    readonly ruleInfo: string;
}
/**
 * Possible output formats of the NagReport
 */
export declare enum NagReportFormat {
    CSV = "csv",
    JSON = "json"
}
/**
 * Props for the NagReportLogger
 */
export interface NagReportLoggerProps {
    readonly formats: NagReportFormat[];
}
/**
 * A NagLogger that creates compliance reports
 */
export declare class NagReportLogger implements INagLogger {
    private reportStacks;
    readonly formats: NagReportFormat[];
    constructor(props: NagReportLoggerProps);
    onCompliance(data: NagLoggerComplianceData): void;
    onNonCompliance(data: NagLoggerNonComplianceData): void;
    onSuppressed(data: NagLoggerSuppressedData): void;
    onError(data: NagLoggerErrorData): void;
    onSuppressedError(data: NagLoggerSuppressedErrorData): void;
    onNotApplicable(data: NagLoggerNotApplicableData): void;
    getFormatStacks(format: NagReportFormat): string[];
    /**
     * Initialize the report for the rule pack's compliance report for the resource's Stack if it doesn't exist
     * @param data
     */
    protected initializeStackReport(data: NagLoggerBaseData): void;
    protected writeToStackComplianceReport(data: NagLoggerBaseData, compliance: NagRuleStates): void;
}
