"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NagPack = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_cdk_lib_1 = require("aws-cdk-lib");
const ignore_suppression_conditions_1 = require("./ignore-suppression-conditions");
const nag_logger_1 = require("./nag-logger");
const nag_rules_1 = require("./nag-rules");
const nag_suppression_helper_1 = require("./utils/nag-suppression-helper");
/**
 * Base class for all rule packs.
 */
class NagPack {
    constructor(props) {
        this.loggers = new Array();
        this.packName = '';
        this.userGlobalSuppressionIgnore = props?.suppressionIgnoreCondition;
        this.loggers.push(new nag_logger_1.AnnotationLogger({
            verbose: props?.verbose,
            logIgnores: props?.logIgnores,
        }));
        if (props?.reports ?? true) {
            const formats = props?.reportFormats
                ? props.reportFormats
                : [nag_logger_1.NagReportFormat.CSV];
            this.loggers.push(new nag_logger_1.NagReportLogger({ formats }));
        }
        if (props?.additionalLoggers) {
            this.loggers.push(...props.additionalLoggers);
        }
    }
    get readPackName() {
        return this.packName;
    }
    /**
     * Create a rule to be used in the NagPack.
     * @param params The @IApplyRule interface with rule details.
     */
    applyRule(params) {
        if (this.packName === '') {
            throw Error('The NagPack does not have a pack name, therefore the rule could not be applied. Set a packName in the NagPack constructor.');
        }
        const allSuppressions = nag_suppression_helper_1.NagSuppressionHelper.getSuppressions(params.node);
        const ruleSuffix = params.ruleSuffixOverride
            ? params.ruleSuffixOverride
            : params.rule.name;
        const ruleId = `${this.packName}-${ruleSuffix}`;
        const base = {
            nagPackName: this.packName,
            resource: params.node,
            ruleId: ruleId,
            ruleInfo: params.info,
            ruleExplanation: params.explanation,
            ruleLevel: params.level,
        };
        try {
            const ruleCompliance = params.rule(params.node);
            if (ruleCompliance === nag_rules_1.NagRuleCompliance.COMPLIANT) {
                this.loggers.forEach((t) => t.onCompliance(base));
            }
            else if (this.isNonCompliant(ruleCompliance)) {
                const findings = this.asFindings(ruleCompliance);
                for (const findingId of findings) {
                    const suppressionReason = this.ignoreRule(allSuppressions, ruleId, findingId, params.node, params.level, params.ignoreSuppressionCondition);
                    if (suppressionReason) {
                        this.loggers.forEach((t) => t.onSuppressed({
                            ...base,
                            suppressionReason,
                            findingId,
                        }));
                    }
                    else {
                        this.loggers.forEach((t) => t.onNonCompliance({
                            ...base,
                            findingId,
                        }));
                    }
                }
            }
            else if (ruleCompliance === nag_rules_1.NagRuleCompliance.NOT_APPLICABLE) {
                this.loggers.forEach((t) => t.onNotApplicable({
                    ...base,
                }));
            }
        }
        catch (error) {
            const reason = this.ignoreRule(allSuppressions, nag_rules_1.VALIDATION_FAILURE_ID, '', params.node, params.level, params.ignoreSuppressionCondition);
            if (reason) {
                this.loggers.forEach((t) => t.onSuppressedError({
                    ...base,
                    errorMessage: error.message,
                    errorSuppressionReason: reason,
                }));
            }
            else {
                this.loggers.forEach((t) => t.onError({
                    ...base,
                    errorMessage: error.message,
                }));
            }
        }
    }
    /**
     * Check whether a specific rule should be ignored.
     * @param suppressions The suppressions listed in the cdk-nag metadata.
     * @param ruleId The id of the rule to ignore.
     * @param resource The resource being evaluated.
     * @param findingId The id of the finding that is being checked.
     * @returns The reason the rule was ignored, or an empty string.
     */
    ignoreRule(suppressions, ruleId, findingId, resource, level, ignoreSuppressionCondition) {
        for (let suppression of suppressions) {
            if (nag_suppression_helper_1.NagSuppressionHelper.doesApply(suppression, ruleId, findingId)) {
                const ignoreMessage = new ignore_suppression_conditions_1.SuppressionIgnoreOr(this.userGlobalSuppressionIgnore ?? new ignore_suppression_conditions_1.SuppressionIgnoreNever(), this.packGlobalSuppressionIgnore ?? new ignore_suppression_conditions_1.SuppressionIgnoreNever(), ignoreSuppressionCondition ?? new ignore_suppression_conditions_1.SuppressionIgnoreNever()).createMessage({
                    resource,
                    reason: suppression.reason,
                    ruleId,
                    findingId,
                    ruleLevel: level,
                });
                if (ignoreMessage) {
                    let id = findingId ? `${ruleId}[${findingId}]` : `${ruleId}`;
                    aws_cdk_lib_1.Annotations.of(resource).addInfo(`The suppression for ${id} was ignored for the following reason(s).\n\t${ignoreMessage}`);
                }
                else {
                    if (!suppression.appliesTo) {
                        // the rule is not granular so it always applies
                        return suppression.reason;
                    }
                    else {
                        return `[${findingId}] ${suppression.reason}`;
                    }
                }
            }
        }
        return '';
    }
    isNonCompliant(ruleResult) {
        return (ruleResult === nag_rules_1.NagRuleCompliance.NON_COMPLIANT ||
            Array.isArray(ruleResult));
    }
    asFindings(ruleResult) {
        if (Array.isArray(ruleResult)) {
            return ruleResult;
        }
        else {
            return [''];
        }
    }
}
exports.NagPack = NagPack;
_a = JSII_RTTI_SYMBOL_1;
NagPack[_a] = { fqn: "cdk-nag.NagPack", version: "2.27.91" };
//# sourceMappingURL=data:application/json;base64,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