"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NIST80053R5Checks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_cdk_lib_1 = require("aws-cdk-lib");
const nag_pack_1 = require("../nag-pack");
const nag_rules_1 = require("../nag-rules");
const apigw_1 = require("../rules/apigw");
const autoscaling_1 = require("../rules/autoscaling");
const cloudtrail_1 = require("../rules/cloudtrail");
const cloudwatch_1 = require("../rules/cloudwatch");
const dms_1 = require("../rules/dms");
const dynamodb_1 = require("../rules/dynamodb");
const ec2_1 = require("../rules/ec2");
const ecs_1 = require("../rules/ecs");
const efs_1 = require("../rules/efs");
const elasticache_1 = require("../rules/elasticache");
const elasticbeanstalk_1 = require("../rules/elasticbeanstalk");
const elb_1 = require("../rules/elb");
const iam_1 = require("../rules/iam");
const kms_1 = require("../rules/kms");
const lambda_1 = require("../rules/lambda");
const opensearch_1 = require("../rules/opensearch");
const rds_1 = require("../rules/rds");
const redshift_1 = require("../rules/redshift");
const s3_1 = require("../rules/s3");
const sagemaker_1 = require("../rules/sagemaker");
const secretsmanager_1 = require("../rules/secretsmanager");
const sns_1 = require("../rules/sns");
const vpc_1 = require("../rules/vpc");
const waf_1 = require("../rules/waf");
/**
 * Check for NIST 800-53 rev 5 compliance.
 * Based on the NIST 800-53 rev 5 AWS operational best practices: https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-nist-800-53_rev_5.html
 */
class NIST80053R5Checks extends nag_pack_1.NagPack {
    constructor(props) {
        super(props);
        this.packName = 'NIST.800.53.R5';
    }
    visit(node) {
        if (node instanceof aws_cdk_lib_1.CfnResource) {
            this.checkAPIGW(node);
            this.checkAutoScaling(node);
            this.checkCloudTrail(node);
            this.checkCloudWatch(node);
            this.checkDMS(node);
            this.checkDynamoDB(node);
            this.checkEC2(node);
            this.checkECS(node);
            this.checkEFS(node);
            this.checkElastiCache(node);
            this.checkElasticBeanstalk(node);
            this.checkELB(node);
            this.checkIAM(node);
            this.checkKMS(node);
            this.checkLambda(node);
            this.checkOpenSearch(node);
            this.checkRDS(node);
            this.checkRedshift(node);
            this.checkS3(node);
            this.checkSageMaker(node);
            this.checkSecretsManager(node);
            this.checkSNS(node);
            this.checkVPC(node);
            this.checkWAF(node);
        }
    }
    /**
     * Check API Gateway Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAPIGW(node) {
        this.applyRule({
            info: 'The REST API stage is not associated with AWS WAFv2 web ACL - (Control ID: AC-4(21)).',
            explanation: 'AWS WAF enables you to configure a set of rules (called a web access control list (web ACL)) that allow, block, or count web requests based on customizable web security rules and conditions that you define. Ensure your Amazon API Gateway stage is associated with a WAF Web ACL to protect it from malicious attacks.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWAssociatedWithWAF,
            node: node,
        });
        this.applyRule({
            info: 'The API Gateway stage does not have caching enabled and encrypted for all methods - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: "To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWCacheEnabledAndEncrypted,
            node: node,
        });
        this.applyRule({
            info: 'The API Gateway stage does not have execution logging enabled for all methods - (Control IDs: AC-4(26), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-4(17), SI-7(8)).',
            explanation: 'API Gateway logging displays detailed views of users who accessed the API and the way they accessed the API. This insight enables visibility of user activities.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWExecutionLoggingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The API Gateway REST API stage is not configured with SSL certificates - (Control IDs: AC-4, AC-4(22), AC-17(2), AC-24(1), AU-9(3), CA-9b, IA-5(1)(c), PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2).',
            explanation: 'Ensure Amazon API Gateway REST API stages are configured with SSL certificates to allow backend systems to authenticate that requests originate from API Gateway.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWSSLEnabled,
            node: node,
        });
    }
    /**
     * Check Auto Scaling Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAutoScaling(node) {
        this.applyRule({
            info: 'The Auto Scaling group (which is associated with a load balancer) does not utilize ELB healthchecks - (Control IDs: AU-12(3), AU-14a, AU-14b, CA-2(2), CA-7, CA-7b, CM-6a, CM-9b, PM-14a.1, PM-14b, PM-31, SC-6, SC-36(1)(a), SI-2a).',
            explanation: 'The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability. The load balancer periodically sends pings, attempts connections, or sends requests to test Amazon EC2 instances health in an auto-scaling group. If an instance is not reporting back, traffic is sent to a new Amazon EC2 instance.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingGroupELBHealthCheckRequired,
            node: node,
        });
        this.applyRule({
            info: 'The Auto Scaling launch configuration does not have public IP addresses disabled - (Control IDs: AC-3, AC-4(21), CM-6a, SC-7(3)).',
            explanation: 'If you configure your Network Interfaces with a public IP address, then the associated resources to those Network Interfaces are reachable from the internet. EC2 resources should not be publicly accessible, as this may allow unintended access to your applications or servers.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingLaunchConfigPublicIpDisabled,
            node: node,
        });
    }
    /**
     * Check CloudTrail Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudTrail(node) {
        this.applyRule({
            info: 'The trail does not have CloudWatch logs enabled - (Control IDs: AC-2(4), AC-3(1), AC-3(10), AC-4(26), AC-6(9), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-4(1), AU-6(1), AU-6(3), AU-6(4), AU-6(5), AU-6(6), AU-6(9), AU-7(1), AU-8b, AU-9(7), AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), AU-16, CA-7b, CM-5(1)(b), CM-6a, CM-9b, IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-1(1)(c), SI-3(8)(b), SI-4(2), SI-4(17), SI-4(20), SI-7(8), SI-10(1)(c)).',
            explanation: 'Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.CloudTrailCloudWatchLogsEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The trail does not have encryption enabled - (Control IDs: AU-9(3), CM-6a, CM-9b, CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.CloudTrailEncryptionEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The trail does not have log file validation enabled - (Control IDs: AU-9a, CM-6a, CM-9b, PM-11b, PM-17b, SA-1(1), SA-10(1), SC-16(1), SI-1a.2, SI-1a.2, SI-1c.2, SI-4d, SI-7a, SI-7(1), SI-7(3), SI-7(7)).',
            explanation: 'Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.CloudTrailLogFileValidationEnabled,
            node: node,
        });
    }
    /**
     * Check CloudWatch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudWatch(node) {
        this.applyRule({
            info: 'The CloudWatch alarm does not have at least one alarm action, one INSUFFICIENT_DATA action, or one OK action enabled - (Control IDs: AU-6(1), AU-6(5), AU-12(3), AU-14a, AU-14b, CA-2(2), CA-7, CA-7b, PM-14a.1, PM-14b, PM-31, SC-36(1)(a), SI-2a, SI-4(12), SI-5b, SI-5(1)).',
            explanation: 'Amazon CloudWatch alarms alert when a metric breaches the threshold for a specified number of evaluation periods. The alarm performs one or more actions based on the value of the metric or expression relative to a threshold over a number of time periods.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.CloudWatchAlarmAction,
            node: node,
        });
        this.applyRule({
            info: 'The CloudWatch Log Group is not encrypted with an AWS KMS key - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.CloudWatchLogGroupEncrypted,
            node: node,
        });
        this.applyRule({
            info: 'The CloudWatch Log Group does not have an explicit retention period configured - (Control IDs: AC-16b, AT-4b, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-10, AU-11(1), AU-11, AU-12(1), AU-12(2), AU-12(3), AU-14a, AU-14b, CA-7b, PM-14a.1, PM-14b, PM-21b, PM-31, SC-28(2), SI-4(17), SI-12).',
            explanation: 'Ensure a minimum duration of event log data is retained for your log groups to help with troubleshooting and forensics investigations. The lack of available past event log data makes it difficult to reconstruct and identify potentially malicious events.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.CloudWatchLogGroupRetentionPeriod,
            node: node,
        });
    }
    /**
     * Check DMS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDMS(node) {
        this.applyRule({
            info: 'The DMS replication instance is public - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'DMS replication instances can contain sensitive information and access control is required for such accounts.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: dms_1.DMSReplicationNotPublic,
            node: node,
        });
    }
    /**
     * Check DynamoDB Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDynamoDB(node) {
        this.applyRule({
            info: "The provisioned capacity DynamoDB table does not have Auto Scaling enabled on it's indexes - (Control IDs: CP-1a.1(b), CP-1a.2, CP-2a, CP-2a.6, CP-2a.7, CP-2d, CP-2e, CP-2(5), CP-2(6), CP-6(2), CP-10, SC-5(2), SC-6, SC-22, SC-36, SI-13(5)).",
            explanation: 'Amazon DynamoDB auto scaling uses the AWS Application Auto Scaling service to adjust provisioned throughput capacity that automatically responds to actual traffic patterns. This enables a table or a global secondary index to increase its provisioned read/write capacity to handle sudden increases in traffic, without throttling.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.DynamoDBAutoScalingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The DynamoDB table is not in an AWS Backup plan - (Control IDs: CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'To help with data back-up processes, ensure your Amazon DynamoDB tables are a part of an AWS Backup plan. AWS Backup is a fully managed backup service with a policy-based backup solution. This solution simplifies your backup management and enables you to meet your business and regulatory backup compliance requirements.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.DynamoDBInBackupPlan,
            node: node,
        });
        this.applyRule({
            info: 'The DynamoDB table does not have Point-in-time Recovery enabled - (Control IDs: CP-1(2), CP-2(5), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'The recovery maintains continuous backups of your table for the last 35 days.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.DynamoDBPITREnabled,
            node: node,
        });
    }
    /**
     * Check EC2 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEC2(node) {
        this.applyRule({
            info: 'The EBS volume is not in an AWS Backup plan - (Control IDs: CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'To help with data back-up processes, ensure your Amazon Elastic Block Store (Amazon EBS) volumes are a part of an AWS Backup plan. AWS Backup is a fully managed backup service with a policy-based backup solution. This solution simplifies your backup management and enables you to meet your business and regulatory backup compliance requirements.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2EBSInBackupPlan,
            node: node,
        });
        this.applyRule({
            info: "The EC2 instance type 'supports' EBS optimization and does not have EBS optimization enabled - (Control IDs: CP-2(5), CP-9a, CP-9b, CP-9c, CP-10, SC-5(2)).",
            explanation: 'An optimized instance in Amazon Elastic Block Store (Amazon EBS) provides additional, dedicated capacity for Amazon EBS I/O operations. This optimization provides the most efficient performance for your EBS volumes by minimizing contention between Amazon EBS I/O operations and other traffic from your instance.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2EBSOptimizedInstance,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance does not have IMDSV2 (Instance Metadata Service Version 2) enabled - (Control ID: AC-2(6), AC-3, AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(7), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-6, AC-24, CM-5(1)(a), MP-2, SC-23(3)).',
            explanation: 'Instance Metadata Service Version 2 (IMDSv2) helps protect access and control of Amazon Elastic Compute Cloud (Amazon EC2) instance metadata. The IMDSv2 method uses session-based controls. With IMDSv2, controls can be implemented to restrict changes to instance metadata.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2IMDSv2Enabled,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance is associated with a public IP address - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'Manage access to the AWS Cloud by ensuring Amazon Elastic Compute Cloud (Amazon EC2) instances cannot be publicly accessed. Amazon EC2 instances can contain sensitive information and access control is required for such accounts.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstanceNoPublicIp,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance does not have an instance profile attached - (Control IDs: AC-3, CM-5(1)(a), CM-6a).',
            explanation: 'EC2 instance profiles pass an IAM role to an EC2 instance. Attaching an instance profile to your instances can assist with least privilege and permissions management.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstanceProfileAttached,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance is not within a VPC - (Control IDs: AC-2(6), AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-25).',
            explanation: 'Deploy Amazon Elastic Compute Cloud (Amazon EC2) instances within an Amazon Virtual Private Cloud (Amazon VPC) to enable secure communication between an instance and other services within the amazon VPC, without requiring an internet gateway, NAT device, or VPN connection. All traffic remains securely within the AWS Cloud. Because of their logical isolation, domains that reside within anAmazon VPC have an extra layer of security when compared to domains that use public endpoints. Assign Amazon EC2 instances to an Amazon VPC to properly manage access.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstancesInVPC,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance allows unrestricted inbound IPv4 TCP traffic on one or more common ports (by default these ports include 20, 21, 3389, 3309, 3306, 4333) - (Control IDs: AC-4(21), AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), CM-2a, CM-2(2), CM-6a, CM-7b, CM-8(6), CM-9b, SC-7a, SC-7c, SC-7(5), SC-7(7), SC-7(11), SC-7(12), SC-7(16), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28)).',
            explanation: 'Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2RestrictedCommonPorts,
            node: node,
        });
        this.applyRule({
            info: 'The Security Group allows unrestricted SSH access - (Control IDs: AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), CM-9b, SC-7a, SC-7c, SC-7(7), SC-7(11), SC-7(12), SC-7(16), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28)).',
            explanation: 'Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2RestrictedSSH,
            node: node,
        });
    }
    /**
     * Check ECS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkECS(node) {
        this.applyRule({
            info: "The ECS task definition is configured for host networking and has at least one container with definitions with 'privileged' set to false or empty or 'user' set to root or empty - (Control IDs: AC-3, AC-5b, CM-5(1)(a)).",
            explanation: 'If a task definition has elevated privileges it is because you have specifically opted-in to those configurations. This rule checks for unexpected privilege escalation when a task definition has host networking enabled but the customer has not opted-in to elevated privileges.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ecs_1.ECSTaskDefinitionUserForHostMode,
            node: node,
        });
    }
    /**
     * Check EFS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEFS(node) {
        this.applyRule({
            info: 'The EFS is not in an AWS Backup plan - (Control IDs: CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'To help with data back-up processes, ensure your Amazon Elastic File System (Amazon EFS) file systems are a part of an AWS Backup plan. AWS Backup is a fully managed backup service with a policy-based backup solution. This solution simplifies your backup management and enables you to meet your business and regulatory backup compliance requirements.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: efs_1.EFSInBackupPlan,
            node: node,
        });
        this.applyRule({
            info: 'The EFS does not have encryption at rest enabled - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: efs_1.EFSEncrypted,
            node: node,
        });
    }
    /**
     * Check ElastiCache Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkElastiCache(node) {
        this.applyRule({
            info: 'The ElastiCache Redis cluster does not retain automatic backups for at least 15 days - (Control IDs: CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'Automatic backups can help guard against data loss. If a failure occurs, you can create a new cluster, which restores your data from the most recent backup.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheRedisClusterAutomaticBackup,
            node: node,
        });
    }
    /**
     * Check Elastic Beanstalk Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkElasticBeanstalk(node) {
        this.applyRule({
            info: 'The Elastic Beanstalk environment does not have enhanced health reporting enabled - (Control IDs: AU-12(3), AU-14a, AU-14b, CA-2(2), CA-7, CA-7b, PM-14a.1, PM-14b, PM-31, SC-6, SC-36(1)(a), SI-2a).',
            explanation: 'AWS Elastic Beanstalk enhanced health reporting enables a more rapid response to changes in the health of the underlying infrastructure. These changes could result in a lack of availability of the application. Elastic Beanstalk enhanced health reporting provides a status descriptor to gauge the severity of the identified issues and identify possible causes to investigate.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elasticbeanstalk_1.ElasticBeanstalkEnhancedHealthReportingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The Elastic Beanstalk environment does not have managed updates enabled - (Control IDs: SI-2c, SI-2d, SI-2(2), SI-2(5)).',
            explanation: 'Enabling managed platform updates for an Amazon Elastic Beanstalk environment ensures that the latest available platform fixes, updates, and features for the environment are installed. Keeping up to date with patch installation is a best practice in securing systems.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elasticbeanstalk_1.ElasticBeanstalkManagedUpdatesEnabled,
            node: node,
        });
    }
    /**
     * Check Elastic Load Balancer Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkELB(node) {
        this.applyRule({
            info: "The ALB's HTTP listeners are not configured to redirect to HTTPS - (Control IDs: AC-4, AC-4(22), AC-17(2), AC-24(1), AU-9(3), CA-9b, IA-5(1)(c), PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2).",
            explanation: 'To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ALBHttpToHttpsRedirection,
            node: node,
        });
        this.applyRule({
            info: 'The ALB is not associated with AWS WAFv2 web ACL - (Control ID: AC-4(21)).',
            explanation: 'A WAF helps to protect your web applications or APIs against common web exploits. These web exploits may affect availability, compromise security, or consume excessive resources within your environment.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ALBWAFEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: AC-4, AC-4(22), AC-17(2), AC-24(1), AU-9(3), CA-9b, IA-5(1)(c), PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SC-23(5), SI-1a.2, SI-1a.2, SI-1c.2).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBACMCertificateRequired,
            node: node,
        });
        this.applyRule({
            info: 'The CLB does not balance traffic between at least 2 Availability Zones - (Control IDs: CP-1a.1(b), CP-1a.2, CP-2a, CP-2a.6, CP-2a.7, CP-2d, CP-2e, CP-2(5), CP-2(6), CP-6(2), CP-10, SC-5(2), SC-6, SC-22, SC-36, SI-13(5)).',
            explanation: "Enable cross-zone load balancing for your Classic Load Balancers (CLBs) to help maintain adequate capacity and availability. The cross-zone load balancing reduces the need to maintain equivalent numbers of instances in each enabled availability zone. It also improves your application's ability to handle the loss of one or more instances.",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBCrossZoneLoadBalancingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The ALB, NLB, or GLB does not have deletion protection enabled - (Control IDs: CA-7(4)(c), CM-2a, CM-2(2), CM-3a, CM-8(6), CP-1a.1(b), CP-1a.2, CP-2a, CP-2a.6, CP-2a.7, CP-2d, CP-2e, CP-2(5), SA-15a.4, SC-5(2), SC-22).',
            explanation: 'This rule ensures that Elastic Load Balancing has deletion protection enabled. Use this feature to prevent your load balancer from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The ELB does not have logging enabled - (Control IDs: AC-4(26), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-4(17), SI-7(8)).',
            explanation: "Elastic Load Balancing activity is a central point of communication within an environment. Ensure ELB logging is enabled. The collected data provides detailed information about requests sent to The ELB. Each log contains information such as the time the request was received, the client's IP address, latencies, request paths, and server responses.",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBLoggingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The CLB does not restrict its listeners to only the SSL and HTTPS protocols - (Control IDs: AC-4, AC-4(22), AC-17(2), AC-24(1), AU-9(3), CA-9b, IA-5(1)(c), PM-17b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1a.2, SI-1a.2, SI-1c.2, SI-1c.2).',
            explanation: 'Ensure that your Classic Load Balancers (CLBs) are configured with SSL or HTTPS listeners. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBTlsHttpsListenersOnly,
            node: node,
        });
        this.applyRule({
            info: 'The ALB, NLB, or GLB listener does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: SC-8(1), SC-23(5)).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBv2ACMCertificateRequired,
            node: node,
        });
    }
    /**
     * Check IAM Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkIAM(node) {
        this.applyRule({
            info: 'The IAM Group, User, or Role contains an inline policy - (Control IDs: AC-2i.2, AC-2(1), AC-2(6), AC-3, AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(7), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-6, AC-6(3), AC-24, CM-5(1)(a), CM-6a, CM-9b, MP-2, SC-23(3)).',
            explanation: 'AWS recommends to use managed policies instead of inline policies. The managed policies allow reusability, versioning and rolling back, and delegating permissions management.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMNoInlinePolicy,
            node: node,
        });
        this.applyRule({
            info: 'The IAM policy grants admin access, meaning the policy allows a principal to perform all actions on all resources - (Control IDs: AC-2i.2, AC-2(1), AC-2(6), AC-3, AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(7), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-5b, AC-6, AC-6(2), AC-6(3), AC-6(10), AC-24, CM-5(1)(a), CM-6a, CM-9b, MP-2, SC-23(3), SC-25).',
            explanation: 'AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, by ensuring that IAM groups have at least one IAM user. Placing IAM users in groups based on their associated permissions or job function is one way to incorporate least privilege.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMPolicyNoStatementsWithAdminAccess,
            node: node,
        });
        this.applyRule({
            info: 'The IAM policy grants full access, meaning the policy allows a principal to perform all actions on individual resources - (Control IDs: AC-3, AC-5b, AC-6(2), AC-6(10), CM-5(1)(a)).',
            explanation: 'Ensure IAM Actions are restricted to only those actions that are needed. Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMPolicyNoStatementsWithFullAccess,
            node: node,
        });
        this.applyRule({
            info: 'The IAM user does not belong to any group(s) - (Control IDs: AC-2i.2, AC-2(1), AC-2(6), AC-3, AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(7), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-6, AC-6(3), AC-24, CM-5(1)(a), CM-6a, CM-9b, MP-2, SC-23(3)).',
            explanation: 'AWS Identity and Access Management (IAM) can help you restrict access permissions and authorizations by ensuring IAM users are members of at least one group. Allowing users more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMUserGroupMembership,
            node: node,
        });
        this.applyRule({
            info: 'The IAM policy is attached at the user level - (Control IDs: AC-2i.2, AC-2(1), AC-2(6), AC-3, AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(7), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-6, AC-6(3), AC-24, CM-5(1)(a), CM-6a, CM-9b, MP-2, SC-23(3), SC-25).',
            explanation: 'Assigning privileges at the group or the role level helps to reduce opportunity for an identity to receive or retain excessive privileges.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMUserNoPolicies,
            node: node,
        });
    }
    /**
     * Check KMS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkKMS(node) {
        this.applyRule({
            info: 'The KMS Symmetric key does not have automatic key rotation enabled - (Control IDs: CM-6a, CM-9b, SA-9(6), SC-12, SC-12(2), SC-12(6)).',
            explanation: 'Enable key rotation to ensure that keys are rotated once they have reached the end of their crypto period.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: kms_1.KMSBackingKeyRotationEnabled,
            node: node,
        });
    }
    /**
     * Check Lambda Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkLambda(node) {
        this.applyRule({
            info: 'The Lambda function is not configured with function-level concurrent execution limits - (Control IDs: AU-12(3), AU-14a, AU-14b, CA-7, CA-7b, PM-14a.1, PM-14b, PM-31, SC-6).',
            explanation: "Ensure that a Lambda function's concurrency high and low limits are established. This can assist in baselining the number of requests that your function is serving at any given time.",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: lambda_1.LambdaConcurrency,
            node: node,
        });
        this.applyRule({
            info: 'The Lambda function is not configured with a dead-letter configuration - (Control IDs: AU-12(3), AU-14a, AU-14b, CA-2(2), CA-7, CA-7b, PM-14a.1, PM-14b, PM-31, SC-36(1)(a), SI-2a).',
            explanation: 'Notify the appropriate personnel through Amazon Simple Queue Service (Amazon SQS) or Amazon Simple Notification Service (Amazon SNS) when a function has failed.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: lambda_1.LambdaDLQ,
            node: node,
        });
        this.applyRule({
            info: 'The Lambda function permission grants public access - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'Public access allows anyone on the internet to perform unauthenticated actions on your function and can potentially lead to degraded availability.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: lambda_1.LambdaFunctionPublicAccessProhibited,
            node: node,
        });
        this.applyRule({
            info: 'The Lambda function is not VPC enabled - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-25).',
            explanation: 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: lambda_1.LambdaInsideVPC,
            node: node,
        });
    }
    /**
     * Check OpenSearch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkOpenSearch(node) {
        this.applyRule({
            info: 'The OpenSearch Service domain does not have encryption at rest enabled - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchEncryptedAtRest,
            node: node,
        });
        this.applyRule({
            info: 'The OpenSearch Service domain is not running within a VPC - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-25).',
            explanation: 'VPCs help secure your AWS resources and provide an extra layer of protection.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchInVPCOnly,
            node: node,
        });
        this.applyRule({
            info: 'The OpenSearch Service domain does not stream error logs (ES_APPLICATION_LOGS) to CloudWatch Logs - (Control ID: AU-10).',
            explanation: 'Ensure Amazon OpenSearch Service domains have error logs enabled and streamed to Amazon CloudWatch Logs for retention and response. Domain error logs can assist with security and access audits, and can help to diagnose availability issues.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchErrorLogsToCloudWatch,
            node: node,
        });
        this.applyRule({
            info: 'The OpenSearch Service domain does not have node-to-node encryption enabled - (Control IDs: AC-4, AC-4(22), AC-24(1), AU-9(3), CA-9b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2).',
            explanation: 'Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchNodeToNodeEncryption,
            node: node,
        });
    }
    /**
     * Check RDS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRDS(node) {
        this.applyRule({
            info: 'The RDS DB Instance does not have enhanced monitoring enabled - (Control IDs: AU-12(3), AU-14a, AU-14b, CA-2(2), CA-7, CA-7b, PM-14a.1, PM-14b, PM-31, SC-36(1)(a), SI-2a).',
            explanation: 'Enable enhanced monitoring to help monitor Amazon RDS availability. This provides detailed visibility into the health of your Amazon RDS database instances.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSEnhancedMonitoringEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB instance is not in an AWS Backup plan - (Control IDs: CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'To help with data back-up processes, ensure your Amazon Relational Database Service (Amazon RDS) instances are a part of an AWS Backup plan. AWS Backup is a fully managed backup service with a policy-based backup solution. This solution simplifies your backup management and enables you to meet your business and regulatory backup compliance requirements.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInBackupPlan,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB Instance does not have backup enabled - (Control IDs: CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'The backup feature of Amazon RDS creates backups of your databases and transaction logs.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceBackupEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB Instance or Aurora Cluster does not have deletion protection enabled - (Control IDs: CA-7(4)(c), CM-3a, CP-1a.1(b), CP-1a.2, CP-2a, CP-2a.6, CP-2a.7, CP-2d, CP-2e, CP-2(5), SA-15a.4, SC-5(2), SC-22, SI-13(5)).',
            explanation: 'Ensure Amazon Relational Database Service (Amazon RDS) instances and clusters have deletion protection enabled. Use deletion protection to prevent your Amazon RDS DB instances and clusters from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB Instance allows public access - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'Amazon RDS database instances can contain sensitive information, and principles and access control is required for such accounts.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstancePublicAccess,
            node: node,
        });
        this.applyRule({
            info: 'The non-Aurora RDS DB instance or Aurora cluster does not have all CloudWatch log types exported - (Control IDs: AC-2(4), AC-3(1), AC-3(10), AC-4(26), AC-6(9), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-1(1)(c), SI-3(8)(b), SI-4(2), SI-4(17), SI-4(20), SI-7(8), SI-10(1)(c)).',
            explanation: 'To help with logging and monitoring within your environment, ensure Amazon Relational Database Service (Amazon RDS) logging is enabled. With Amazon RDS logging, you can capture events such as connections, disconnections, queries, or tables queried.' +
                "This is a granular rule that returns individual findings that can be suppressed with 'appliesTo'. The findings are in the format 'LogExport::<log>' for exported logs. Example: appliesTo: ['LogExport::audit'].",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSLoggingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB Instance does not have multi-AZ support - (Control IDs: CP-1a.1(b), CP-1a.2, CP-2a, CP-2a.6, CP-2a.7, CP-2d, CP-2e, CP-2(5), CP-2(6), CP-6(2), CP-10, SC-5(2), SC-6, SC-22, SC-36, SI-13(5)).',
            explanation: 'Multi-AZ support in Amazon Relational Database Service (Amazon RDS) provides enhanced availability and durability for database instances. When you provision a Multi-AZ database instance, Amazon RDS automatically creates a primary database instance, and synchronously replicates the data to a standby instance in a different Availability Zone. In case of an infrastructure failure, Amazon RDS performs an automatic failover to the standby so that you can resume database operations as soon as the failover is complete.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSMultiAZSupport,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB Instance or Aurora Cluster does not have storage encrypted - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'Because sensitive data can exist at rest in Amazon RDS instances, enable encryption at rest to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSStorageEncrypted,
            node: node,
        });
    }
    /**
     * Check Redshift Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRedshift(node) {
        this.applyRule({
            info: 'The Redshift cluster does not have automated snapshots enabled or the retention period is not between 1 and 35 days - (Control IDs: CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'To help with data back-up processes, ensure your Amazon Redshift clusters have automated snapshots. When automated snapshots are enabled for a cluster, Redshift periodically takes snapshots of that cluster. By default, Redshift takes a snapshot every eight hours or every 5 GB per node of data changes, or whichever comes first.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftBackupEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster does not have encryption or audit logging enabled - (Control IDs: AC-2(4), AC-3(1), AC-3(10), AC-4(26), AC-6(9), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-9(3), AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), CP-9d, IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-1(1)(c), SI-3(8)(b), SI-4(2), SI-4(17), SI-4(20), SI-7(8), SI-10(1)(c), SI-19(4)).',
            explanation: 'To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterConfiguration,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster does not have version upgrades enabled, automated snapshot retention periods enabled, and an explicit maintenance window configured - (Control IDs: CM-2b, CM-2b.1, CM-2b.2, CM-2b.3, CM-3(3), CP-9a, CP-9b, CP-9c, SC-5(2), SI-2c, SI-2d, SI-2(2), SI-2(5)).',
            explanation: 'Ensure that Amazon Redshift clusters have the preferred settings for your organization. Specifically, that they have preferred maintenance windows and automated snapshot retention periods for the database.                                                                            ',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterMaintenanceSettings,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster allows public access - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'Amazon Redshift clusters can contain sensitive information and principles and access control is required for such accounts.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterPublicAccess,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster does not have enhanced VPC routing enabled - (Control IDs: AC-4(21), SC-7b).',
            explanation: 'Enhanced VPC routing forces all COPY and UNLOAD traffic between the cluster and data repositories to go through your Amazon VPC. You can then use VPC features such as security groups and network access control lists to secure network traffic. You can also use VPC flow logs to monitor network traffic.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftEnhancedVPCRoutingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster does not require TLS/SSL encryption - (Control IDs: AC-4, AC-4(22), AC-24(1), AU-9(3), CA-9b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2).',
            explanation: 'Ensure that your Amazon Redshift clusters require TLS/SSL encryption to connect to SQL clients. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftRequireTlsSSL,
            node: node,
        });
    }
    /**
     * Check Amazon S3 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkS3(node) {
        this.applyRule({
            info: 'The S3 bucket does not prohibit public access through bucket level settings - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'Keep sensitive data safe from unauthorized remote users by preventing public access at the bucket level.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketLevelPublicAccessProhibited,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Buckets does not have server access logs enabled - (Control IDs: AC-2(4), AC-3(1), AC-3(10), AC-4(26), AC-6(9), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), CM-6a, CM-9b, IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-1(1)(c), SI-3(8)(b), SI-4(2), SI-4(17), SI-4(20), SI-7(8), SI-10(1)(c)).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketLoggingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), CM-6a, CM-9b, MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketPublicReadProhibited,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), CM-6a, CM-9b, MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketPublicWriteProhibited,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not have replication enabled - (Control IDs: AU-9(2), CM-6a, CM-9b, CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) Cross-Region Replication (CRR) supports maintaining adequate capacity and availability. CRR enables automatic, asynchronous copying of objects across Amazon S3 buckets to help ensure that data availability is maintained.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketReplicationEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket or bucket policy does not require requests to use SSL - (Control IDs: AC-4, AC-4(22), AC-17(2), AC-24(1), AU-9(3), CA-9b, CM-6a, CM-9b, IA-5(1)(c), PM-11b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-16(1), SC-23, SI-1a.2, SI-1a.2, SI-1c.2).',
            explanation: 'To help protect data in transit, ensure that your Amazon Simple Storage Service (Amazon S3) buckets require requests to use Secure Socket Layer (SSL). Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketSSLRequestsOnly,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not have versioning enabled - (Control IDs: AU-9(2), CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), PM-11b, PM-17b, SC-5(2), SC-16(1), SI-1a.2, SI-1a.2, SI-1c.2, SI-13(5)).',
            explanation: 'Use versioning to preserve, retrieve, and restore every version of every object stored in your Amazon S3 bucket. Versioning helps you to easily recover from unintended user actions and application failures.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketVersioningEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket is not encrypted with a KMS Key by default - (Control IDs: AU-9(3), CP-9d, CP-9(8), SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'Ensure that encryption is enabled for your Amazon Simple Storage Service (Amazon S3) buckets. Because sensitive data can exist at rest in an Amazon S3 bucket, enable encryption at rest to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3DefaultEncryptionKMS,
            node: node,
        });
    }
    /**
     * Check SageMaker Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSageMaker(node) {
        this.applyRule({
            info: 'The SageMaker resource endpoint is not encrypted with a KMS key - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'Because sensitive data can exist at rest in SageMaker endpoint, enable encryption at rest to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerEndpointConfigurationKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            info: 'The SageMaker notebook is not encrypted with a KMS key - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'Because sensitive data can exist at rest in SageMaker notebook, enable encryption at rest to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookInstanceKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            info: 'The SageMaker notebook does not disable direct internet access - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookNoDirectInternetAccess,
            node: node,
        });
    }
    /**
     * Check Secrets Manager Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSecretsManager(node) {
        this.applyRule({
            info: 'The secret does not have automatic rotation scheduled - (Control IDs: AC-4, AC-4(22), AC-24(1), AU-9(3), CA-9b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2).',
            explanation: 'Rotating secrets on a regular schedule can shorten the period a secret is active, and potentially reduce the business impact if the secret is compromised.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: secretsmanager_1.SecretsManagerRotationEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The secret is not encrypted with a KMS Customer managed key - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'To help protect data at rest, ensure encryption with AWS Key Management Service (AWS KMS) is enabled for AWS Secrets Manager secrets. Because sensitive data can exist at rest in Secrets Manager secrets, enable encryption at rest to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: secretsmanager_1.SecretsManagerUsingKMSKey,
            node: node,
        });
    }
    /**
     * Check Amazon SNS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSNS(node) {
        this.applyRule({
            info: 'The SNS topic does not have KMS encryption enabled - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1)).',
            explanation: 'To help protect data at rest, ensure that your Amazon Simple Notification Service (Amazon SNS) topics require encryption using AWS Key Management Service (AWS KMS) Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: sns_1.SNSEncryptedKMS,
            node: node,
        });
    }
    /**
     * Check VPC Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkVPC(node) {
        this.applyRule({
            info: "The VPC's default security group allows inbound or outbound traffic - (Control IDs: AC-4(21), AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), CM-6a, CM-9b, SC-7a, SC-7c, SC-7(5), SC-7(7), SC-7(11), SC-7(12), SC-7(16), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28)).",
            explanation: 'Amazon Elastic Compute Cloud (Amazon EC2) security groups can help in the management of network access by providing stateful filtering of ingress and egress network traffic to AWS resources. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources.',
            level: nag_rules_1.NagMessageLevel.WARN,
            rule: vpc_1.VPCDefaultSecurityGroupClosed,
            node: node,
        });
        this.applyRule({
            info: 'The VPC does not have an associated Flow Log - (Control IDs: AC-4(26), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), CM-6a, CM-9b, IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SI-4(17), SI-7(8)).',
            explanation: 'The VPC flow logs provide detailed records for information about the IP traffic going to and from network interfaces in your Amazon Virtual Private Cloud (Amazon VPC). By default, the flow log record includes values for the different components of the IP flow, including the source, destination, and protocol.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: vpc_1.VPCFlowLogsEnabled,
            node: node,
        });
        this.applyRule({
            info: "The route table may contain one or more unrestricted route(s) to an IGW ('0.0.0.0/0' or '::/0') - (Control IDs: AC-4(21), CM-7b).",
            explanation: 'Ensure Amazon EC2 route tables do not have unrestricted routes to an internet gateway. Removing or limiting the access to the internet for workloads within Amazon VPCs can reduce unintended access within your environment.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: vpc_1.VPCNoUnrestrictedRouteToIGW,
            node: node,
        });
        this.applyRule({
            info: 'The subnet auto-assigns public IP addresses - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'Manage access to the AWS Cloud by ensuring Amazon Virtual Private Cloud (VPC) subnets are not automatically assigned a public IP address. Amazon Elastic Compute Cloud (EC2) instances that are launched into subnets that have this attribute enabled have a public IP address assigned to their primary network interface.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: vpc_1.VPCSubnetAutoAssignPublicIpDisabled,
            node: node,
        });
    }
    /**
     * Check WAF Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkWAF(node) {
        this.applyRule({
            info: 'The WAFv2 web ACL does not have logging enabled - (Control IDs: AC-4(26), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-4(17), SI-7(8)).',
            explanation: 'AWS WAF logging provides detailed information about the traffic that is analyzed by your web ACL. The logs record the time that AWS WAF received the request from your AWS resource, information about the request, and an action for the rule that each request matched.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: waf_1.WAFv2LoggingEnabled,
            node: node,
        });
    }
}
exports.NIST80053R5Checks = NIST80053R5Checks;
_a = JSII_RTTI_SYMBOL_1;
NIST80053R5Checks[_a] = { fqn: "cdk-nag.NIST80053R5Checks", version: "2.27.91" };
//# sourceMappingURL=data:application/json;base64,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