"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_msk_1 = require("aws-cdk-lib/aws-msk");
const nag_rules_1 = require("../../nag-rules");
/**
 * MSK clusters send broker logs to a supported destination
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_msk_1.CfnCluster) {
        const loggingInfo = aws_cdk_lib_1.Stack.of(node).resolve(node.loggingInfo);
        if (loggingInfo == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const resolvedBrokerLogs = aws_cdk_lib_1.Stack.of(node).resolve(loggingInfo.brokerLogs);
        let enabled = false;
        const s3 = aws_cdk_lib_1.Stack.of(node).resolve(resolvedBrokerLogs.s3);
        if (s3 != undefined) {
            const s3Enabled = nag_rules_1.NagRules.resolveIfPrimitive(node, s3.enabled);
            if (s3Enabled) {
                enabled = true;
            }
        }
        const cloudWatchLogs = aws_cdk_lib_1.Stack.of(node).resolve(resolvedBrokerLogs.cloudWatchLogs);
        if (cloudWatchLogs != undefined) {
            const cloudWatchLogsEnabled = nag_rules_1.NagRules.resolveIfPrimitive(node, cloudWatchLogs.enabled);
            if (cloudWatchLogsEnabled) {
                enabled = true;
            }
        }
        const firehose = aws_cdk_lib_1.Stack.of(node).resolve(resolvedBrokerLogs.firehose);
        if (firehose != undefined) {
            const firehoseEnabled = nag_rules_1.NagRules.resolveIfPrimitive(node, firehose.enabled);
            if (firehoseEnabled) {
                enabled = true;
            }
        }
        if (!enabled) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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