"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_sqs_1 = require("aws-cdk-lib/aws-sqs");
const nag_rules_1 = require("../../nag-rules");
const flatten_cfn_reference_1 = require("../../utils/flatten-cfn-reference");
/**
 * SQS queues have a dead-letter queue enabled if they are not used as a dead-letter queue
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_sqs_1.CfnQueue) {
        const redrivePolicy = aws_cdk_lib_1.Stack.of(node).resolve(node.redrivePolicy);
        if (redrivePolicy === undefined) {
            const queueLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
            const queueName = aws_cdk_lib_1.Stack.of(node).resolve(node.queueName);
            let found = false;
            for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
                if (child instanceof aws_sqs_1.CfnQueue) {
                    if (isMatchingQueue(child, queueLogicalId, queueName)) {
                        found = true;
                        break;
                    }
                }
            }
            if (!found) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether a given SQS Queue uses the target SQS queue as a DLQ
 * @param node the CfnQueue to check
 * @param queueLogicalId the Cfn Logical ID of the target queue
 * @param queueName the name of the target queue
 * returns whether the CfnQueue uses the target SQS queue as a DLQ
 */
function isMatchingQueue(node, queueLogicalId, queueName) {
    const redrivePolicy = aws_cdk_lib_1.Stack.of(node).resolve(node.redrivePolicy);
    const deadLetterTargetArn = flatten_cfn_reference_1.flattenCfnReference(redrivePolicy?.deadLetterTargetArn ?? '');
    if (new RegExp(`${queueLogicalId}(?![\\w])`).test(deadLetterTargetArn) ||
        (queueName !== undefined &&
            new RegExp(`:${queueName}(?![\\w\\-_\\.])`).test(deadLetterTargetArn))) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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