import datetime

import boto3
import numpy as np
import os
import pandas as pd

from datupapi.configure.config import Config
from datupapi.extract.io import IO
from pandas.tseries.offsets import DateOffset


class Forecasting(Config):

    #DOCKER_CONFIG_PATH = os.path.join('./', 'config.yml')
    DOCKER_CONFIG_PATH = os.path.join('/opt/ml/processing/input', 'config.yml')
    io = IO(config_file=DOCKER_CONFIG_PATH, logfile='data_extraction', log_path='output/logs')

    def __init__(self, config_file, logfile, log_path, *args, **kwargs):
        Config.__init__(self, config_file=config_file, logfile=logfile)
        self.log_path = log_path

    def format_forecasts_export(self, df_forecasts):
        """
        Return a dataframe including all forecasts partitions generated by create_forecast_export API's calling

        :param df_forecasts: Dataframe with all contatenated forecasts
        :return df_forecasts: Dataframe concatenating all backtests estimates

        >>> df = format_forecasts_export(df_forecast)
        >>> df =
                        var1    var2    var3
                idx0     1       2       3
        """
        try:
            df_forecasts['date'] = pd.to_datetime(df_forecasts['date'], format='%Y-%m-%d')
            df_forecasts = df_forecasts.rename(
                columns={
                    'item_id': 'Item',
                    'date': 'Date',
                    'location': 'Location',
                    'p5': 'ForecastLo95',
                    'p20': 'ForecastLo80',
                    'p40': 'ForecastLo60',
                    'p50': 'ForecastPoint',
                    'p60': 'ForecastUp60',
                    'p80': 'ForecastUp80',
                    'p95': 'ForecastUp95'
                })
            if self.use_location:
                forecast_cols = [col for col in df_forecasts.columns if all(subcol not in col for subcol in ['Item', 'Date', 'Location'])]
            else:
                forecast_cols = [col for col in df_forecasts.columns if all(subcol not in col for subcol in ['Item', 'Date'])]
            df_forecasts[forecast_cols] = df_forecasts[forecast_cols].applymap(lambda x: 0 if x < 0 else x)\
                                                                     .applymap(lambda x: round(x))
            df_forecasts['Target'] = 0
            df_forecasts['Week'] = df_forecasts['Date'].dt.isocalendar()['week']
        except KeyError as err:
            self.logger.exception(f'Invalid column name. Please check dataframe metadata: {err}')
            raise
        return df_forecasts

    def compute_errors_from_backtests(self, df_forecast, df_backtest, error_types=['WMAPE', 'MASE']):
        """
        Return a dataframe attaching the specified errors from the averaged out values in backtests

        :param df_forecast: Original dataframe including all items' forecasts
        :param df_backtest: Dataframe including all items' backtests
        :param error_types: List of error to attach to the forecast dataframe. Default WMAPE and MASE
        :return df_forecast: Output forecast dataframe including specified errors

        >>> df = compute_errors_from_backtests(df_forecast, df_backtest, error_types=['WMAPE', 'MASE'])
        >>> df =
                        var1    var2    var3
                idx0     1       2       3
        """
        error_dict = {error_type: 'mean' for error_type in error_types}
        try:
            df_errors = df_backtest.groupby('Item', as_index=True)\
                                   .agg(error_dict)\
                                   .applymap(lambda x: round(x, 2))\
                                   .reset_index(drop=False)
            df_forecast = pd.merge(df_forecast, df_errors, on='Item', how='left')
        except KeyError as err:
            self.logger.exception(f'Invalid column name. Please check dataframe metadata: {err}')
            raise
        return df_forecast


    def compute_interval_per_backtests(self, df_forecast, df_backtest, backtest_names=None):
        """
        Return a dataframe attaching the forecast interval for each backtest dataset

        :param df_forecast: Original dataframe including all items' forecasts
        :param q_backtest: Backtesting dataset name stored in the datalake
        :param backtest_names: List of backtests names to concat, e.g. alpha, bravo, gamma, delta and eps
        :param datalake_path_: Datalake path to download the backtest files from. Do not include bucket name
        :param use_location:
        :param location:
        :return df_forecast: Output forecast dataframe including specified interval per backtest

        >>> df = compute_interval_per_backtests(df_forecast, backtest_names=['alpha', 'bravo'], datalake_path_='path/tp/backtests/folder')
        >>> df =
                        var1    var2    var3
                idx0     1       2       3
        """
        try:
            df_forecast = df_forecast.sort_values(by=["Item", "Date"])
            for backtest in backtest_names:
                df_back = df_backtest[df_backtest.backtest == backtest]
                df_back = df_back.sort_values(by=["Item", "Date"])
                df_forecast["SuggIntervalBacktest" + backtest] = df_back.SuggestedInterval.values
            sugg_interval_cols = ['SuggIntervalBacktest' + b for b in backtest_names]
            df_forecast[sugg_interval_cols] = df_forecast[sugg_interval_cols].fillna('Point')

        except KeyError as err:
            self.logger.exception(f'Invalid column name. Please check dataframe metadata: {err}')
            raise
        return df_forecast

    def compute_suggested_interval(self, df_forecast, backtest_names, method="Robust"):
        """
            Return a dataframe attaching the suggested interval per date and item record

            :param df_forecast: Original dataframe including all items' forecasts
            :param backtest_names: List of backtests names to concat, e.g. alpha, bravo, etc.
            :param datalake_path_: Datalake path to download the backtest files from. Do not include bucket name
            :return df_forecast: Output forecast dataframe including sugguested interval

            >>> df = compute_suggested_interval(df_forecast, backtest_names=['alpha', 'bravo'])
            >>> df =
                            var1    var2    var3
                    idx0     1       2       3
            """
        suggested_interval, suggested_forecast = [], []
        num_backtests = len(backtest_names)
        try:
            if method == "Lo95":
                df_forecast['SuggestedInterval'] = "Lo95"
                df_forecast['SuggestedForecast'] = df_forecast["ForecastLo95"]
            elif method == "Lo80":
                df_forecast['SuggestedInterval'] = "Lo80"
                df_forecast['SuggestedForecast'] = df_forecast["ForecastLo80"]
            elif method == "Lo60":
                df_forecast['SuggestedInterval'] = "Lo60"
                df_forecast['SuggestedForecast'] = df_forecast["ForecastLo60"]
            elif method == "Point":
                df_forecast['SuggestedInterval'] = "Point"
                df_forecast['SuggestedForecast'] = df_forecast["ForecastPoint"]
            elif method == "Up60":
                df_forecast['SuggestedInterval'] = "Up60"
                df_forecast['SuggestedForecast'] = df_forecast["ForecastUp60"]
            elif method == "Up80":
                df_forecast['SuggestedInterval'] = "Up80"
                df_forecast['SuggestedForecast'] = df_forecast["ForecastUp80"]
            elif method == "Up95":
                df_forecast['SuggestedInterval'] = "Up95"
                df_forecast['SuggestedForecast'] = df_forecast["ForecastUp95"]
            elif method == "Normal":
                for ixr, row in df_forecast.iterrows():
                    acc_point, acc_lo95, acc_lo80, acc_lo60, acc_up60, acc_up80, acc_up95 = 0, 0, 0, 0, 0, 0, 0
                    for back_ix, backtest in enumerate(self.backtest_ids):
                        backtest_factor = num_backtests - (back_ix)
                        if row['SuggIntervalBacktest' + backtest] == 'Point':
                            acc_point += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Lo95':
                            acc_lo95 += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Lo80':
                            acc_lo80 += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Lo60':
                            acc_lo60 += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Up60':
                            acc_up60 += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Up80':
                            acc_up80 += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Up95':
                            acc_up95 += backtest_factor
                    suggested_interval_dict = {
                        'Point': acc_point,
                        'Up60': acc_up60,
                        'Lo60': acc_lo60,
                        'Up80': acc_up80,
                        'Lo80': acc_lo80,
                        'Lo95': acc_lo95,
                        'Up95': acc_up95
                    }
                    suggested_interval.append(max(suggested_interval_dict, key=suggested_interval_dict.get))
                    suggested_forecast.append(row['Forecast' + max(suggested_interval_dict, key=suggested_interval_dict.get)])
                df_forecast['SuggestedInterval'] = suggested_interval
                df_forecast['SuggestedForecast'] = suggested_forecast
            elif method == "Robust":
                for ixr, row in df_forecast.iterrows():
                    acc_point, acc_lo95, acc_lo80, acc_lo60, acc_up60, acc_up80, acc_up95 = 0, 0, 0, 0, 0, 0, 0
                    for back_ix, backtest in enumerate(self.backtest_ids):
                        backtest_factor = num_backtests - (back_ix)
                        if row['SuggIntervalBacktest' + backtest] == 'Point':
                            acc_point += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Lo95':
                            acc_lo60 += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Lo80':
                            acc_lo60 += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Lo60':
                            acc_point += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Up60':
                            acc_point += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Up80':
                            acc_up60 += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Up95':
                            acc_up60 += backtest_factor
                    suggested_interval_dict = {
                        'Point': acc_point,
                        'Up60': acc_up60,
                        'Lo60': acc_lo60,
                        'Up80': acc_up80,
                        'Lo80': acc_lo80,
                        'Lo95': acc_lo95,
                        'Up95': acc_up95
                    }
                    suggested_interval.append(max(suggested_interval_dict, key=suggested_interval_dict.get))
                    suggested_forecast.append(row['Forecast' + max(suggested_interval_dict, key=suggested_interval_dict.get)])
                df_forecast['SuggestedInterval'] = suggested_interval
                df_forecast['SuggestedForecast'] = suggested_forecast
            elif method == "Up":
                for ixr, row in df_forecast.iterrows():
                    acc_point, acc_lo95, acc_lo80, acc_lo60, acc_up60, acc_up80, acc_up95 = 0, 0, 0, 0, 0, 0, 0
                    for back_ix, backtest in enumerate(self.backtest_ids):
                        backtest_factor = num_backtests - (back_ix)
                        if row['SuggIntervalBacktest' + backtest] == 'Point':
                            acc_point += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Lo95':
                            acc_up60 += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Lo80':
                            acc_up60 += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Lo60':
                            acc_point += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Up60':
                            acc_point += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Up80':
                            acc_up60 += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Up95':
                            acc_up60 += backtest_factor
                    suggested_interval_dict = {
                        'Point': acc_point,
                        'Up60': acc_up60,
                        'Lo60': acc_lo60,
                        'Up80': acc_up80,
                        'Lo80': acc_lo80,
                        'Lo95': acc_lo95,
                        'Up95': acc_up95
                    }
                    suggested_interval.append(max(suggested_interval_dict, key=suggested_interval_dict.get))
                    suggested_forecast.append(row['Forecast' + max(suggested_interval_dict, key=suggested_interval_dict.get)])
                df_forecast['SuggestedInterval'] = suggested_interval
                df_forecast['SuggestedForecast'] = suggested_forecast

            elif method == "RobustInverse":
                for ixr, row in df_forecast.iterrows():
                    acc_point, acc_lo95, acc_lo80, acc_lo60, acc_up60, acc_up80, acc_up95 = 0, 0, 0, 0, 0, 0, 0
                    for back_ix, backtest in enumerate(self.backtest_ids):
                        backtest_factor = num_backtests - (back_ix)
                        if row['SuggIntervalBacktest' + backtest] == 'Point':
                            acc_point += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Lo95':
                            acc_up60 += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Lo80':
                            acc_up60 += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Lo60':
                            acc_point += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Up60':
                            acc_point += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Up80':
                            acc_lo60 += backtest_factor
                        elif row['SuggIntervalBacktest' + backtest] == 'Up95':
                            acc_lo60 += backtest_factor
                    suggested_interval_dict = {
                        'Point': acc_point,
                        'Up60': acc_up60,
                        'Lo60': acc_lo60,
                        'Up80': acc_up80,
                        'Lo80': acc_lo80,
                        'Lo95': acc_lo95,
                        'Up95': acc_up95
                    }
                    suggested_interval.append(max(suggested_interval_dict, key=suggested_interval_dict.get))
                    suggested_forecast.append(row['Forecast' + max(suggested_interval_dict, key=suggested_interval_dict.get)])
                df_forecast['SuggestedInterval'] = suggested_interval
                df_forecast['SuggestedForecast'] = suggested_forecast
        except KeyError as err:
            self.logger.exception(f'Invalid column name. Please check dataframe metadata: {err}')
            raise
        return df_forecast

    def compute_next_suggested_interval_forecast(self, df_forecast):
        """
        Return a dataframe attaching the next suggested interval and forecast per date and item record

        :param df_forecast: Original dataframe including all items' forecasts
        :return df_forecast: Output forecast dataframe including next suggested interval and forecast

        >>> df = compute_next_suggested_interval_forecast(df_forecast)
        >>> df =
                        var1    var2    var3
                idx0     1       2       3
        """
        intervals_dict = {'0.05': 'Lo95', '0.2': 'Lo80', '0.4': 'Lo60', '0.5': 'Point', '0.6': 'Up60', '0.8': 'Up80', '0.95': 'Up95'}
        next_interval, next_forecast = [], []

        forecast_types = [col for col in df_forecast if col.startswith('Forecast')]

        try:
            if len(forecast_types) == 5:
                forecast_types_trim = [
                    self.forecast_types[0], self.forecast_types[1], self.forecast_types[3], self.forecast_types[5], self.forecast_types[6]
                ]
                for ix, row in df_forecast.iterrows():
                    if row['SuggestedInterval'] == intervals_dict.get(forecast_types_trim[0]):
                        next_interval.append(intervals_dict.get(forecast_types_trim[1]))
                        next_forecast.append(row['Forecast' + intervals_dict.get(forecast_types_trim[1])])
                    elif row['SuggestedInterval'] == intervals_dict.get(forecast_types_trim[1]):
                        next_interval.append(intervals_dict.get(forecast_types_trim[2]))
                        next_forecast.append(row['Forecast' + intervals_dict.get(forecast_types_trim[2])])
                    elif row['SuggestedInterval'] == intervals_dict.get(forecast_types_trim[2]):
                        next_interval.append(intervals_dict.get(forecast_types_trim[3]))
                        next_forecast.append(row['Forecast' + intervals_dict.get(forecast_types_trim[3])])
                    elif row['SuggestedInterval'] == intervals_dict.get(forecast_types_trim[3]):
                        next_interval.append(intervals_dict.get(forecast_types_trim[4]))
                        next_forecast.append(row['Forecast' + intervals_dict.get(forecast_types_trim[4])])
                    elif row['SuggestedInterval'] == intervals_dict.get(forecast_types_trim[4]):
                        next_interval.append(intervals_dict.get(forecast_types_trim[4]))
                        next_forecast.append(row['Forecast' + intervals_dict.get(forecast_types_trim[4])])
                    else:
                        next_interval.append(intervals_dict.get(forecast_types_trim[2]))
                        next_forecast.append(row['Forecast' + intervals_dict.get(forecast_types_trim[2])])
            else:
                for ix, row in df_forecast.iterrows():
                    if row['SuggestedInterval'] == intervals_dict.get(self.forecast_types[0]):
                        next_interval.append(intervals_dict.get(self.forecast_types[1]))
                        next_forecast.append(row['Forecast' + intervals_dict.get(self.forecast_types[1])])
                    elif row['SuggestedInterval'] == intervals_dict.get(self.forecast_types[1]):
                        next_interval.append(intervals_dict.get(self.forecast_types[2]))
                        next_forecast.append(row['Forecast' + intervals_dict.get(self.forecast_types[2])])
                    elif row['SuggestedInterval'] == intervals_dict.get(self.forecast_types[2]):
                        next_interval.append(intervals_dict.get(self.forecast_types[3]))
                        next_forecast.append(row['Forecast' + intervals_dict.get(self.forecast_types[3])])
                    elif row['SuggestedInterval'] == intervals_dict.get(self.forecast_types[3]):
                        next_interval.append(intervals_dict.get(self.forecast_types[4]))
                        next_forecast.append(row['Forecast' + intervals_dict.get(self.forecast_types[4])])
                    elif row['SuggestedInterval'] == intervals_dict.get(self.forecast_types[4]):
                        next_interval.append(intervals_dict.get(self.forecast_types[5]))
                        next_forecast.append(row['Forecast' + intervals_dict.get(self.forecast_types[5])])
                    elif row['SuggestedInterval'] == intervals_dict.get(self.forecast_types[5]):
                        next_interval.append(intervals_dict.get(self.forecast_types[6]))
                        next_forecast.append(row['Forecast' + intervals_dict.get(self.forecast_types[6])])
                    elif row['SuggestedInterval'] == intervals_dict.get(self.forecast_types[6]):
                        next_interval.append(intervals_dict.get(self.forecast_types[6]))
                        next_forecast.append(row['Forecast' + intervals_dict.get(self.forecast_types[6])])
                    else:
                        next_interval.append(intervals_dict.get(self.forecast_types[3]))
                        next_forecast.append(row['Forecast' + intervals_dict.get(self.forecast_types[3])])
            df_forecast['NextSuggestedInterval'] = next_interval
            df_forecast['NextSuggestedForecast'] = next_forecast
        except KeyError as err:
            self.logger.exception(f'Invalid column name. Please check dataframe metadata: {err}')
            raise
        return df_forecast

    def compute_back_suggested_interval_forecast(self, df_forecast):
        """
        Return a dataframe attaching the back suggested interval and forecast per date and item record

        :param df_forecast: Original dataframe including all items' forecasts
        :return df_forecast: Output forecast dataframe including back suggested interval and forecast

        >>> df = compute_back_suggested_interval_forecast(df_forecast)
        >>> df =
                        var1    var2    var3
                idx0     1       2       3
        """
        intervals_dict = {'0.05': 'Lo95', '0.2': 'Lo80', '0.4': 'Lo60', '0.5': 'Point', '0.6': 'Up60', '0.8': 'Up80', '0.95': 'Up95'}
        back_interval, back_forecast = [], []
        forecast_types = [col for col in df_forecast if col.startswith('Forecast')]
        try:
            if len(forecast_types) == 5:
                forecast_types_trim = [
                    self.forecast_types[0], self.forecast_types[1], self.forecast_types[3], self.forecast_types[5], self.forecast_types[6]
                ]
                for ix, row in df_forecast.iterrows():
                    if row['SuggestedInterval'] == intervals_dict.get(forecast_types_trim[0]):
                        back_interval.append(intervals_dict.get(forecast_types_trim[0]))
                        back_forecast.append(row['Forecast' + intervals_dict.get(forecast_types_trim[0])])
                    elif row['SuggestedInterval'] == intervals_dict.get(forecast_types_trim[1]):
                        back_interval.append(intervals_dict.get(forecast_types_trim[0]))
                        back_forecast.append(row['Forecast' + intervals_dict.get(forecast_types_trim[0])])
                    elif row['SuggestedInterval'] == intervals_dict.get(forecast_types_trim[2]):
                        back_interval.append(intervals_dict.get(forecast_types_trim[1]))
                        back_forecast.append(row['Forecast' + intervals_dict.get(forecast_types_trim[1])])
                    elif row['SuggestedInterval'] == intervals_dict.get(forecast_types_trim[3]):
                        back_interval.append(intervals_dict.get(forecast_types_trim[2]))
                        back_forecast.append(row['Forecast' + intervals_dict.get(forecast_types_trim[2])])
                    elif row['SuggestedInterval'] == intervals_dict.get(forecast_types_trim[4]):
                        back_interval.append(intervals_dict.get(forecast_types_trim[3]))
                        back_forecast.append(row['Forecast' + intervals_dict.get(forecast_types_trim[3])])
                    else:
                        back_interval.append(intervals_dict.get(forecast_types_trim[2]))
                        back_forecast.append(row['Forecast' + intervals_dict.get(forecast_types_trim[2])])
            else:
                for ix, row in df_forecast.iterrows():
                    if row['SuggestedInterval'] == intervals_dict.get(self.forecast_types[0]):
                        back_interval.append(intervals_dict.get(self.forecast_types[0]))
                        back_forecast.append(row['Forecast' + intervals_dict.get(self.forecast_types[0])])
                    elif row['SuggestedInterval'] == intervals_dict.get(self.forecast_types[1]):
                        back_interval.append(intervals_dict.get(self.forecast_types[0]))
                        back_forecast.append(row['Forecast' + intervals_dict.get(self.forecast_types[0])])
                    elif row['SuggestedInterval'] == intervals_dict.get(self.forecast_types[2]):
                        back_interval.append(intervals_dict.get(self.forecast_types[1]))
                        back_forecast.append(row['Forecast' + intervals_dict.get(self.forecast_types[1])])
                    elif row['SuggestedInterval'] == intervals_dict.get(self.forecast_types[3]):
                        back_interval.append(intervals_dict.get(self.forecast_types[2]))
                        back_forecast.append(row['Forecast' + intervals_dict.get(self.forecast_types[2])])
                    elif row['SuggestedInterval'] == intervals_dict.get(self.forecast_types[4]):
                        back_interval.append(intervals_dict.get(self.forecast_types[3]))
                        back_forecast.append(row['Forecast' + intervals_dict.get(self.forecast_types[3])])
                    elif row['SuggestedInterval'] == intervals_dict.get(self.forecast_types[5]):
                        back_interval.append(intervals_dict.get(self.forecast_types[4]))
                        back_forecast.append(row['Forecast' + intervals_dict.get(self.forecast_types[4])])
                    elif row['SuggestedInterval'] == intervals_dict.get(self.forecast_types[6]):
                        back_interval.append(intervals_dict.get(self.forecast_types[5]))
                        back_forecast.append(row['Forecast' + intervals_dict.get(self.forecast_types[5])])
                    else:
                        back_interval.append(intervals_dict.get(self.forecast_types[3]))
                        back_forecast.append(row['Forecast' + intervals_dict.get(self.forecast_types[3])])
            df_forecast['BackSuggestedInterval'] = back_interval
            df_forecast['BackSuggestedForecast'] = back_forecast
        except KeyError as err:
            self.logger.exception(f'Invalid column name. Please check dataframe metadata: {err}')
            raise
        return df_forecast


    def compute_naive_forecast(self,df_forecast, num_periods=io.forecast_horizon, naive_type='same'):
        """
        Return a dataframe attaching the back suggested interval and forecast per date and item record

        :param df_forecast: Original dataframe including all items' forecasts
        :param num_periods: Number of periods to look back to naively forecast. Default config's forecast horizon
        :param naive_type: Naive forecast type. Choose from last observation (last) or rolling average average (roll)
        :return df_forecast: Output forecast dataframe including back suggested interval and forecast

        >>> df = compute_back_suggested_interval_forecast(df_forecast)
        >>> df =
                        var1    var2    var3
                idx0     1       2       3
        """
        try:
            df_forecast = df_forecast.sort_values(by=["Item", "Date"])
            target = df_forecast.Target.values
            n_values = int(len(target) / df_forecast.Item.nunique())
            rolling_list = []
            if naive_type == "last":
                for i in range(df_forecast.Item.nunique()):
                    rolling = pd.Series(target[num_periods * (self.backtests + 1) * i:num_periods * (self.backtests + 1) *
                                               (i + 1)]).shift(num_periods).fillna(0).values
                    rolling_list = np.append(rolling_list, rolling)
            elif naive_type =="same":
                for i in range(df_forecast.Item.nunique()):
                    target_item=target[n_values*i:n_values*(i+1)]
                    target_item=np.append(target_item[0], target_item)
                    target_item=target_item[:-1:num_periods]
                    rolling = np.repeat(target_item, num_periods)
                    rolling_list = np.append(rolling_list, rolling)
            elif naive_type=="mean":
                for i in range(df_forecast.Item.nunique()):
                    target_item=target[n_values*i:n_values*(i+1)]
                    target_item_means=[]
                    for back in range(self.backtests + 1):
                      if back==0:
                        mean_aux=target_item[0:1]
                      else:
                        mean_aux=target_item[0:back*num_periods].mean()
                      target_item_means=np.append(target_item_means, mean_aux)
                    rolling = np.repeat(target_item_means, num_periods)
                    rolling_list = np.append(rolling_list, np.round(rolling))              
            elif naive_type=="median":
                for i in range(df_forecast.Item.nunique()):
                    target_item=target[n_values*i:n_values*(i+1)]
                    target_item_means=[]
                    for back in range(self.backtests+1):
                      if back==0:
                        mean_aux=target_item[0:1]
                      else:
                        mean_aux=np.median(target_item[0:back*num_periods])
                      target_item_means=np.append(target_item_means, mean_aux)
                    rolling = np.repeat(target_item_means, num_periods)
                    rolling_list = np.append(rolling_list, np.round(rolling))  
            else:
                self.logger.debug(f'Invalid naive forecast option. Please check options.')
            df_forecast["ForecastNaive"] = rolling_list
        except KeyError as err:
            self.logger.exception(f'Invalid column name. Please check dataframe metadata: {err}')
            raise
        return df_forecast

    def concat_forecast_backtests(self, df_forecast, df_backtest, debug_cols=False):
        """
        Return a dataframe attaching forecast and backtests records

        :param df_forecast: Original dataframe including all items' forecasts
        :param df_backtest: Dataframe including all items' backtests
        :param debug_cols: True or false to include debug columns from backtest and forecast computation
        :return df_forecast: Output forecast dataframe attaching all backtests

        >>> df = concat_forecast_backtests(df_forecast, df_backtest)
        >>> df =
                        var1    var2    var3
                idx0     1       2       3
        """
        try:
            if debug_cols:
                df_forecast = pd.concat([df_forecast, df_backtest], axis='rows').fillna(0)
            else:
                forecast_cols = [col for col in df_forecast.columns if all(subcol not in col for subcol in ['SuggInterval'])]
                backtest_cols = [col for col in df_backtest.columns if all(subcol not in col for subcol in ['Bias'])]
                df_forecast = pd.concat([df_forecast[forecast_cols], df_backtest[backtest_cols]], axis='rows')
            df_forecast['Date'] = pd.to_datetime(df_forecast['Date'], format='%Y-%m-%d', utc=True)
            df_forecast = df_forecast.sort_values('Date', ascending=False)
        except KeyError as err:
            self.logger.exception(f'Invalid column name. Please check dataframe metadata: {err}')
            raise
        return df_forecast

    def concat_forecast_ranking(self, df_forecast, df_ranking):
        """
        Return a dataframe attaching forecast and item ranking

        :param df_forecast: Original dataframe including all items' forecasts
        :param df_ranking: Dataframe including items ranking
        :return df_forecast: Output forecast dataframe attaching item ranking

        >>> df = concat_forecast_ranking(df_forecast, df_ranking)
        >>> df =
                        var1    var2    var3
                idx0     1       2       3
        """
        try:
            rank_dict = {item: df_ranking.loc[df_ranking['Item'] == item, 'Ranking'].unique()[0] for item in df_ranking['Item']}
            df_forecast['Ranking'] = df_forecast['Item'].map(rank_dict)
        except KeyError as err:
            self.logger.exception(f'Invalid column name. Please check dataframe metadata: {err}')
            raise
        return df_forecast

    def make_forecast_strategy(self, df_forecast, item_col, use_location=False):
        """
        Return a dataframe with the strategy forecast analysis per item, including ranking, WMAPE, MASE and interval

        :param df_forecast: Dataframe with forecasts per dates and items
        :param item_col: Item column that identify each reference
        :param use_location: True or False whether the location dimension is included
        :return df_strategy: Dataframe with strategy forecast analysis

        >>> df = make_forecast_strategy(df_forecast, item_col='Item')
        >>> df =
                        var1    var2    var3
                idx0     1       2       3
        """
        if self.dataset_frequency == 'D':
            df_forecast = df_forecast.loc[df_forecast['Date'] >= df_forecast['Date'].max() - DateOffset(days=self.forecast_horizon)]
        elif self.dataset_frequency == 'W':
            df_forecast = df_forecast.loc[df_forecast['Date'] >= df_forecast['Date'].max() - DateOffset(weeks=self.forecast_horizon)]
        elif self.dataset_frequency == '3W':
            df_forecast = df_forecast.loc[df_forecast['Date'] >= df_forecast['Date'].max() - DateOffset(weeks=3*(self.forecast_horizon))]              
        elif self.dataset_frequency == 'M':
            df_forecast = df_forecast.loc[df_forecast['Date'] >= df_forecast['Date'].max() - DateOffset(months=self.forecast_horizon)]
        elif self.dataset_frequency == 'Y':
            df_forecast = df_forecast.loc[df_forecast['Date'] >= df_forecast['Date'].max() - DateOffset(years=self.forecast_horizon)]
        else:
            self.logger.debug(f'No valid dataset frequency. Please check config parameters in config.yml')
        try:
            if use_location:
                df_strategy = df_forecast.groupby([item_col], as_index=False).agg({
                    'Location': 'first',
                    'Ranking': 'first',
                    'WMAPE': 'mean',
                    'MASE': 'mean',
                    'SuggestedInterval': 'first'
                })
            else:
                df_strategy = df_forecast.groupby([item_col], as_index=False).agg({
                    'Ranking': 'first',
                    'WMAPE': 'mean',
                    'MASE': 'mean',
                    'SuggestedInterval': 'first'
                })
            df_strategy['WMAPE'] = df_strategy['WMAPE'].apply(lambda x: round(x, 2))
            df_strategy['MASE'] = df_strategy['MASE'].apply(lambda x: round(x, 2))
        except KeyError as err:
            self.logger.exception(f'Invalid column name. Please check dataframe metadata: {err}')
            raise
        return df_strategy


    def concat_forecast_horizons(self, model_type=None, forecast_horizons=None):
        """
        Return a dataframe contatenating short-, mid- and long-term forecasts.

        :param model_type: Deep learning model type. Either deepar or attup
        :param forecast_horizons: List of integers of short-, mid- and long-term forecast horizons. Frequency is autotaken from config file
        :return dataframe: Dataframe containing short-, mid- and long-term forecasts

        >>> df = concat_forecast_horizons(model_type='deepar', forecast_horizons=[1, 3, 6])
        >>> df =
                        var1    var2    var3
                idx0     1       2       3
        """

        try:
          if self.io.dataset_frequency == '3W': ##
            df_fcst_s = self.io.download_csv(q_name='Qfcst', datalake_path=os.path.join(self.multiforecast_path,
                                                                            model_type + str(forecast_horizons[0]) +'p-'+ self.dataset_frequency.lower(), 'output'), date_cols=['Date'])
            df_fcst_s.Item = df_fcst_s.Item.astype(str)
            if len(forecast_horizons) == 2:
                df_fcst_m = self.io.download_csv(q_name='Qfcst', datalake_path=os.path.join(self.multiforecast_path,
                                                                            model_type + str(forecast_horizons[1]) +'p-'+ self.dataset_frequency.lower(), 'output'), date_cols=['Date'])
                df_fcst_m.Item = df_fcst_m.Item.astype(str)
            elif len(forecast_horizons) == 3:
                df_fcst_m = self.io.download_csv(q_name='Qfcst', datalake_path=os.path.join(self.multiforecast_path,
                                                                            model_type + str(forecast_horizons[1]) +'p-'+ self.dataset_frequency.lower(), 'output'), date_cols=['Date'])
                df_fcst_m.Item = df_fcst_m.Item.astype(str)
                df_fcst_l = self.io.download_csv(q_name='Qfcst', datalake_path=os.path.join(self.multiforecast_path,
                                                                            model_type + str(forecast_horizons[2]) +'p-'+ self.dataset_frequency.lower(), 'output'), date_cols=['Date'])
                df_fcst_l.Item = df_fcst_l.Item.astype(str)
            else:
                self.logger.debug(f'No valid number of forecasts. It should minimum 1 and maximum 3.')

            if len(forecast_horizons) == 2:      
                df_bckt_s = df_fcst_s[df_fcst_s['Date'] <= df_fcst_s['Date'].max() - pd.DateOffset(weeks=3*forecast_horizons[0])]
                df_bckt_m = df_fcst_m[df_fcst_m['Date'] < df_bckt_s['Date'].min()]
                df_fcst_s = df_fcst_s[df_fcst_s['Date'] > df_fcst_s['Date'].max() - pd.DateOffset(weeks=3*forecast_horizons[0])]
                df_fcst_m = df_fcst_m[df_fcst_m['Date'] > df_fcst_m['Date'].max() - pd.DateOffset(weeks=3*forecast_horizons[1] - 3*forecast_horizons[0])]
            
                df_fcst = pd.concat([df_fcst_m, df_fcst_s, df_bckt_s, df_bckt_m], axis='rows') \
                            .sort_values(['Date', 'Item'], ascending=False)

            elif len(forecast_horizons) == 3:
                df_bckt_s = df_fcst_s[df_fcst_s['Date'] <= df_fcst_s['Date'].max() - pd.DateOffset(weeks=3*forecast_horizons[0])]
                df_bckt_m = df_fcst_m[df_fcst_m['Date'] < df_bckt_s['Date'].min()]
                df_bckt_l = df_fcst_l[df_fcst_l['Date'] < df_bckt_m['Date'].min()]
                df_fcst_s = df_fcst_s[df_fcst_s['Date'] > df_fcst_s['Date'].max() - pd.DateOffset(weeks=3*forecast_horizons[0])]
                df_fcst_m = df_fcst_m[df_fcst_m['Date'] > df_fcst_m['Date'].max() - pd.DateOffset(weeks=3*forecast_horizons[1] - 3*forecast_horizons[0])]
                df_fcst_l = df_fcst_l[df_fcst_l['Date'] > df_fcst_l['Date'].max() - pd.DateOffset(weeks=3*forecast_horizons[2] - 3*forecast_horizons[1])]

                df_fcst = pd.concat([df_fcst_l, df_fcst_m, df_fcst_s, df_bckt_s, df_bckt_m, df_bckt_l], axis='rows') \
                            .sort_values(['Date', 'Item'], ascending=False)
            else:
                self.logger.debug(f'No valid number of forecasts. It should minimum 1 and maximum 3.') ##
                          
          else:  
            df_fcst_s = self.io.download_csv(q_name='Qfcst',
                                             datalake_path=os.path.join(self.multiforecast_path,
                                                                        model_type + str(forecast_horizons[0]) + self.dataset_frequency.lower(),
                                                                        'output'),
                                             date_cols=['Date'])
            df_fcst_s.Item = df_fcst_s.Item.astype(str)
            if len(forecast_horizons) == 2:
                df_fcst_m = self.io.download_csv(q_name='Qfcst',
                                                 datalake_path=os.path.join(self.multiforecast_path,
                                                                            model_type + str(forecast_horizons[1]) + self.dataset_frequency.lower(),
                                                                            'output'),
                                                 date_cols=['Date'])
                df_fcst_m.Item = df_fcst_m.Item.astype(str)
            elif len(forecast_horizons) == 3:
                df_fcst_m = self.io.download_csv(q_name='Qfcst',
                                                 datalake_path=os.path.join(self.multiforecast_path,
                                                                            model_type + str(forecast_horizons[1]) + self.dataset_frequency.lower(),
                                                                            'output'),
                                                 date_cols=['Date'])
                df_fcst_m.Item = df_fcst_m.Item.astype(str)
                df_fcst_l = self.io.download_csv(q_name='Qfcst',
                                                 datalake_path=os.path.join(self.multiforecast_path,
                                                                            model_type + str(forecast_horizons[2]) + self.dataset_frequency.lower(),
                                                                            'output'),
                                                 date_cols=['Date'])
                df_fcst_l.Item = df_fcst_l.Item.astype(str)
            else:
                self.logger.debug(f'No valid number of forecasts. It should minimum 1 and maximum 3.')

            if len(forecast_horizons) == 2:
                if self.io.dataset_frequency == 'M':
                    df_bckt_s = df_fcst_s[df_fcst_s['Date'] <= df_fcst_s['Date'].max() - pd.DateOffset(months=forecast_horizons[0])]
                    df_bckt_m = df_fcst_m[df_fcst_m['Date'] < df_bckt_s['Date'].min()]
                    df_fcst_s = df_fcst_s[df_fcst_s['Date'] > df_fcst_s['Date'].max() - pd.DateOffset(months=forecast_horizons[0])]
                    df_fcst_m = df_fcst_m[df_fcst_m['Date'] > df_fcst_m['Date'].max() -
                                          pd.DateOffset(months=forecast_horizons[1] - forecast_horizons[0])]
                elif self.io.dataset_frequency == 'W':
                    df_bckt_s = df_fcst_s[df_fcst_s['Date'] <= df_fcst_s['Date'].max() - pd.DateOffset(weeks=forecast_horizons[0])]
                    df_bckt_m = df_fcst_m[df_fcst_m['Date'] < df_bckt_s['Date'].min()]
                    df_fcst_s = df_fcst_s[df_fcst_s['Date'] > df_fcst_s['Date'].max() - pd.DateOffset(weeks=forecast_horizons[0])]
                    df_fcst_m = df_fcst_m[df_fcst_m['Date'] > df_fcst_m['Date'].max() -
                                          pd.DateOffset(weeks=forecast_horizons[1] - forecast_horizons[0])]
                elif self.io.dataset_frequency == 'D':
                    df_bckt_s = df_fcst_s[df_fcst_s['Date'] <= df_fcst_s['Date'].max() - pd.DateOffset(days=forecast_horizons[0])]
                    df_bckt_m = df_fcst_m[df_fcst_m['Date'] < df_bckt_s['Date'].min()]
                    df_fcst_s = df_fcst_s[df_fcst_s['Date'] > df_fcst_s['Date'].max() - pd.DateOffset(days=forecast_horizons[0])]
                    df_fcst_m = df_fcst_m[df_fcst_m['Date'] > df_fcst_m['Date'].max() -
                                          pd.DateOffset(days=forecast_horizons[1] - forecast_horizons[0])]
                else:
                    self.logger.debug(f'No valid dataset frequency. Please check dataset_frequency in config.yml file.')

                df_fcst = pd.concat([df_fcst_m, df_fcst_s, df_bckt_s, df_bckt_m], axis='rows') \
                            .sort_values(['Date', 'Item'], ascending=False)

            elif len(forecast_horizons) == 3:
                if self.io.dataset_frequency == 'M':
                    df_bckt_s = df_fcst_s[df_fcst_s['Date'] <= df_fcst_s['Date'].max() - pd.DateOffset(months=forecast_horizons[0])]
                    df_bckt_m = df_fcst_m[df_fcst_m['Date'] < df_bckt_s['Date'].min()]
                    df_bckt_l = df_fcst_l[df_fcst_l['Date'] < df_bckt_m['Date'].min()]
                    df_fcst_s = df_fcst_s[df_fcst_s['Date'] > df_fcst_s['Date'].max() - pd.DateOffset(months=forecast_horizons[0])]
                    df_fcst_m = df_fcst_m[df_fcst_m['Date'] > df_fcst_m['Date'].max() -
                                          pd.DateOffset(months=forecast_horizons[1] - forecast_horizons[0])]
                    df_fcst_l = df_fcst_l[df_fcst_l['Date'] > df_fcst_l['Date'].max() -
                                          pd.DateOffset(months=forecast_horizons[2] - forecast_horizons[1])]
                elif self.io.dataset_frequency == 'W':
                    df_bckt_s = df_fcst_s[df_fcst_s['Date'] <= df_fcst_s['Date'].max() - pd.DateOffset(weeks=forecast_horizons[0])]
                    df_bckt_m = df_fcst_m[df_fcst_m['Date'] < df_bckt_s['Date'].min()]
                    df_bckt_l = df_fcst_l[df_fcst_l['Date'] < df_bckt_m['Date'].min()]
                    df_fcst_s = df_fcst_s[df_fcst_s['Date'] > df_fcst_s['Date'].max() - pd.DateOffset(weeks=forecast_horizons[0])]
                    df_fcst_m = df_fcst_m[df_fcst_m['Date'] > df_fcst_m['Date'].max() -
                                          pd.DateOffset(weeks=forecast_horizons[1] - forecast_horizons[0])]
                    df_fcst_l = df_fcst_l[df_fcst_l['Date'] > df_fcst_l['Date'].max() -
                                          pd.DateOffset(weeks=forecast_horizons[2] - forecast_horizons[1])]
                elif self.io.dataset_frequency == 'D':
                    df_bckt_s = df_fcst_s[df_fcst_s['Date'] <= df_fcst_s['Date'].max() - pd.DateOffset(days=forecast_horizons[0])]
                    df_bckt_m = df_fcst_m[df_fcst_m['Date'] < df_bckt_s['Date'].min()]
                    df_bckt_l = df_fcst_l[df_fcst_l['Date'] < df_bckt_m['Date'].min()]
                    df_fcst_s = df_fcst_s[df_fcst_s['Date'] > df_fcst_s['Date'].max() - pd.DateOffset(days=forecast_horizons[0])]
                    df_fcst_m = df_fcst_m[df_fcst_m['Date'] > df_fcst_m['Date'].max() -
                                          pd.DateOffset(days=forecast_horizons[1] - forecast_horizons[0])]
                    df_fcst_l = df_fcst_l[df_fcst_l['Date'] > df_fcst_l['Date'].max() -
                                          pd.DateOffset(days=forecast_horizons[2] - forecast_horizons[1])]
                else:
                    self.logger.debug(f'No valid dataset frequency. Please check dataset_frequency in config.yml file.')

                df_fcst = pd.concat([df_fcst_l, df_fcst_m, df_fcst_s, df_bckt_s, df_bckt_m, df_bckt_l], axis='rows') \
                            .sort_values(['Date', 'Item'], ascending=False)
            else:
                self.logger.debug(f'No valid number of forecasts. It should minimum 1 and maximum 3.')

        except KeyError as err:
            self.logger.exception(f'Invalid column name. Please check dataframe metadata: {err}')
            raise
        return df_fcst

    def concat_forecast_horizons_monthly(self, model_type=None, forecast_horizons=None):
        """
        Return a dataframe contatenating short-, mid- and long-term monthly forecasts.

        :param model_type: Deep learning model type. Either deepar or attup
        :param forecast_horizons: List of integers of short-, mid- and long-term forecast horizons. Frequency is autotaken from config file
        :return dataframe: Dataframe containing short-, mid- and long-term forecasts

        >>> df = concat_forecast_horizons_monthly(model_type='deepar', forecast_horizons=[1, 3, 6])
        >>> df =
                        var1    var2    var3
                idx0     1       2       3
        """

        try:
            df_fcst_s = self.io.download_csv(q_name='Qfcst-upsample',
                                             datalake_path=os.path.join(self.multiforecast_path,
                                                                        model_type + str(forecast_horizons[0]) + self.dataset_frequency.lower(),
                                                                        'output'),
                                             date_cols=['Date'])
            df_fcst_s.Item = df_fcst_s.Item.astype(str)
            if len(forecast_horizons) == 2:
                df_fcst_m = self.io.download_csv(q_name='Qfcst-upsample',
                                                 datalake_path=os.path.join(self.multiforecast_path,
                                                                            model_type + str(forecast_horizons[1]) + self.dataset_frequency.lower(),
                                                                            'output'),
                                                 date_cols=['Date'])
                df_fcst_m.Item = df_fcst_m.Item.astype(str)
            elif len(forecast_horizons) == 3:
                df_fcst_m = self.io.download_csv(q_name='Qfcst-upsample',
                                                 datalake_path=os.path.join(self.multiforecast_path,
                                                                            model_type + str(forecast_horizons[1]) + self.dataset_frequency.lower(),
                                                                            'output'),
                                                 date_cols=['Date'])
                df_fcst_m.Item = df_fcst_m.Item.astype(str)
                df_fcst_l = self.io.download_csv(q_name='Qfcst-upsample',
                                                 datalake_path=os.path.join(self.multiforecast_path,
                                                                            model_type + str(forecast_horizons[2]) + self.dataset_frequency.lower(),
                                                                            'output'),
                                                 date_cols=['Date'])
                df_fcst_l.Item = df_fcst_l.Item.astype(str)
            else:
                self.logger.debug(f'No valid number of forecasts. It should minimum 1 and maximum 3.')

            if len(forecast_horizons) == 2:
                if self.io.dataset_frequency == 'M':
                    df_bckt_s = df_fcst_s[df_fcst_s['Date'] <= df_fcst_s['Date'].max() - pd.DateOffset(months=forecast_horizons[0])]
                    df_bckt_m = df_fcst_m[df_fcst_m['Date'] < df_bckt_s['Date'].min()]
                    df_fcst_s = df_fcst_s[df_fcst_s['Date'] > df_fcst_s['Date'].max() - pd.DateOffset(months=forecast_horizons[0])]
                    df_fcst_m = df_fcst_m[df_fcst_m['Date'] > df_fcst_m['Date'].max() -
                                          pd.DateOffset(months=forecast_horizons[1] - forecast_horizons[0])]
                elif self.io.dataset_frequency == 'W':
                    df_bckt_s = df_fcst_s[df_fcst_s['Date'] <= df_fcst_s['Date'].max() - pd.DateOffset(months=forecast_horizons[0] / 4)]
                    df_bckt_m = df_fcst_m[df_fcst_m['Date'] < df_bckt_s['Date'].min()]
                    df_fcst_s = df_fcst_s[df_fcst_s['Date'] > df_fcst_s['Date'].max() - pd.DateOffset(months=forecast_horizons[0] / 4)]
                    df_fcst_m = df_fcst_m[df_fcst_m['Date'] > df_fcst_m['Date'].max() -
                                          pd.DateOffset(months=forecast_horizons[1] / 4 - forecast_horizons[0] / 4)]
                elif self.io.dataset_frequency == 'D':
                    df_bckt_s = df_fcst_s[df_fcst_s['Date'] <= df_fcst_s['Date'].max() - pd.DateOffset(months=forecast_horizons[0] / 30)]
                    df_bckt_m = df_fcst_m[df_fcst_m['Date'] < df_bckt_s['Date'].min()]
                    df_fcst_s = df_fcst_s[df_fcst_s['Date'] > df_fcst_s['Date'].max() - pd.DateOffset(months=forecast_horizons[0] / 30)]
                    df_fcst_m = df_fcst_m[df_fcst_m['Date'] > df_fcst_m['Date'].max() -
                                          pd.DateOffset(months=forecast_horizons[1] / 30 - forecast_horizons[0] / 30)]
                else:
                    self.logger.debug(f'No valid dataset frequency. Please check dataset_frequency in config.yml file.')

                df_fcst = pd.concat([df_fcst_m, df_fcst_s, df_bckt_s, df_bckt_m], axis='rows') \
                            .sort_values(['Date', 'Item'], ascending=False)

            elif len(forecast_horizons) == 3:
                if self.io.dataset_frequency == 'M':
                    df_bckt_s = df_fcst_s[df_fcst_s['Date'] <= df_fcst_s['Date'].max() - pd.DateOffset(months=forecast_horizons[0])]
                    df_bckt_m = df_fcst_m[df_fcst_m['Date'] < df_bckt_s['Date'].min()]
                    df_bckt_l = df_fcst_l[df_fcst_l['Date'] < df_bckt_m['Date'].min()]
                    df_fcst_s = df_fcst_s[df_fcst_s['Date'] > df_fcst_s['Date'].max() - pd.DateOffset(months=forecast_horizons[0])]
                    df_fcst_m = df_fcst_m[df_fcst_m['Date'] > df_fcst_m['Date'].max() -
                                          pd.DateOffset(months=forecast_horizons[1] - forecast_horizons[0])]
                    df_fcst_l = df_fcst_l[df_fcst_l['Date'] > df_fcst_l['Date'].max() -
                                          pd.DateOffset(months=forecast_horizons[2] - forecast_horizons[1])]
                elif self.io.dataset_frequency == 'W':
                    df_bckt_s = df_fcst_s[df_fcst_s['Date'] <= df_fcst_s['Date'].max() - pd.DateOffset(months=forecast_horizons[0] / 4)]
                    df_bckt_m = df_fcst_m[df_fcst_m['Date'] < df_bckt_s['Date'].min()]
                    df_bckt_l = df_fcst_l[df_fcst_l['Date'] < df_bckt_m['Date'].min()]
                    df_fcst_s = df_fcst_s[df_fcst_s['Date'] > df_fcst_s['Date'].max() - pd.DateOffset(months=forecast_horizons[0] / 4)]
                    df_fcst_m = df_fcst_m[df_fcst_m['Date'] > df_fcst_m['Date'].max() -
                                          pd.DateOffset(months=forecast_horizons[1] / 4 - forecast_horizons[0] / 4)]
                    df_fcst_l = df_fcst_l[df_fcst_l['Date'] > df_fcst_l['Date'].max() -
                                          pd.DateOffset(months=forecast_horizons[2] / 4 - forecast_horizons[1] / 4)]
                elif self.io.dataset_frequency == 'D':
                    df_bckt_s = df_fcst_s[df_fcst_s['Date'] <= df_fcst_s['Date'].max() - pd.DateOffset(months=forecast_horizons[0] / 30)]
                    df_bckt_m = df_fcst_m[df_fcst_m['Date'] < df_bckt_s['Date'].min()]
                    df_bckt_l = df_fcst_l[df_fcst_l['Date'] < df_bckt_m['Date'].min()]
                    df_fcst_s = df_fcst_s[df_fcst_s['Date'] > df_fcst_s['Date'].max() - pd.DateOffset(months=forecast_horizons[0] / 30)]
                    df_fcst_m = df_fcst_m[df_fcst_m['Date'] > df_fcst_m['Date'].max() -
                                          pd.DateOffset(months=forecast_horizons[1] / 30 - forecast_horizons[0] / 30)]
                    df_fcst_l = df_fcst_l[df_fcst_l['Date'] > df_fcst_l['Date'].max() -
                                          pd.DateOffset(months=forecast_horizons[2] / 30 - forecast_horizons[1] / 30)]
                else:
                    self.logger.debug(f'No valid dataset frequency. Please check dataset_frequency in config.yml file.')
                df_fcst = pd.concat([df_fcst_l, df_fcst_m, df_fcst_s, df_bckt_s, df_bckt_m, df_bckt_l], axis='rows') \
                            .sort_values(['Date', 'Item'], ascending=False)
            else:
                self.logger.debug(f'No valid number of forecasts. It should minimum 1 and maximum 3.')

        except KeyError as err:
            self.logger.exception(f'Invalid column name. Please check dataframe metadata: {err}')
            raise
        return df_fcst
