class SteamTraderError(Exception):
    """Базовый класс, представляющий исключения общего характера."""

class UnsupportedAppID(SteamTraderError):
    """Класс исключения, вызываемый в случае использования неподдерживаемого AppID."""

class ClientError(SteamTraderError):
    """Класс исключения, вызываемый в случае ошибки с клиентом."""

class Unauthorized(ClientError):
    """Класс исключения, вызываемый в случае, если клиент не зарегистрирован."""

class AuthenticatorError(ClientError):
    """Класс исключения, вызываемый в случае, если мобильный аутентификатор не поключён, или не прошло 7 ней с момента его активации."""

class TradeError(SteamTraderError):
    """Базовый класс исключений, вызываемых для ошибок, связанных с обменом."""

class TradeCreationFail(TradeError):
    """Класс исключения, вызываемый в случае, если не удалось создать предложение обмена."""

class NoTradeLink(TradeError):
    """Класс исключения, вызываемый в случае, если нет ссылки на обмен."""

class NoSteamAPIKey(TradeError):
    """Класс исключения, вызываемый в случае, если нет ключа Steam API."""

class WrongTradeLink(TradeError):
    """Класс исключения, вызываемый в случае, если клиент указал ссылку для обмена от другого Steam аккаунта."""

class ExpiredTradeLink(TradeError):
    """Класс исключения, вызываемый в случае, если ссылка для обмена больше недействительна."""

class NoBuyOrders(TradeError):
    """Класс исключения, вызываемый в случае, если у клиента нет запросов на покупку."""

class TradeBlockError(TradeError):
    """Класс исключения, вызываемый в случае, если не включён Steam Guard или стоит блокировка обменов."""

class MissingRequiredItems(TradeError):
    """Класс исключения, вызываемый в случае, если в инвентаре Steam отсутствуют необходимые для передачи предметы."""

class HiddenInventory(TradeError):
    """Класс исключения, вызываемый в случае, если инвентарь скрыт."""

class NoTradeItems(TradeError):
    """Класс исключения, вызываемый в случае, если у клиента нет предметов для обмена."""

class IncorrectPrice(TradeError):
    """Класс исключения, вызываемый в случае, если выставленная цена ниже минимальной или больше максимальной."""

class ItemAlreadySold(TradeError):
    """Класс исключения, вызываемый в случае, если предмет уже выставлен на продажу."""

class NoLongerExists(TradeError):
    """Класс исключения, вызываемый в случае, если предмет больше не существует."""

class NotEnoughMoney(TradeError):
    """Класс исключения, вызываемый в случае, если на балансе недостаточно средств для совершения операции."""

class NetworkError(SteamTraderError):
    """Базовый класс исключений, вызываемых для ошибок, связанных с запросами к серверу."""

class OperationFail(SteamTraderError):
    """Класс представляющий исключения, вызываемый в случае, если запрос был правильным, но операция не прошла успешно."""

class UnknownItem(OperationFail):
    """Класс исключения, вызываемый в случае, если предмет не был найден."""

class SaveFail(OperationFail):
    """Класс исключения, вызываемый в случае, если не удалось сохранить изменённый праметр на сайте."""

class InternalError(OperationFail):
    """Класс исключения, вызываемый в случае, если при выполнении запроса произошла неизвестая ошибка."""

class BadRequestError(NetworkError):
    """Класс исключения, вызываемый в случае отправки неправильного запроса."""

class TooManyRequests(NetworkError):
    """Класс исключения, вызываемый в случае отправки чрезмерно большого количества запросов на сервер."""

class NotFoundError(NetworkError):
    """Класс исключения, вызываемый в случае ответа от сервера со статус кодом 404."""

class TimedOutError(NetworkError):
    """Класс исключения, вызываемого для случаев истечения времени ожидания."""
