

# Nexios and FastAPI

### Asynchronous Programming Model

Python has historically been synchronous, where each function call blocks until completion. This worked fine for many applications but became a bottleneck for modern web services handling thousands of concurrent connections. To address this, **asyncio** was introduced into the Python standard library, providing an event loop and asynchronous primitives.

However, early async code was cumbersome, often requiring callbacks or generator-based coroutines with `yield from`. This led to complexities similar to JavaScript’s “callback hell.” The solution came with the **async/await syntax**, introduced in Python 3.5, making asynchronous programming clean and intuitive.

For example:

```python
async def get_user():
    return {"id": 1, "name": "Dunamis"}

async def fetch_posts(user_id: int):
    return [{"title": "Nexios vs FastAPI"}, {"title": "Async Python"}]

async def main():
    user = await get_user()
    posts = await fetch_posts(user["id"])
    return {"user": user, "posts": posts}
```

The `await` keyword allows writing asynchronous code that *looks* synchronous but runs efficiently without blocking the event loop.

---

## FastAPI

**FastAPI** is one of the most popular Python frameworks for building modern web APIs. It is designed around **ASGI (Asynchronous Server Gateway Interface)**, enabling high-performance async request handling. Its key strengths include:

* **Automatic validation and documentation** via Pydantic.
* **Dependency injection system** for modular applications.
* **Async-first design**, allowing efficient concurrency.

FastAPI is widely used in production due to its developer experience and rich ecosystem.

---

## Nexios

**Nexios** is a younger, lightweight ASGI framework inspired by Express.js but tailored for Python’s async world. Unlike FastAPI, which emphasizes schema validation and documentation, Nexios focuses on **speed, simplicity, and extensibility**. It aims to be the minimal but powerful foundation for APIs, with explicit coding style and no hidden “magic.”

Key differences from FastAPI include:

* **Minimalistic core** — you bring your own ORM, validation, or tools.
* **Explicit responses** — developers directly control `Request` and `Response` objects.
* **High performance** — optimized for async with a small footprint.
* **Extensible backends** — authentication and middleware can be swapped or customized easily.

---

## Middleware

FastAPI middleware runs in a simple stack: each middleware processes a request once on the way in and once on the way out.

Nexios adopts a similar layered approach but with **greater emphasis on control**. Middleware wraps around requests in an “onion-like” pattern:

* Pre-processing before passing control to the next handler.
* Post-processing when control returns.

Example in Nexios:

```python
from nexios import NexiosApp

app = NexiosApp()

async def log_requests(request, response, call_next):
    print(f"Incoming: {request.path}")
    response = await call_next()
    print(f"Completed: {request.path}")
    return response

app.add_middleware(log_requests)
```

---

## Context

* In **FastAPI**, request data is usually accessed via dependency injection and function parameters. For example, query parameters or body models are injected directly into endpoint functions.

* In **Nexios**, a `Context`-like pattern emerges via the `Request` and `Response` objects. All state, headers, and attributes related to a request lifecycle are available there. Developers can also extend these objects to add their own utilities — similar to how Egg extended Koa’s `Context`.

---

## Exception Handling

Both frameworks allow centralized exception handling.

* **FastAPI** uses decorators like `@app.exception_handler(Exception)` to register custom handlers.
* **Nexios** allows writing middleware that catches and handles errors in a simple async flow:

```python
@app.add_exception_handler(Exception)
async def catch_exceptions(request, response ,exception):
    return Response.json({"error": str(exception)}, status=500)
```

This makes it easy to define global or fine-grained error policies.

---

## Extensions and Plugins

FastAPI achieves extensibility through:

* Dependencies (for injecting reusable logic).
* Event hooks (`startup`, `shutdown`).
* Third-party packages (authentication, databases, background tasks).

Nexios, being minimal, introduces **backends and extensions**:

* Developers can write authentication backends (JWT, API key, custom).
* Extend `Request`/`Response` objects with project-specific utilities.
* Build plugins for features like sessions, rate limiting, or analytics.

This plugin system is more **explicit and lightweight** than FastAPI’s dependency injection model.

---

## Roadmap

* **FastAPI**: Already mature, used in large-scale companies, with ongoing improvements around Pydantic v2 integration and background task optimizations.

* **Nexios**: Still evolving, but its vision is clear — a framework that is:

  * Async-native from the ground up.
  * Minimal and fast (developer chooses the stack).
  * Extensible through explicit code, not implicit magic.
  * Positioned as a “modern Express.js for Python.”

---

🔑 **In summary**:

* FastAPI = feature-rich, batteries-included, validation-first.
* Nexios = lean, fast, explicit, extensible.

Both embrace Python’s async/await world, but their philosophies diverge: FastAPI optimizes for developer convenience with validation and docs, while Nexios optimizes for speed, simplicity, and extensibility.

