"""
Base persona class for all skill personas.

Following Snarky Senior Engineer's advice: Simple, clear interface.
Following Pragmatic Architect's advice: Abstract base class for evolution.
"""

from abc import ABC, abstractmethod
from typing import List, Dict, Optional


class BasePersona(ABC):
    """
    Base class for all skill personas.

    Each persona represents a specialized engineering role (e.g., Security Sentinel,
    Pragmatic Architect) with specific expertise, principles, and perspective.
    """

    def __init__(self, skill_data: Dict):
        """
        Initialize persona from parsed skill data.

        Args:
            skill_data: Dict containing parsed SKILL.md content
        """
        self._skill_data = skill_data
        self._metadata = skill_data.get('metadata', {})
        self._principles = skill_data.get('principles', [])
        self._personality = skill_data.get('personality', '')
        self._expertise = skill_data.get('expertise', [])
        self._full_content = skill_data.get('full_content', '')
        # v0.4.0: Enhanced metadata
        self._examples = skill_data.get('examples', [])
        self._use_when = skill_data.get('use_when', '')
        self._related = skill_data.get('related', [])
        self._quick_tip = skill_data.get('quick_tip', '')

    @property
    def name(self) -> str:
        """Unique persona identifier (e.g., 'snarky_senior_engineer')"""
        return self._metadata.get('name', 'unknown')

    @property
    def description(self) -> str:
        """One-line persona description"""
        return self._metadata.get('description', '')

    @property
    def expertise_areas(self) -> List[str]:
        """Keywords/topics this persona should be consulted for"""
        return self._expertise

    @property
    def core_principles(self) -> List[str]:
        """10 core principles from Section 0 of the skill"""
        return self._principles

    @property
    def personality(self) -> str:
        """Personality and tone description from Section 1"""
        return self._personality

    @property
    def full_content(self) -> str:
        """Full SKILL.md content for detailed analysis"""
        return self._full_content

    # v0.4.0: Enhanced metadata properties
    @property
    def examples(self) -> List[str]:
        """Example queries where this persona would be helpful"""
        return self._examples

    @property
    def use_when(self) -> str:
        """Description of when to consult this persona"""
        return self._use_when

    @property
    def related_personas(self) -> List[str]:
        """Names of related/complementary personas"""
        return self._related

    @property
    def quick_tip(self) -> str:
        """One-line expertise summary for quick reference"""
        return self._quick_tip or self.description

    def analyze(self, query: str, context: Optional[Dict] = None) -> str:
        """
        Provide this persona's perspective on the query.

        Args:
            query: The question or scenario to analyze
            context: Optional session context (constraints, decisions, patterns)

        Returns:
            The persona's perspective as a string
        """
        # Default implementation: return persona's viewpoint based on principles
        # Subclasses can override for more sophisticated analysis

        context_str = ""
        if context:
            if context.get('active_constraints'):
                context_str += f"\nActive Constraints: {', '.join(context['active_constraints'])}"
            if context.get('patterns_agreed'):
                context_str += f"\nAgreed Patterns: {', '.join(context['patterns_agreed'])}"

        return f"""
**{self.name.replace('_', ' ').title()}** perspective:

Query: {query}
{context_str}

Based on my core principles:
{chr(10).join(f"- {p}" for p in self.core_principles[:3])}

My analysis: [This would be generated by the orchestrator using the full skill content]
"""

    def relevance_score(self, query: str) -> float:
        """
        Calculate 0-1 relevance score for this query.

        Uses keyword matching against expertise areas.

        Args:
            query: The query to score

        Returns:
            Float between 0 and 1 indicating relevance
        """
        if not self.expertise_areas:
            return 0.0

        query_lower = query.lower()
        matches = sum(1 for keyword in self.expertise_areas if keyword.lower() in query_lower)

        # Normalize by expertise areas count, cap at 1.0
        return min(matches / max(len(self.expertise_areas), 1), 1.0)

    def __repr__(self) -> str:
        return f"<{self.__class__.__name__}: {self.name}>"
