# Prompt-Set A/B/N Testing

## Overview

A minimal prompt experimentation mechanism that allows testing different prompt variants without code changes. The active prompt set is controlled via configuration, environment variable, or CLI flag.

## Quick Start

### 1. Directory Structure

```
prompts/
  summarize_day.md          # Default prompt
  extract_facts.md
  profile_update.md
  experiments/
    variant-a/
      summarize_day.md      # Override for variant-a
      extract_facts.md
    variant-b/
      summarize_day.md      # Override for variant-b
```

### 2. Configuration

**config.yaml:**
```yaml
prompts:
  active_set: variant-a    # Use variant-a prompts
```

**Environment variable:**
```bash
export AIJOURNAL_PROMPT_SET=variant-b
```

**CLI flag:**
```bash
aijournal summarize --date 2025-01-15 --prompt-set variant-a
```

### 3. Precedence

1. CLI flag (`--prompt-set`)
2. Environment variable (`AIJOURNAL_PROMPT_SET`)
3. Config file (`prompts.active_set`)
4. Default (no experiment, uses `prompts/*.md`)

## Implementation Details

### Prompt Resolution

The `resolve_prompt_path()` function checks for experiment overrides:

```python
from aijournal.utils.paths import resolve_prompt_path
from aijournal.common.config_loader import resolve_prompt_set

# Resolve active prompt set
active_set = resolve_prompt_set(cli_override=None, config=config)

# Resolve prompt path with experiment override
path = resolve_prompt_path("prompts/summarize_day.md", prompt_set=active_set)
# Returns: prompts/experiments/variant-a/summarize_day.md (if exists)
# Otherwise: prompts/summarize_day.md (default)
```

### Metadata Tracking

Prompt metadata is automatically propagated through the system:

**LLMResult:**
```python
# result.prompt_kind == "summarize_day"
# result.prompt_set == "variant-a"
```

**Artifact Metadata:**
```yaml
# derived/summaries/2025-01-15.yaml
kind: summaries.daily
meta:
  created_at: "2025-01-15T10:00:00Z"
  model: "gpt-oss:20b"
  prompt_path: "prompts/summarize_day.md"
  prompt_hash: "abc123..."
  prompt_kind: "summarize_day"             # NEW
  prompt_set: "variant-a"                  # NEW
data:
  day: "2025-01-15"
  bullets: [...]
```

**Structured Logs:**
```json
{
  "timestamp": "2025-01-15T10:00:05Z",
  "event": "llm_call",
  "prompt_path": "prompts/summarize_day.md",
  "prompt_kind": "summarize_day",
  "prompt_set": "variant-a",
  "model": "gpt-oss:20b",
  "duration_ms": 1234
}
```

## Analysis Workflow

### 1. Extract Results by Prompt Set

```bash
# Find all artifacts created with variant-a
find derived -name "*.yaml" -exec yq '.meta.prompt_set' {} + | grep variant-a

# Extract structured logs for a specific prompt set
jq 'select(.prompt_set == "variant-a")' derived/logs/structured_metrics.jsonl
```

### 2. Compare Performance

```python
import pandas as pd
import json

# Load structured metrics
with open("derived/logs/structured_metrics.jsonl") as f:
    logs = [json.loads(line) for line in f]

df = pd.DataFrame(logs)

# Compare by prompt kind and set
summary = df.groupby(["prompt_kind", "prompt_set"]).agg({
    "attempts": "mean",
    "repair_attempts": "mean",
    "coercion_count": "sum"
})
print(summary)
```

### 3. Quality Assessment

Manually review artifacts generated by different prompt sets:

```bash
# Compare summaries from control vs variant
diff <(yq '.data' derived/summaries/2025-01-15.yaml) \
     <(yq '.data' derived/summaries/2025-01-16.yaml)
```

## Testing

### Unit Tests

```python
# tests/test_prompt_sets.py
def test_resolve_prompt_set_precedence():
    """CLI > env > config precedence."""
    config = AppConfig(prompts=PromptsConfig(active_set="config-set"))
    with mock.patch.dict(os.environ, {"AIJOURNAL_PROMPT_SET": "env-set"}):
        result = resolve_prompt_set(cli_override="cli-set", config=config)
        assert result == "cli-set"

def test_resolve_prompt_path_experiment_override(tmp_path):
    """Should find experiment override when it exists."""
    # Create experiment override
    override = tmp_path / "prompts/experiments/test-set/summarize_day.md"
    override.parent.mkdir(parents=True)
    override.write_text("Experiment prompt")

    # Should resolve to override
    result = resolve_prompt_path("prompts/summarize_day.md", prompt_set="test-set")
    assert result == override
```

### Integration Tests

```python
# tests/test_prompt_metadata.py
def test_metadata_propagation():
    """Prompt metadata flows through LLMResult and artifacts."""
    result = run_ollama_agent(
        config,
        prompt="Test prompt",
        prompt_kind="summarize_day",
        prompt_set="variant-a",
    )
    assert result.prompt_kind == "summarize_day"
    assert result.prompt_set == "variant-a"
```

## Migration Guide

### Adding a New Experiment

1. **Create experiment directory:**
   ```bash
   mkdir -p prompts/experiments/my-experiment
   ```

2. **Copy and modify prompts:**
   ```bash
   cp prompts/summarize_day.md prompts/experiments/my-experiment/
   # Edit the copy to test your changes
   ```

3. **Activate experiment:**
   ```yaml
   # config.yaml
   prompts:
     active_set: my-experiment
   ```

4. **Run commands normally:**
   ```bash
   uv run aijournal summarize --date 2025-01-15
   ```

5. **Check metadata:**
   ```bash
   yq '.meta.prompt_set' derived/summaries/2025-01-15.yaml
   ```

### Reverting to Default

```yaml
# config.yaml
prompts:
  active_set: null  # or remove the line
```

Or:
```bash
unset AIJOURNAL_PROMPT_SET
```

## Design Principles

1. **Minimal Code Changes**: Extends existing infrastructure, no new abstractions
2. **Backward Compatible**: Old artifacts without prompt_kind/prompt_set still work
3. **Config-Driven**: No code changes needed to run experiments
4. **Transparent Metadata**: Experiment context automatically recorded everywhere
5. **Analysis-Ready**: Standard tools (jq, yq, pandas) handle all analysis

## Schema Reference

### PromptsConfig

```python
class PromptsConfig(BaseModel):
    """Prompts configuration for A/B/N testing."""
    active_set: str | None = None
```

### ArtifactMeta

```python
class ArtifactMeta(StrictModel):
    created_at: TimestampStr
    model: str | None = None
    prompt_path: str | None = None
    prompt_hash: str | None = None
    prompt_kind: str | None = None      # NEW: e.g., "summarize_day"
    prompt_set: str | None = None       # NEW: e.g., "variant-a"
    # ... other fields ...
```

### LLMResult

```python
class LLMResult(StrictModel, Generic[T]):
    model: str
    prompt_path: str
    prompt_hash: str | None = None
    prompt_kind: str | None = None      # NEW
    prompt_set: str | None = None       # NEW
    created_at: TimestampStr
    payload: T
    # ... other fields ...
```

## Future Enhancements

- Automatic performance metrics comparison
- Experiment results dashboard
- Multi-armed bandit selection strategies
