def flag_criteria(flags):

 
    """
Flag Criteria for Filtering Elements:
=====================================

In data processing and analysis, flag criteria are essential for selecting and managing elements based on specific conditions or quality assessments. These criteria allow users to filter data effectively, focusing on elements that meet predefined standards or have undergone particular processing stages.

Possible flag Criteria:
=======================

- **--flag_criteria all** (all elements): Selects all available elements without applying restrictions based on specific bits.
- **--flag_criteria** assimilee` (assimilated elements): Selects elements that have the BIT12 (4096) active, indicating that the element has influenced the analysis.
- **--flag_criteria** bgckalt` (approved by Background Check): Excludes elements that have the BIT9 (512) and BIT11 (2048) active, indicating rejections due to AO quality control and the selection process (thinning or canal), respectively.
- **--flag_criteria** bgckalt_qc` (approved by Background Check and QC-Var): Similar to `bgckalt`, but also excludes elements with BIT9 (512) and BIT11 (2048) active.
- **--flag_criteria** monitoring` (monitoring): This filter excludes elements with BIT9 (512) and BIT7 (128) active, indicating rejection by AO quality control (Background Check or QC-Var) and being in reserve, respectively.
- **--flag_criteria** postalt` (postal): This filter excludes elements with BIT17 (131072), BIT9 (512), BIT11 (2048), and BIT8 (256) active, indicating rejection by QC-Var, AO quality control, the selection process (thinning or canal), and being in the blacklist, respectively.

Description of Active Bits:
===========================

- **BIT0** (1) - Modified or generated by ADE.
- **BIT1** (2) - Element that exceeds a climatological extreme or fails consistency test.
- **BIT2** (4) - Erroneous element.
- **BIT3** (8) - Potentially erroneous element.
- **BIT4** (16) - Dubious element.
- **BIT5** (32) - Interpolated element, generated by DERIVATE.
- **BIT6** (64) - Corrected by DERIVATE sequence or bias correction.
- **BIT7** (128) - Rejected by satellite QC.
- **BIT8** (256) - Element rejected because it is on a blacklist.
- **BIT9** (512) - Element rejected by AO quality control (Background Check or QC-Var).
- **BIT10** (1024) - Generated by AO.
- **BIT11** (2048) - Rejected by a selection process (thinning or canal).
- **BIT12** (4096) - Assimilated.
- **BIT13** (8192) - Comparison against test field, level 1.
- **BIT14** (16384) - Comparison against test field, level 2.
- **BIT15** (32768) - Comparison against test field, level 3.
- **BIT16** (65536) - Rejected by comparison against test field (Background Check).
- **BIT17** (131072) - Rejected by QC-Var.
- **BIT18** (262144) - Not used due to orography.
- **BIT19** (524288) - Not used due to land-sea mask.
- **BIT20** (1048576) - Aircraft position error detected by TrackQc.
- **BIT21** (2097152) - Inconsistency detected by a CQ process.
- **BIT22** (4194304) - Rejected due to transmittance above model top
- **BIT23** (8388608) - Cloud affected radiance
    """

    # Define bit masks for flag criteria
    BIT6_BIASCORR = 64  
    BIT7_REJ = 128  
    BIT8_BLACKLIST = 256
    BIT9_REJ = 512
    BIT11_SELCOR = 2048  
    BIT12_VUE = 4096 
    BIT16_REJBGCK = 65536
    BIT17_QCVAR = 131072

    # Generate filtering criteria based on the selected flag
    if flags == "all":
        FLAG = "flag >= 0"
    elif flags == "assimilee":
        FLAG = f"and (flag & {BIT12_VUE}) = {BIT12_VUE}"
    elif flags == "bgckalt":
        FLAG = f"and (flag & {BIT9_REJ}) = 0 and (flag & {BIT11_SELCOR}) = 0 and (flag & {BIT8_BLACKLIST}) = 0"
    elif flags == "bgckalt_qc":
        FLAG = f"and (flag & {BIT9_REJ}) = 0 and (flag & {BIT11_SELCOR}) = 0"
    elif flags == "monitoring":
        FLAG = f"and (flag & {BIT9_REJ}) = 0 and (flag & {BIT7_REJ}) = 0"
    elif flags == "postalt":
        FLAG = f"and (flag & {BIT17_QCVAR}) = 0 and (flag & {BIT9_REJ}) = 0 and (flag & {BIT11_SELCOR}) = 0 and (flag & {BIT8_BLACKLIST}) = 0"
    elif flags == "rejets_qc":
        FLAG = f"and (flag & {BIT9_REJ}) = {BIT9_REJ}"
    elif flags == "rejets_bgck":
        FLAG = f"and (flag & {BIT16_REJBGCK}) = {BIT16_REJBGCK}"
    elif flags == "bias_corr":
        FLAG = f"and (flag & {BIT6_BIASCORR}) = {BIT6_BIASCORR}"
    elif flags == "qc":
        FLAG = f"and (flag & {BIT9_REJ}) = 0"
    else:
        raise ValueError(f'Invalid flag option: {flags}')

    return FLAG
