"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn = require("@aws-cdk/aws-cloudformation");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const path = require("path");
/**
 * A stack that will make a Lambda that will launch a lambda to glue
 * together all the DynamoDB tables into a global table
 */
class GlobalTableCoordinator extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, props);
        const lambdaFunction = new lambda.SingletonFunction(this, "SingletonLambda", {
            code: lambda.Code.fromAsset(path.resolve(__dirname, "../", "lambda-packages", "aws-global-table-coordinator", "lib")),
            description: "Lambda to make DynamoDB a global table",
            handler: "index.handler",
            runtime: lambda.Runtime.NODEJS_10_X,
            timeout: cdk.Duration.minutes(5),
            uuid: "D38B65A6-6B54-4FB6-9BAD-9CD40A6DAC12",
        });
        grantCreateGlobalTableLambda(lambdaFunction.role);
        new cfn.CustomResource(this, "CfnCustomResource", {
            provider: cfn.CustomResourceProvider.lambda(lambdaFunction),
            properties: {
                regions: props.regions,
                resourceType: "Custom::DynamoGlobalTableCoordinator",
                tableName: props.tableName,
            },
            removalPolicy: props.removalPolicy,
        });
    }
}
exports.GlobalTableCoordinator = GlobalTableCoordinator;
/**
 * Permits an IAM Principal to create a global dynamodb table.
 * @param principal The principal (no-op if undefined)
 */
function grantCreateGlobalTableLambda(principal) {
    if (principal) {
        principal.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                "iam:CreateServiceLinkedRole",
                "application-autoscaling:DeleteScalingPolicy",
                "application-autoscaling:DeregisterScalableTarget",
                "dynamodb:CreateGlobalTable", "dynamodb:DescribeLimits",
                "dynamodb:DeleteTable", "dynamodb:DescribeGlobalTable",
                "dynamodb:UpdateGlobalTable",
            ]
        }));
    }
}
//# sourceMappingURL=data:application/json;base64,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