# Copyright (c) 2024 XX Xiao

# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files(the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and / or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:

# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

r""" 
This module contains the Layout class, which is designed to select suitable layouts 
for quantum circuits on hardware backends.
"""

import os
#import warnings
import copy
import networkx as nx
import numpy as np
from typing import Literal
from itertools import combinations,zip_longest,product
from multiprocessing import Pool
from functools import partial
from quark.circuit.backend import Backend
from quark.circuit.quantumcircuit import QuantumCircuit
from quark.circuit.dag import split_qubits

class Layout:
    """
    Responsible for selecting suitable qubit layouts from a given chip for a quantum circuit.

    This class generates qubit layouts based on the required number of qubits, performance metrics, and the topology of the chip. It is designed to help map and execute quantum circuits on specific quantum hardware.
    """
    def __init__(self, chip_backend: Backend):
        """Initialize the Layout class with the required number of qubits and chip backend.

        Args:
            nqubits (int): The number of qubits needed in the layout.
            chip_backend (Backend): An instance of the Backend class that contains the information 
            about the quantum chip to be used for layout selection
        """
        self.priority_qubits = chip_backend.priority_qubits
        self.graph = chip_backend.edge_filtered_graph(thres=0.6)
        self.ncore = os.cpu_count() // 2 
        self.fidelity_mean_threshold = 0.9
        self.edge_fidelitys = nx.get_edge_attributes(self.graph,'fidelity') #提前存下边信息节约计算资源
        self.algorithm_switch_threshold = 10

    def _get_node_neighbours(self,node:int):
        return list(self.graph.neighbors(node))
    
    def _get_node_connect_dict(self,node:int,nqubits:int):
        """ Generates a dictionary representing the multi-level neighbor connectivity of a given node.

        Args:
            node (int):The starting node for generating the connectivity dictionary.

        Returns:
            dict: A dictionary where keys are nodes and values are lists of neighboring nodes 
            up to `nqubits - 1` levels deep, each list representing the connectivity at that level.
        """
        current_neighbours = [i for i in self._get_node_neighbours(node) if i > node]
        dd = {node:current_neighbours}
        remove = list(range(node+1))
        for _ in range(nqubits-2):
            current = []
            for node0 in current_neighbours:
                node0_neighbours = self._get_node_neighbours(node0)
                node0_neighbours = [i for i in node0_neighbours if i not in remove]
                current.append(node0_neighbours)
                dd[node0] = node0_neighbours
            current_neighbours = list(set(item for sublist in current for item in sublist))
        return dd

    def get_one_node_subgraph(self,node:int,nqubits:int):
        """Generates all possible subgraph combinations for a given node up to a specified number of nodes.

        Args:
            node (int): The starting node for generating subgraph combinations.

        Returns:
            list[tuple]:A list of tuples, each representing a unique combination of nodes that form
              a subgraph up to the specified `nqubits` in size.
        """

        def post_combinations(mid,dd,cut):
            rr = set([elem for node in mid if node in dd for elem in dd[node]])
            cc = []
            mm = min(cut,len(dd)) +1
            for idx in range(1,mm):
                cc +=  [list(comb) for comb in combinations(rr, idx)]
            return cc
        
        dd = self._get_node_connect_dict(node,nqubits)
        collect = []
        init = [{'pre':[],'mid':[node],'post':post_combinations([node],dd,nqubits-1)}]
        for _ in range(nqubits):
            update = []
            for c0 in init:
                new_pre = c0['pre'] + c0['mid']
                new_pre.sort()
                new_pre = list(set(new_pre))
                if len(new_pre) == nqubits:
                    new_pre.sort()
                    collect.append(tuple(new_pre))
                elif len(new_pre) < nqubits:
                    if c0['post'] == []:
                        continue
                    else:
                        for mid0 in c0['post']:
                            mid = [i for i in mid0 if i not in new_pre]
                            c1 = {'pre':new_pre,'mid':mid,'post':post_combinations(mid,dd,nqubits-len(new_pre+mid))}
                            update.append(c1)
            init = update
        return list(set(collect))
    
    def collect_all_subgraph_in_parallel(self,nqubits):
        """Collects all possible subgraph combinations for all nodes in the graph in parallel.

        Returns:
            list[tuple]:  A list of tuples, each representing a unique combination of nodes that 
                     form subgraphs for all nodes in the graph.
        """
        collect_all = []
        with Pool(processes = self.ncore) as pool:
            res = pool.map(partial(self.get_one_node_subgraph,nqubits=nqubits),self.graph.nodes())
        for collect in res:
            collect_all += collect
        return collect_all

    def get_one_subgraph_info(self,nodes:tuple|list):
        """Retrieves information about a specified subgraph.
        
        This method generates a subgraph from the given list of nodes, calculates the degree of each node within the subgraph, 
        and computes the mean and variance of the edge weights (fidelity) in the subgraph.It returns the subgraph information 
        only if the mean fidelity meets the specified threshold.

        Args:
            nodes (tuple|list): A list of nodes that define the subgraph.

        Returns:
            tuple or None: A tuple containing the nodes, their degrees, mean fidelity, and variance of fidelity 
                           if the mean fidelity is greater than or equal to `fidelity_mean_threshold`. Otherwise, returns None.
        """
        subgraph = self.graph.subgraph(nodes)
        subgraph_degree = dict(subgraph.degree())
        subgraph_fidelity = np.array([self.edge_fidelitys[(min(edge),max(edge))] for edge in subgraph.edges])
        fidelity_mean = np.mean(subgraph_fidelity)
        fidelity_var  = np.var(subgraph_fidelity)  
        if fidelity_mean >= self.fidelity_mean_threshold:
            nodes_info = (nodes,subgraph_degree,fidelity_mean,fidelity_var)
            return nodes_info
        else:
            return None
     
    def collect_all_subgraph_info_in_parallel(self,nqubits:int):
        """Collects information about all subgraphs in parallel.

        Returns:
            list: A list of results, where each entry corresponds to the information of a subgraph. 
        """
        all_subgraph = self.collect_all_subgraph_in_parallel(nqubits)
        with Pool(processes = self.ncore) as pool:
            res = pool.map(partial(self.get_one_subgraph_info),all_subgraph)
        return res  

    def classify_all_subgraph_according_topology(self,nqubits:int) -> tuple[list,list,list,list]:
        """
        Classify the collected subgraphs based on their topological structure into four categories.

        This function sorts the subgraphs into the following four categories:
        1. Linear and connected, with all nodes in the same row of the chip.
        2. Linear and connected, with nodes not in the same row.
        3. Contains a cycle within the subgraph.
        4. Non-linear and connected, where some nodes have more than three edges.
    
        Returns:
            tuple[list, list, list, list]: A tuple containing four lists, each corresponding 
            to one of the four categories of subgraphs.
        """

        linear_subgraph_list  = []
        nonlinear_subgraph_list = []
        all_subgraph_info = self.collect_all_subgraph_info_in_parallel(nqubits)

        for subgraph_info in filter(lambda x: x is not None, all_subgraph_info):
            nodes,subgraph_degree,fidelity_mean,fidelity_var = subgraph_info
            nodes_info = (nodes, fidelity_mean, fidelity_var)
            if max(subgraph_degree.values()) <= 2:
                linear_subgraph_list.append(nodes_info)
            else:
                nonlinear_subgraph_list.append(nodes_info)
        return linear_subgraph_list,nonlinear_subgraph_list
    
    def sort_subgraph_according_mean_fidelity(self, nqubits:int, num:int=1,  printdetails: bool = True):
        """Sort each of the four subgraph categories based on the main of fidelity on the edges (couplers), 
        in ascending order.

        Args:
            printdetails (bool, optional): If True, print details of the sorting process. Defaults to True.

        Returns:
            tuple[list, list, list, list]: Four sorted lists, each corresponding to one of the four 
            subgraph categories, with subgraphs sorted by edge fidelity variance.
        """
        linear_subgraph_list, nonlinear_subgraph_list = self.classify_all_subgraph_according_topology(nqubits)
        linear_subgraph_list_sort = sorted(linear_subgraph_list,key=lambda x: x[1],reverse=True)
        nonlinear_subgraph_list_sort = sorted(nonlinear_subgraph_list,key=lambda x: x[1],reverse=True)
        if printdetails:
            print(len(linear_subgraph_list_sort),len(nonlinear_subgraph_list_sort))
            print('The average fidelity is arranged in descending order,only print the first ten.')
            length = nqubits*5+22

            print('{:<3} | {:^{}} | {:^{}} '.format(\
                'idx','subgraph with linear topology',length,'subgraph with nonlinear topology',length))
            for i, (linear,nonlinear) in enumerate(zip_longest(linear_subgraph_list_sort,nonlinear_subgraph_list_sort, fillvalue=' ')):
                if i >= len(linear_subgraph_list_sort):
                    linear = ('(                  )',0.0,0.0)
                if i >= len(nonlinear_subgraph_list_sort):
                    nonlinear = ('(                  )',0.0,0.0)
                if i <= num:
                    print('{:<3} | {:<{}} {:<10.6f} {:<10.6f} | {:<{}} {:<10.6f} {:<10.6f} '\
                          .format(i, \
                                  str(linear[0]),nqubits*5,linear[1],linear[2],\
                                  str(nonlinear[0]),nqubits*5,nonlinear[1],nonlinear[2])\
                                  )
                    
        return linear_subgraph_list_sort[:num],nonlinear_subgraph_list_sort[:num]
    
    def sort_subgraph_according_var_fidelity(self, nqubits:int, num:int = 1, printdetails: bool = True):
        """
        Sort each of the four subgraph categories based on the variance of fidelity on the edges (couplers), 
        in ascending order.
    
        This function sorts the subgraphs within each category (from the previous classification) by the 
        variance of fidelity across the edges in each subgraph, from lowest to highest.
    
        Args:
            printdetails (bool, optional): If True, print details of the sorting process. Defaults to True.
    
        Returns:
            tuple[list, list, list, list]: Four sorted lists, each corresponding to one of the four 
            subgraph categories, with subgraphs sorted by edge fidelity variance.
        """
        linear_subgraph_list, nonlinear_subgraph_list = self.classify_all_subgraph_according_topology(nqubits)
        linear_subgraph_list_sort = sorted(linear_subgraph_list,key=lambda x: x[2])
        nonlinear_subgraph_list_sort = sorted(nonlinear_subgraph_list,key=lambda x: x[2])
        
        if printdetails:
            print(len(linear_subgraph_list_sort),len(nonlinear_subgraph_list_sort))
            print('The average fidelity is arranged in descending order, only print the first ten.')
            length = nqubits*5+22

            print('{:<3} | {:^{}} | {:^{}} '.format(\
                'idx','subgraph with linear topology',length,'subgraph with nonlinear topology',length))
            for i, (linear,nonlinear) in enumerate(zip_longest(linear_subgraph_list_sort, nonlinear_subgraph_list_sort, fillvalue=' ')):
                if i >= len(linear_subgraph_list_sort):
                    linear = ('(                  )',0.0,0.0)
                if i >= len(nonlinear_subgraph_list_sort):
                    nonlinear = ('(                  )',0.0,0.0)
                
                if i <= num:
                    print('{:<3} | {:<{}} {:<10.6f} {:<10.6f} | {:<{}} {:<10.6f} {:<10.6f} '\
                          .format(i, \
                                  str(linear[0]),nqubits*5,linear[1],linear[2],\
                                  str(nonlinear[0]),nqubits*5,nonlinear[1],nonlinear[2])\
                                  )

        return linear_subgraph_list_sort[:num], nonlinear_subgraph_list_sort[:num]
    
    def select_one_qubit_from_backend(self,):
        # 当线路中只有一个qubit时，挑选单比特门保真度最大的比特
        for nodes in nx.connected_components(self.graph):
            if len(nodes) > 1:
                subgraph = self.graph.subgraph(nodes)
                break
        node_fidelity_dic = nx.get_node_attributes(subgraph,'fidelity')
        sorted_dict = dict(sorted(node_fidelity_dic.items(), key=lambda item: item[1], reverse=True))
        qubit = [list(sorted_dict.keys())[0]]
        print(f'Physical qubits layout {qubit} is selected based on maximum single-qubit gate fidelity.')
        return qubit 

    def select_few_qubits_from_backend(self, 
                                         nqubits:int,
                                         key: Literal['fidelity_mean', 'fidelity_var'] = 'fidelity_var',
                                         topology: Literal['linear', 'nonlinear'] = 'linear',
                                         printdetails: bool = False):
        """
        Select a qubit layout based on the given performance metric and topology.
    
        This function chooses a layout for the quantum circuit from the available subgraphs based on 
        the specified key (performance metric) and topology type.
    
        Args:
            key (Literal['fidelity_mean', 'fidelity_var'], optional): The performance metric to use for 
                selecting the layout. Either the mean fidelity ('fidelity_mean') or fidelity variance 
                ('fidelity_var'). Defaults to 'fidelity_var'.
            topology (Literal['cycle', 'linear1', 'linear', 'nonlinear'], optional): The desired topology 
                of the layout. It can be 'cycle', 'linear1' (connected, in the same row), 'linear' (connected, 
                not necessarily in the same row), or 'nonlinear'. Defaults to 'linear1'.
            printdetails (bool, optional): If True, prints details about the selected layout. Defaults to False.
    
        Returns:
            list: A list of qubits representing the selected layout.
        """
        if key == 'fidelity_mean':
            linear_list,nonlinear_list = self.sort_subgraph_according_mean_fidelity(nqubits,printdetails=printdetails)
        elif key == 'fidelity_var':
            linear_list,nonlinear_list = self.sort_subgraph_according_var_fidelity(nqubits, printdetails=printdetails)
        
        if topology == 'linear':
            layouts = linear_list
        elif topology == 'nonlinear':
            layouts = nonlinear_list

        if len(layouts) == 0:
            raise(ValueError(f'There is no {nqubits} qubits that meets both key = {key} and topology = {topology}. Please change the conditions.'))
        else:
            print(f'Physical qubits layout {layouts[0][0]} are selected by the local algorithm using key = {key} and topology = {topology}.')
            return list(layouts[0][0])

    # large qubit layout 
    def _get_largest_component(self):
        components = list(nx.connected_components(self.graph))
        len_comp = [len(comp) for comp in components]
        idx = len_comp.index(max(len_comp))
        return self.graph.subgraph(components[idx])
        
    
    def select_much_qubits_from_backend(self,nqubits): #get_BFS_layout(self,nqubits:int):
        """ Perform a breadth-first search (BFS) on the graph starting from the start node,
        collecting up to `nqubits` unique nodes including the start node.

        Returns:
            list: A list of up to `nqubits` unique nodes, discovered in BFS order.
        """
        # self.graph = chip_backend.graph #chip_backend.edge_filtered_graph(thres=0.95)

        one_subgraph = self._get_largest_component()
        if len(one_subgraph.nodes()) < nqubits:
            raise(ValueError(f'The user circuit requires {nqubits} qubits exceeds the qubit capacity of the largest connected subgraph. This triggered by low fidelity filtering, if you insist on using more qubits, please contact the developer.'))
        start_node = np.random.choice(list(one_subgraph.nodes))

        visited = set([start_node])
        queue = [(start_node, 0)]  
        while queue and len(visited) < nqubits:
            current_node, depth = queue.pop(0)  
            if depth >= nqubits - 1:
                continue
            for neighbor in one_subgraph.neighbors(current_node):
                if neighbor not in visited:
                    visited.add(neighbor)
                    queue.append((neighbor, depth + 1))
                    if len(visited) == nqubits:
                        break
        print(f'Physical qubits layout {list(visited)} are selected by BFS algorithm.') # with the corresponding coupling being {coupling_map}
        return list(visited)
        
    def select_qubits_by_local_algorithm(self,nqubits,select_criteria):
        if nqubits == 1:
            qubit = self.select_one_qubit_from_backend()
            return qubit
        elif 1 < nqubits <= self.algorithm_switch_threshold:
            key_first = select_criteria['key']
            topology_first = select_criteria['topology']
            all_keys = ['fidelity_var','fidelity_mean']
            all_topologys = ['linear','nonlinear']
            all_keys.remove(key_first)
            all_topologys.remove(topology_first)
            sorted_keys = [key_first,] + all_keys
            sorted_topologys = [topology_first,] + all_topologys
            physical_qubits_layout = []
            for key, topology in product(sorted_keys,sorted_topologys):
                try:
                    physical_qubits_layout = self.select_few_qubits_from_backend(nqubits, key=key, topology=topology)
                except Exception as e:
                    physical_qubits_layout = []
                    print(f'Warning! {e}')
                if physical_qubits_layout != []:
                    break
            if physical_qubits_layout == []:
                raise(ValueError(f'Unable to find a suitable layout. If this message appears, please contact the developer for assistance.'))
            return physical_qubits_layout
        elif nqubits > self.algorithm_switch_threshold:
            physical_qubits_layout = self.select_much_qubits_from_backend(nqubits)
            return physical_qubits_layout
        else:
            raise(ValueError('Wrong qubits error!'))

    def select_layout(self,
                      qc: QuantumCircuit,
                      target_qubits:list = [], 
                      use_chip_priority: bool = True, 
                      select_criteria: dict = {'key':'fidelity_var','topology':'linear'}, skip_split_qc:bool = True,
                        ):
        
        nqubits = len(qc.qubits)
        if skip_split_qc is True:
            all_qubits = [qc.qubits]
        else:
            all_qubits = split_qubits(qc)
        self.source_graph = copy.deepcopy(self.graph)

        if target_qubits != []:
            if len(set(target_qubits)) != nqubits:
                raise(ValueError(f'The number of qubits {len(target_qubits)} in target_qubits does not match the number of qubits {nqubits} in the circuit.'))
            # check qubits existance 
            lose_nodes = []
            for qubit in target_qubits:
                if self.graph.has_node(qubit) is False:
                    lose_nodes.append(qubit)
            if lose_nodes != []:
                raise(ValueError(f"These qubit(s) {lose_nodes} does not exist.This maybe due to an incorrected input index or low fidelity filtering. For the latter case, if you insist on using it, please contact the developer."))
            # check subgraph connection according non-zero fidelity
            idx = 0
            for qubits0 in all_qubits:
                part_target_qubits = target_qubits[idx:idx+len(qubits0)]
                idx += len(qubits0)
                if len(part_target_qubits)>1:
                    subgraph0 = self.graph.subgraph(part_target_qubits).copy()
                    if nx.is_connected(subgraph0) is False:
                        raise(ValueError(f'The target physical qubits {part_target_qubits} corresponding to virtual qubits {qubits0} are not connected.'))  
            # print information             
            subgraph = self.graph.subgraph(target_qubits).copy()
            subgraph.graph["normal_order"] = target_qubits
            coupling_map = list(subgraph.edges)
            print(f'Physical qubits layout {target_qubits} are user-defined, with the corresponding coupling being {coupling_map}.')       
            if len(subgraph.edges())>0:     
                subgraph_fidelity = np.array([self.edge_fidelitys[(min(edge),max(edge))] for edge in subgraph.edges])
                fidelity_mean = np.mean(subgraph_fidelity)
                fidelity_var  = np.var(subgraph_fidelity)  
                print(f'The average fidelity of the coupler(s) between the selected qubits is {fidelity_mean}, and the variance of the fidelity is {fidelity_var}.')
            return subgraph
        
        if use_chip_priority is True: #priority qubits是测量人员选择出来处理全联通线路的比特组，如果线路可以被分割，用分割后的线路去匹配比特。
            priority_qubits_list = self.priority_qubits
            new_qubits = []
            for qubits0 in all_qubits:
                is_priority_provided = False
                priority_qubits_list = [q for q in priority_qubits_list if q not in new_qubits]
                for qubits in priority_qubits_list:
                    if len(qubits0) == len(qubits):
                        subgraph0 = self.source_graph.subgraph(qubits).copy()
                        is_overlap = any(x in qubits for sub in new_qubits for x in sub)
                        if nx.is_connected(subgraph0) is True and is_overlap is False:
                            new_qubits.append(qubits)
                            #print('+++++++++++++++++++',qubits,'from priority_qubits')
                            is_priority_provided = True
                        break
                    continue
                if is_priority_provided is False: 
                    print(f'No more priority qubits were found. it will check the select_criteria for search')
                    # 分别匹配失败，启动筛选，筛选前graph需要过滤掉已经使用的qubit
                    self.graph.remove_nodes_from([x for sub in new_qubits for x in sub])
                    qubits = self.select_qubits_by_local_algorithm(len(qubits0),select_criteria)
                    new_qubits.append(qubits)
                    #print('*******************',qubits,'from local algorithm')
            #print(new_qubits)
            subgraph = self.source_graph.subgraph([x for sub in new_qubits for x in sub])
            subgraph.graph['normal_order'] = [x for sub in new_qubits for x in sub]
            return subgraph
        else:
            new_qubits = []
            for qubits0 in all_qubits:
                self.graph.remove_nodes_from(list(set(e for sub in new_qubits for e in sub)))
                try:
                    qubits = self.select_qubits_by_local_algorithm(len(qubits0),select_criteria)
                    new_qubits.append(qubits)
                except Exception as e:
                    raise(ValueError(f'Local algorithm search layout Faild {e}'))
            subgraph = self.source_graph.subgraph([x for sub in new_qubits for x in sub])
            subgraph.graph['normal_order'] = [x for sub in new_qubits for x in sub]
            return subgraph
