# AgenticlyPay

A Python library for processing agentic payments (ACP, AP2, x402) via Stripe. Built for agentic developers who need automated payment processing, monthly Zelle payouts, and tax compliance.

## Features

- **Multi-Protocol Support**: Process payments using ACP (Agentic Commerce Protocol), AP2 (Agent Payments Protocol), or x402 protocols
- **Stripe Connect Integration**: Seamless onboarding and account management for developers
- **Automated Zelle Payouts**: Monthly automated payouts via Zelle to your email address
- **Tax Compliance**: Automatic 1099 form generation and filing
- **Transparent Pricing**: 6.5% + $0.30 per transaction

## Important: Email Requirements

**Your email address must be Zelle-compatible** (US bank account required). Monthly payouts are automatically sent via Zelle to the email address you use when creating your developer account.

## Installation

```bash
pip install agenticlypay
```

## Quick Start

### Basic Usage

```python
from agenticlypay import PaymentProcessor, ConnectManager

# Initialize components
payment_processor = PaymentProcessor()
connect_manager = ConnectManager()

# Create a developer account
# IMPORTANT: Use a Zelle-compatible email address (US bank account required)
account = connect_manager.create_developer_account(
    email="developer@example.com",  # Must be Zelle-compatible
    country="US"
)

# Unified payment (AUTO): include `mandate` for AP2 or `resource_url` for x402
result = payment_processor.process_payment(
    protocol="AUTO",
    amount=10000,  # $100.00 in cents
    developer_account_id=account["account_id"],
    currency="usd",
    # mandate={...},         # uncomment to select AP2
    # resource_url="/...",  # uncomment to select x402
)
```

### ACP Payment Example

```python
from agenticlypay import PaymentProcessor

processor = PaymentProcessor()

result = processor.process_payment(
    protocol="ACP",
    amount=10000,  # $100.00 in cents
    currency="usd",
    developer_account_id="acct_xxxxx",
    description="Payment for service"
)
```

**Note**: When creating your developer account, use a Zelle-compatible email address. Monthly payouts will be automatically sent to this email via Zelle.

### AP2 Payment Example

```python
result = processor.process_payment(
    protocol="AP2",
    amount=10000,
    currency="usd",
    developer_account_id="acct_xxxxx",
    mandate={
        "agent_id": "agent_123",
        "user_id": "user_456",
        "permissions": ["create_payment", "complete_purchase"],
        "expires_at": 1735689600,
        "mandate_id": "mandate_789"
    }
)
```

### x402 Payment Example

```python
result = processor.process_payment(
    protocol="x402",
    amount=10000,
    currency="usd",
    developer_account_id="acct_xxxxx",
    resource_url="/api/data/endpoint"
)
```

## Configuration

Set environment variables or use a `.env` file:

```env
STRIPE_SECRET_KEY=sk_live_...
STRIPE_WEBHOOK_SECRET=whsec_...
PLATFORM_FEE_PERCENTAGE=0.065
PLATFORM_FEE_FIXED=30
```

## Monthly Zelle Payouts

Monthly payouts are automatically sent via Zelle to the email address associated with your developer account. No additional configuration needed - just ensure your email is Zelle-compatible (linked to a US bank account).

## API Documentation

For complete API documentation and integration guides, visit [https://agenticlypay.com](https://agenticlypay.com).

## License

MIT License

## Support

For issues and questions, please visit our GitHub repository or contact support@agenticlypay.com

