"""Payout management routes."""

from fastapi import APIRouter, HTTPException
from pydantic import BaseModel, Field
from typing import Optional, List
from agenticlypay.payouts import PayoutManager
from agenticlypay.stripe_client import StripeClient

router = APIRouter()
payout_manager = PayoutManager()
stripe_client = StripeClient()


class CreatePayoutRequest(BaseModel):
    """Request model for creating a payout."""

    developer_account_id: str
    year: int
    month: int


class ProcessPayoutsRequest(BaseModel):
    """Request model for processing payouts for multiple accounts."""

    year: int
    month: int
    developer_account_ids: Optional[List[str]] = None


class TransferRequest(BaseModel):
    """Request model for creating a direct transfer to a connected account."""

    developer_account_id: str = Field(..., description="Connected account ID (acct_...)")
    amount: int = Field(..., gt=0, description="Amount in cents")
    currency: str = Field("usd", description="ISO currency")
    reference: Optional[str] = Field(None, description="Reference or description")


@router.get("/earnings/{account_id}/{year}/{month}")
async def get_monthly_earnings(account_id: str, year: int, month: int):
    """Get monthly earnings for a developer."""
    try:
        earnings = payout_manager.calculate_monthly_earnings(account_id, year, month)
        return {"success": True, "earnings": earnings}
    except Exception as e:
        raise HTTPException(status_code=400, detail=str(e))


@router.post("/transfer")
async def create_direct_transfer(request: TransferRequest):
    """Create a direct transfer to a connected account."""
    try:
        metadata = {}
        if request.reference:
            metadata["reference"] = request.reference
            metadata["payout_type"] = "manual"
        transfer = stripe_client.create_transfer(
            amount=request.amount,
            currency=request.currency,
            destination=request.developer_account_id,
            metadata=metadata or None,
        )
        return {
            "success": True,
            "transfer_id": transfer.id,
            "amount": transfer.amount,
            "currency": transfer.currency,
            "destination": transfer.destination,
            "created": transfer.created,
            "status": transfer.status,
        }
    except Exception as e:
        raise HTTPException(status_code=400, detail=str(e))

@router.post("/create")
async def create_payout(request: CreatePayoutRequest):
    """Create a monthly payout for a developer."""
    try:
        payout = payout_manager.create_monthly_payout(
            developer_account_id=request.developer_account_id,
            year=request.year,
            month=request.month,
        )
        return {"success": payout["success"], "payout": payout}
    except Exception as e:
        raise HTTPException(status_code=400, detail=str(e))


@router.post("/process-all")
async def process_all_payouts(request: ProcessPayoutsRequest):
    """Process monthly payouts for all or specified developers."""
    try:
        results = payout_manager.process_all_monthly_payouts(
            year=request.year,
            month=request.month,
            developer_account_ids=request.developer_account_ids,
        )
        return {"success": True, "results": results}
    except Exception as e:
        raise HTTPException(status_code=400, detail=str(e))


@router.get("/history/{account_id}")
async def get_payout_history(account_id: str, limit: int = 10):
    """Get payout history for a developer."""
    try:
        history = payout_manager.get_payout_history(account_id, limit=limit)
        return {"success": True, "history": history}
    except Exception as e:
        raise HTTPException(status_code=400, detail=str(e))

