from typing import Generic, Literal, Optional, Self, TypeAlias, TypeVar, Union, overload

import numpy as np
from numpy.typing import NDArray

from relife._typing import _Any_Number

__all__ = ["ParametricModel", "FrozenParametricModel"]

class _Parameters:
    parent: Optional[Self]
    _leaves = dict[str, Self]
    _mapping = dict[str, float]
    _all_values: tuple[float, ...]
    _all_names: tuple[str, ...]

    def __init__(self, **kwargs: Optional[float]) -> None: ...
    @property
    def all_names(self) -> tuple[str, ...]: ...
    @property
    def all_values(self) -> tuple[float, ...]: ...
    @property
    def size(self) -> int: ...
    def set_leaf(self, leaf_name: str, leaf: _Parameters) -> None: ...
    def set_all_values(self, values: tuple[float, ...]) -> None: ...
    def __getitem__(self, name: str) -> float: ...
    def update_tree(self) -> None: ...

class ParametricModel:
    _params: _Parameters

    def __init__(self, **kwparams: Optional[float]) -> None: ...
    @property
    def params(self) -> NDArray[np.float64]: ...
    # noinspection PyUnresolvedReferences
    @params.setter
    def params(self, new_params: NDArray[np.float64]): ...
    @property
    def params_names(self) -> tuple[str, ...]: ...
    @property
    def nb_params(self) -> int: ...

_ParametricModel_T_co = TypeVar("_ParametricModel_T_co", bound=ParametricModel, covariant=True)
_Frozen_Args: TypeAlias = list[_Any_Number]

# covariance matters :
# if func(model : FrozenParametricModel[ParametricModel]), then because of covariance
# func(model : FrozenParametricModel[ParametricLifetimeModel]) is acceptable because ParametricLifetimeModel is subtype of ParametricModel
class FrozenParametricModel(ParametricModel, Generic[_ParametricModel_T_co]):
    _nb_assets: int
    _args: _Frozen_Args
    unfrozen_model: _ParametricModel_T_co
    def __init__(self, model: _ParametricModel_T_co, *args: _Any_Number) -> None: ...
    def unfreeze(self) -> _ParametricModel_T_co: ...
    @property
    def nb_assets(self) -> int: ...
    @property
    def args(self) -> _Frozen_Args: ...
    @args.setter
    def args(self, value: _Frozen_Args) -> None: ...

@overload
def is_frozen(model: ParametricModel) -> Literal[False]: ...
@overload
def is_frozen(model: FrozenParametricModel[ParametricModel]) -> Literal[True]: ...
def is_frozen(model: Union[ParametricModel, FrozenParametricModel[ParametricModel]]) -> bool: ...
def get_nb_assets(*args: float | NDArray[np.float64]) -> int: ...

_ParametricModel_T = TypeVar("_ParametricModel_T", bound=ParametricModel)

def freeze_model_args(
    model: _ParametricModel_T, model_args: _Frozen_Args = ()
) -> FrozenParametricModel[_ParametricModel_T]: ...
