from aws_cdk import (
    App,
    Stack,
    StackProps,
    CfnOutput,
    SecretValue
)
from constructs import Construct
from must_cdk import AmplifyApp

class AdvancedAmplifyStack(Stack):
    def __init__(self, scope: Construct, id: str, **kwargs) -> None:
        super().__init__(scope, id, **kwargs)

        # Advanced Amplify app with multiple environments and custom domain
        amplify_app = AmplifyApp(
            self, "ProductionWebApp",
            app_name="enterprise-web-app",
            repository="https://github.com/company/web-app",
            access_token=SecretValue.secrets_manager("github-token"),
            
            # Build configuration
            build_settings={
                "compute_type": "STANDARD_8GB",
                "environment_variables": {
                    "NODE_ENV": "production",
                    "API_ENDPOINT": "https://api.company.com",
                    "ANALYTICS_ID": "GA-XXXXXXXXX"
                }
            },

            # Branch configurations
            branches=[
                {
                    "branch_name": "main",
                    "stage": "PRODUCTION",
                    "environment_variables": {
                        "ENVIRONMENT": "production"
                    }
                },
                {
                    "branch_name": "develop",
                    "stage": "DEVELOPMENT",
                    "environment_variables": {
                        "ENVIRONMENT": "development",
                        "API_ENDPOINT": "https://dev-api.company.com"
                    }
                },
                {
                    "branch_name": "staging",
                    "stage": "BETA",
                    "environment_variables": {
                        "ENVIRONMENT": "staging",
                        "API_ENDPOINT": "https://staging-api.company.com"
                    }
                }
            ],

            # Custom domain configuration
            custom_domain={
                "domain_name": "app.company.com",
                "sub_domains": [
                    {"branch_name": "main", "prefix": ""},
                    {"branch_name": "develop", "prefix": "dev"},
                    {"branch_name": "staging", "prefix": "staging"}
                ]
            },

            # Security and performance settings
            enable_branch_auto_deletion=True,
            enable_branch_auto_building=True,
            enable_pull_request_preview=True,
            
            # Custom build spec for Next.js optimization
            build_spec="""
                version: 1
                applications:
                  - frontend:
                      phases:
                        preBuild:
                          commands:
                            - npm ci
                            - npm run lint
                            - npm run test:ci
                        build:
                          commands:
                            - npm run build
                      artifacts:
                        baseDirectory: .next
                        files:
                          - '**/*'
                      cache:
                        paths:
                          - node_modules/**/*
                          - .next/cache/**/*
            """
        )

        # Output the app URL
        CfnOutput(
            self, "AmplifyAppUrl",
            value=amplify_app.app_url,
            description="Amplify App URL"
        )

# Example usage
app = App()
AdvancedAmplifyStack(app, "AdvancedAmplifyStack")
app.synth()
