import * as cdk from 'aws-cdk-lib';
import { Construct } from 'constructs';
import { AmplifyApp } from 'must-cdk';

export class AdvancedAmplifyStack extends cdk.Stack {
  constructor(scope: Construct, id: string, props?: cdk.StackProps) {
    super(scope, id, props);

    // Advanced Amplify app with multiple environments and custom domain
    const amplifyApp = new AmplifyApp(this, 'ProductionWebApp', {
      appName: 'enterprise-web-app',
      repository: 'https://github.com/company/web-app',
      accessToken: cdk.SecretValue.secretsManager('github-token'),
      
      // Build configuration
      buildSettings: {
        computeType: 'STANDARD_8GB',
        environmentVariables: {
          NODE_ENV: 'production',
          API_ENDPOINT: 'https://api.company.com',
          ANALYTICS_ID: 'GA-XXXXXXXXX'
        }
      },

      // Branch configurations
      branches: [
        {
          branchName: 'main',
          stage: 'PRODUCTION',
          environmentVariables: {
            ENVIRONMENT: 'production'
          }
        },
        {
          branchName: 'develop',
          stage: 'DEVELOPMENT',
          environmentVariables: {
            ENVIRONMENT: 'development',
            API_ENDPOINT: 'https://dev-api.company.com'
          }
        },
        {
          branchName: 'staging',
          stage: 'BETA',
          environmentVariables: {
            ENVIRONMENT: 'staging',
            API_ENDPOINT: 'https://staging-api.company.com'
          }
        }
      ],

      // Custom domain configuration
      customDomain: {
        domainName: 'app.company.com',
        subDomains: [
          { branchName: 'main', prefix: '' },
          { branchName: 'develop', prefix: 'dev' },
          { branchName: 'staging', prefix: 'staging' }
        ]
      },

      // Security and performance settings
      enableBranchAutoDeletion: true,
      enableBranchAutoBuilding: true,
      enablePullRequestPreview: true,
      
      // Custom build spec for Next.js optimization
      buildSpec: `
        version: 1
        applications:
          - frontend:
              phases:
                preBuild:
                  commands:
                    - npm ci
                    - npm run lint
                    - npm run test:ci
                build:
                  commands:
                    - npm run build
              artifacts:
                baseDirectory: .next
                files:
                  - '**/*'
              cache:
                paths:
                  - node_modules/**/*
                  - .next/cache/**/*
      `
    });

    // Output the app URL
    new cdk.CfnOutput(this, 'AmplifyAppUrl', {
      value: amplifyApp.appUrl,
      description: 'Amplify App URL'
    });
  }
}
