import aws_cdk as cdk
from aws_cdk import (
    aws_ec2 as ec2,
    aws_ecs as ecs,
    aws_iam as iam,
    aws_ssm as ssm,
    aws_secretsmanager as secretsmanager,
    aws_certificatemanager as acm,
    Duration,
    CfnOutput,
    Stack
)
from constructs import Construct
from must_cdk import EcsCodeDeploy


class EnhancedEcsCodeDeployStack(Stack):
    """
    Enhanced ECS CodeDeploy Stack demonstrating new features:
    - Environment variables support
    - Secrets management
    - Custom container names
    - Additional container configuration options
    - Container access after creation
    """

    def __init__(self, scope: Construct, construct_id: str, **kwargs) -> None:
        super().__init__(scope, construct_id, **kwargs)

        # Create VPC for ECS cluster
        vpc = ec2.Vpc(
            self, "EcsVpc",
            max_azs=3,
            nat_gateways=2,
            subnet_configuration=[
                ec2.SubnetConfiguration(
                    cidr_mask=24,
                    name="Public",
                    subnet_type=ec2.SubnetType.PUBLIC
                ),
                ec2.SubnetConfiguration(
                    cidr_mask=24,
                    name="Private",
                    subnet_type=ec2.SubnetType.PRIVATE_WITH_EGRESS
                )
            ]
        )

        # Create ECS cluster
        cluster = ecs.Cluster(
            self, "EcsCluster",
            vpc=vpc,
            cluster_name="enhanced-cluster",
            container_insights=True
        )

        # Create security group
        security_group = ec2.SecurityGroup(
            self, "EcsSecurityGroup",
            vpc=vpc,
            description="Security group for ECS service",
            allow_all_outbound=True
        )

        security_group.add_ingress_rule(
            ec2.Peer.any_ipv4(),
            ec2.Port.tcp(80),
            "Allow HTTP traffic from ALB"
        )

        security_group.add_ingress_rule(
            ec2.Peer.any_ipv4(),
            ec2.Port.tcp(443),
            "Allow HTTPS traffic from ALB"
        )

        # Create SSL certificate (replace with your domain)
        certificate = acm.Certificate(
            self, "Certificate",
            domain_name="example.com",
            validation=acm.CertificateValidation.from_dns()
        )

        # Create secrets for sensitive data
        db_secret = secretsmanager.Secret(
            self, "DatabaseSecret",
            description="Database connection credentials",
            generate_secret_string=secretsmanager.SecretStringGenerator(
                secret_string_template='{"username": "admin"}',
                generate_string_key="password",
                exclude_characters='"@/\\'
            )
        )

        # Create SSM parameters for configuration
        api_key_param = ssm.StringParameter(
            self, "ApiKeyParam",
            parameter_name="/myapp/api-key",
            string_value="your-api-key-here",
            description="API key for external service"
        )

        # Enhanced ECS service with new features
        ecs_service = EcsCodeDeploy(
            self, "EnhancedWebApiService",
            vpc=vpc,
            cluster=cluster,
            service_name="enhanced-web-api",
            task_cpu=1024,
            memory_limit=2048,
            security_groups=[security_group],
            subnets=ec2.SubnetSelection(
                subnet_type=ec2.SubnetType.PRIVATE_WITH_EGRESS
            ),
            enable_public_load_balancer=True,
            certificates=[certificate],
            
            # Enhanced container configuration with new features
            containers=[
                {
                    # Custom container name instead of generic "Container0"
                    "name": "web-api-container",
                    "image": ecs.ContainerImage.from_registry("nginx:latest"),
                    "container_port": 80,
                    "memory_reservation": 512,
                    "cpu": 512,
                    "essential": True,
                    
                    # Environment variables - now supported!
                    "environment": {
                        "NODE_ENV": "production",
                        "PORT": "80",
                        "LOG_LEVEL": "info",
                        "APP_VERSION": "1.0.0",
                        "REGION": self.region,
                        # Environment-specific configuration
                        "DB_TABLE_NAME": f"{self.stack_name}-users",
                        "CACHE_TTL": "3600"
                    },
                    
                    # Secrets from AWS Secrets Manager and SSM Parameter Store
                    "secrets": {
                        "DB_PASSWORD": ecs.Secret.from_secrets_manager(db_secret, "password"),
                        "DB_USERNAME": ecs.Secret.from_secrets_manager(db_secret, "username"),
                        "API_KEY": ecs.Secret.from_ssm_parameter(api_key_param)
                    },
                    
                    # Additional container configuration options
                    "working_directory": "/app",
                    "user": "nginx",
                    "entry_point": ["/docker-entrypoint.sh"],
                    "command": ["nginx", "-g", "daemon off;"],
                    
                    # Custom health check
                    "health_check": ecs.HealthCheck(
                        command=[
                            "CMD-SHELL",
                            "curl -f http://localhost:80/health || exit 1"
                        ],
                        interval=Duration.seconds(30),
                        timeout=Duration.seconds(5),
                        retries=3,
                        start_period=Duration.seconds(60)
                    )
                },
                
                # Second container example (sidecar pattern)
                {
                    "name": "log-router",
                    "image": ecs.ContainerImage.from_registry("fluent/fluent-bit:latest"),
                    "container_port": 24224,
                    "memory_reservation": 256,
                    "cpu": 256,
                    "essential": False,  # Non-essential sidecar
                    
                    "environment": {
                        "FLB_LOG_LEVEL": "info",
                        "AWS_REGION": self.region
                    },
                    
                    "working_directory": "/fluent-bit",
                    "command": ["/fluent-bit/bin/fluent-bit", "--config=/fluent-bit/etc/fluent-bit.conf"]
                }
            ],

            # Auto-scaling configuration
            auto_scaling={
                "min_capacity": 2,
                "max_capacity": 10,
                "cpu_scale": {
                    "target_utilization_percent": 70,
                    "scale_in_cooldown": Duration.minutes(5),
                    "scale_out_cooldown": Duration.minutes(2)
                },
                "memory_scale": {
                    "target_utilization_percent": 80,
                    "scale_in_cooldown": Duration.minutes(5),
                    "scale_out_cooldown": Duration.minutes(2)
                }
            },

            # Tags for resource management
            tags={
                "Environment": "production",
                "Application": "web-api",
                "Team": "platform",
                "CostCenter": "engineering"
            }
        )

        # Demonstrate accessing containers after creation (new feature!)
        web_api_container = ecs_service.containers[0]
        log_router_container = ecs_service.containers[1]

        # Grant additional permissions to the web API container's task role
        ecs_service.task_role.add_to_policy(iam.PolicyStatement(
            effect=iam.Effect.ALLOW,
            actions=[
                "s3:GetObject",
                "s3:PutObject"
            ],
            resources=["arn:aws:s3:::my-app-bucket/*"]
        ))

        # Output important values
        CfnOutput(
            self, "LoadBalancerDnsName",
            value=ecs_service.load_balancer_dns_name(),
            description="Load Balancer DNS Name"
        )

        CfnOutput(
            self, "ServiceArn",
            value=ecs_service.service_arn(),
            description="ECS Service ARN"
        )

        CfnOutput(
            self, "WebApiContainerName",
            value=web_api_container.container_name,
            description="Web API Container Name"
        )

        CfnOutput(
            self, "LogRouterContainerName",
            value=log_router_container.container_name,
            description="Log Router Container Name"
        )

        CfnOutput(
            self, "BlueTargetGroupArn",
            value=ecs_service.blue_target_group.target_group_arn,
            description="Blue Target Group ARN"
        )

        CfnOutput(
            self, "GreenTargetGroupArn",
            value=ecs_service.green_target_group.target_group_arn,
            description="Green Target Group ARN"
        )


# Example usage in app.py:
# app = cdk.App()
# EnhancedEcsCodeDeployStack(app, "EnhancedEcsCodeDeployStack",
#     env=cdk.Environment(
#         account=os.getenv('CDK_DEFAULT_ACCOUNT'),
#         region=os.getenv('CDK_DEFAULT_REGION')
#     )
# )
