import * as cdk from 'aws-cdk-lib';
import * as ec2 from 'aws-cdk-lib/aws-ec2';
import * as ecs from 'aws-cdk-lib/aws-ecs';
import * as iam from 'aws-cdk-lib/aws-iam';
import * as ssm from 'aws-cdk-lib/aws-ssm';
import * as secretsmanager from 'aws-cdk-lib/aws-secretsmanager';
import * as acm from 'aws-cdk-lib/aws-certificatemanager';
import { Construct } from 'constructs';
import { EcsCodeDeploy } from 'must-cdk';

/**
 * Enhanced ECS CodeDeploy Stack demonstrating new features:
 * - Environment variables support
 * - Secrets management
 * - Custom container names
 * - Additional container configuration options
 * - Container access after creation
 */
export class EnhancedEcsCodeDeployStack extends cdk.Stack {
  constructor(scope: Construct, id: string, props?: cdk.StackProps) {
    super(scope, id, props);

    // Create VPC for ECS cluster
    const vpc = new ec2.Vpc(this, 'EcsVpc', {
      maxAzs: 3,
      natGateways: 2,
      subnetConfiguration: [
        {
          cidrMask: 24,
          name: 'Public',
          subnetType: ec2.SubnetType.PUBLIC,
        },
        {
          cidrMask: 24,
          name: 'Private',
          subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS,
        }
      ]
    });

    // Create ECS cluster
    const cluster = new ecs.Cluster(this, 'EcsCluster', {
      vpc,
      clusterName: 'enhanced-cluster',
      containerInsights: true
    });

    // Create security group
    const securityGroup = new ec2.SecurityGroup(this, 'EcsSecurityGroup', {
      vpc,
      description: 'Security group for ECS service',
      allowAllOutbound: true
    });

    securityGroup.addIngressRule(
      ec2.Peer.anyIpv4(),
      ec2.Port.tcp(80),
      'Allow HTTP traffic from ALB'
    );

    securityGroup.addIngressRule(
      ec2.Peer.anyIpv4(),
      ec2.Port.tcp(443),
      'Allow HTTPS traffic from ALB'
    );

    // Create SSL certificate (replace with your domain)
    const certificate = new acm.Certificate(this, 'Certificate', {
      domainName: 'example.com',
      validation: acm.CertificateValidation.fromDns(),
    });

    // Create secrets for sensitive data
    const dbSecret = new secretsmanager.Secret(this, 'DatabaseSecret', {
      description: 'Database connection credentials',
      generateSecretString: {
        secretStringTemplate: JSON.stringify({ username: 'admin' }),
        generateStringKey: 'password',
        excludeCharacters: '"@/\\'
      }
    });

    // Create SSM parameters for configuration
    const apiKeyParam = new ssm.StringParameter(this, 'ApiKeyParam', {
      parameterName: '/myapp/api-key',
      stringValue: 'your-api-key-here',
      description: 'API key for external service'
    });

    // Enhanced ECS service with new features
    const ecsService = new EcsCodeDeploy(this, 'EnhancedWebApiService', {
      vpc,
      cluster,
      serviceName: 'enhanced-web-api',
      taskCPU: 1024,
      memoryLimit: 2048,
      securityGroups: [securityGroup],
      subnets: {
        subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS
      },
      enablePublicLoadBalancer: true,
      certificates: [certificate],
      
      // Enhanced container configuration with new features
      containers: [
        {
          // Custom container name instead of generic "Container0"
          name: 'web-api-container',
          image: ecs.ContainerImage.fromRegistry('nginx:latest'),
          containerPort: 80,
          memoryReservation: 512,
          cpu: 512,
          essential: true,
          
          // Environment variables - now supported!
          environment: {
            NODE_ENV: 'production',
            PORT: '80',
            LOG_LEVEL: 'info',
            APP_VERSION: '1.0.0',
            REGION: cdk.Stack.of(this).region,
            // Environment-specific configuration
            DB_TABLE_NAME: `${cdk.Stack.of(this).stackName}-users`,
            CACHE_TTL: '3600'
          },
          
          // Secrets from AWS Secrets Manager and SSM Parameter Store
          secrets: {
            DB_PASSWORD: ecs.Secret.fromSecretsManager(dbSecret, 'password'),
            DB_USERNAME: ecs.Secret.fromSecretsManager(dbSecret, 'username'),
            API_KEY: ecs.Secret.fromSsmParameter(apiKeyParam)
          },
          
          // Additional container configuration options
          workingDirectory: '/app',
          user: 'nginx',
          entryPoint: ['/docker-entrypoint.sh'],
          command: ['nginx', '-g', 'daemon off;'],
          
          // Custom health check
          healthCheck: {
            command: [
              'CMD-SHELL',
              'curl -f http://localhost:80/health || exit 1'
            ],
            interval: cdk.Duration.seconds(30),
            timeout: cdk.Duration.seconds(5),
            retries: 3,
            startPeriod: cdk.Duration.seconds(60)
          }
        },
        
        // Second container example (sidecar pattern)
        {
          name: 'log-router',
          image: ecs.ContainerImage.fromRegistry('fluent/fluent-bit:latest'),
          containerPort: 24224,
          memoryReservation: 256,
          cpu: 256,
          essential: false, // Non-essential sidecar
          
          environment: {
            FLB_LOG_LEVEL: 'info',
            AWS_REGION: cdk.Stack.of(this).region
          },
          
          workingDirectory: '/fluent-bit',
          command: ['/fluent-bit/bin/fluent-bit', '--config=/fluent-bit/etc/fluent-bit.conf']
        }
      ],

      // Auto-scaling configuration
      autoScaling: {
        minCapacity: 2,
        maxCapacity: 10,
        cpuScale: {
          targetUtilizationPercent: 70,
          scaleInCooldown: cdk.Duration.minutes(5),
          scaleOutCooldown: cdk.Duration.minutes(2)
        },
        memoryScale: {
          targetUtilizationPercent: 80,
          scaleInCooldown: cdk.Duration.minutes(5),
          scaleOutCooldown: cdk.Duration.minutes(2)
        }
      },

      // Tags for resource management
      tags: {
        Environment: 'production',
        Application: 'web-api',
        Team: 'platform',
        CostCenter: 'engineering'
      }
    });

    // Demonstrate accessing containers after creation (new feature!)
    const webApiContainer = ecsService.containers[0];
    const logRouterContainer = ecsService.containers[1];

    // Grant additional permissions to the web API container's task role
    ecsService.taskRole.addToPolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        's3:GetObject',
        's3:PutObject'
      ],
      resources: ['arn:aws:s3:::my-app-bucket/*']
    }));

    // Output important values
    new cdk.CfnOutput(this, 'LoadBalancerDnsName', {
      value: ecsService.loadBalancerDnsName(),
      description: 'Load Balancer DNS Name'
    });

    new cdk.CfnOutput(this, 'ServiceArn', {
      value: ecsService.serviceArn(),
      description: 'ECS Service ARN'
    });

    new cdk.CfnOutput(this, 'WebApiContainerName', {
      value: webApiContainer.containerName,
      description: 'Web API Container Name'
    });

    new cdk.CfnOutput(this, 'LogRouterContainerName', {
      value: logRouterContainer.containerName,
      description: 'Log Router Container Name'
    });

    new cdk.CfnOutput(this, 'BlueTargetGroupArn', {
      value: ecsService.blueTargetGroup.targetGroupArn,
      description: 'Blue Target Group ARN'
    });

    new cdk.CfnOutput(this, 'GreenTargetGroupArn', {
      value: ecsService.greenTargetGroup.targetGroupArn,
      description: 'Green Target Group ARN'
    });
  }
}

// Example usage in app.ts:
// const app = new cdk.App();
// new EnhancedEcsCodeDeployStack(app, 'EnhancedEcsCodeDeployStack', {
//   env: {
//     account: process.env.CDK_DEFAULT_ACCOUNT,
//     region: process.env.CDK_DEFAULT_REGION,
//   },
// });
