#!/usr/bin/env node
import * as cdk from 'aws-cdk-lib';
import * as lambda from 'aws-cdk-lib/aws-lambda';
import * as s3 from 'aws-cdk-lib/aws-s3';
import { Construct } from 'constructs';
import { AmplifyApp, ApiGatewayToLambda, CloudFrontToOrigins } from 'must-cdk';

/**
 * Example demonstrating Must CDK's unified tagging system
 * 
 * Environment tags are set via TAGS environment variable:
 * export TAGS="Product=MyApp,Environment=production,Owner=TeamName,CostCenter=Engineering"
 * 
 * These tags will be applied to all resources, with environment tags taking precedence
 * over construct-specific tags.
 */
class TaggingExampleStack extends cdk.Stack {
  constructor(scope: Construct, id: string, props?: cdk.StackProps) {
    super(scope, id, props);

    // Example 1: Amplify App with construct-specific tags
    // Environment tags will be merged with these tags
    new AmplifyApp(this, 'MyAmplifyApp', {
      appName: 'my-tagged-app',
      repository: 'https://github.com/user/my-app',
      accessToken: 'your-github-token', // In production, use AWS Secrets Manager
      tags: {
        Component: 'frontend',
        Framework: 'nextjs',
        Team: 'frontend-team',
        Version: 'v1.0.0',
        // Note: If TAGS contains "Team=platform-team", it will override "frontend-team"
      }
    });

    // Example 2: API Gateway with Lambda
    const apiFunction = new lambda.Function(this, 'ApiFunction', {
      runtime: lambda.Runtime.NODEJS_18_X,
      handler: 'index.handler',
      code: lambda.Code.fromInline(`
        exports.handler = async (event) => {
          return {
            statusCode: 200,
            body: JSON.stringify({ message: 'Hello from tagged API!' })
          };
        };
      `),
    });

    new ApiGatewayToLambda(this, 'MyAPI', {
      lambdaFunction: apiFunction,
      apiName: 'my-tagged-api',
      tags: {
        Component: 'api',
        Runtime: 'nodejs18',
        Team: 'backend-team',
        APIType: 'rest',
        // Environment tags will be merged with these
      }
    });

    // Example 3: CloudFront Distribution
    const assetsBucket = new s3.Bucket(this, 'AssetsBucket', {
      bucketName: 'my-tagged-assets-bucket',
      publicReadAccess: false,
    });

    new CloudFrontToOrigins(this, 'MyCDN', {
      s3Origins: [
        {
          id: 'assets-origin',
          bucket: assetsBucket,
          originPath: '/static',
        }
      ],
      tags: {
        Component: 'cdn',
        CacheStrategy: 'aggressive',
        Team: 'platform-team',
        ContentType: 'static-assets',
        // These will be merged with environment tags
      }
    });

    // Output information about tagging
    new cdk.CfnOutput(this, 'TaggingInfo', {
      value: 'Check AWS Console to see tags applied to all resources',
      description: 'Tags are applied automatically from TAGS environment variable and construct props'
    });
  }
}

// Example usage with different environments
const app = new cdk.App();

// Development stack
new TaggingExampleStack(app, 'TaggingExampleDev', {
  env: {
    account: process.env.CDK_DEFAULT_ACCOUNT,
    region: process.env.CDK_DEFAULT_REGION,
  },
  // Additional stack-level tags can be added here
  tags: {
    StackType: 'example',
    Purpose: 'demonstration'
  }
});

// Production stack (would typically be in separate deployment)
// new TaggingExampleStack(app, 'TaggingExampleProd', {
//   env: {
//     account: 'your-prod-account',
//     region: 'us-east-1',
//   },
//   tags: {
//     StackType: 'example',
//     Purpose: 'demonstration',
//     Environment: 'production' // This would be overridden by TAGS env var if set
//   }
// });

app.synth();

/*
Usage Examples:

1. Development Environment:
   export TAGS="Product=MyApp,Environment=development,Owner=DevTeam,AutoShutdown=true"
   cdk deploy TaggingExampleDev

2. Staging Environment:
   export TAGS="Product=MyApp,Environment=staging,Owner=QATeam,AutoShutdown=true"
   cdk deploy TaggingExampleDev

3. Production Environment:
   export TAGS="Product=MyApp,Environment=production,Owner=OpsTeam,Backup=required,Monitoring=enhanced"
   cdk deploy TaggingExampleProd

4. Complex Tags with Special Characters:
   export TAGS="Product=MyApp,URL=https://api.example.com/v1,Config=key1=val1&key2=val2"
   cdk deploy TaggingExampleDev

Final tags on resources will be a merge of:
- Environment tags (from TAGS variable) - highest precedence
- Construct-specific tags (from construct props)
- Stack-level tags (from stack props) - lowest precedence

Example final tags for AmplifyApp with TAGS="Product=MyApp,Environment=production,Team=platform":
- Product=MyApp (from environment)
- Environment=production (from environment)  
- Team=platform (from environment, overrides construct's "frontend-team")
- Component=frontend (from construct)
- Framework=nextjs (from construct)
- Version=v1.0.0 (from construct)
- StackType=example (from stack)
- Purpose=demonstration (from stack)
*/
