"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmplifyApp = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const child_process_1 = require("child_process");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const amplify = require("aws-cdk-lib/aws-amplify");
const constructs_1 = require("constructs");
const tags_1 = require("../common/tags");
class AmplifyApp extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.branches = [];
        // Get GitHub authentication
        const accessToken = this.getGitHubAuthentication(props);
        if (!accessToken) {
            throw new Error(`
❌ GitHub authentication is required for Amplify App deployment!

Please provide authentication using one of these methods:

1. 🔑 Personal Access Token (Current approach):
   AmplifyApp(this, 'MyApp', {
     appName: 'my-app',
     repository: 'https://github.com/user/repo',
     accessToken: 'your-github-token'
   })

2. 📱 Use GitHub CLI (for development):
   gh auth login
   
3. 🌍 Set environment variable:
   export MUFIN_PUSH_TOKEN=your-github-token

4. 🔒 Use AWS Secrets Manager:
   AmplifyApp(this, 'MyApp', {
     appName: 'my-app',
     repository: 'https://github.com/user/repo',
     accessToken: SecretValue.secretsManager('github-token').unsafeUnwrap()
   })

💡 Production Recommendation:
   After your project is set up and working, consider migrating to GitHub Apps
   for better security, organization support, and higher rate limits.
   See: https://docs.github.com/en/apps/creating-github-apps
      `);
        }
        // Apply tags to both construct and CloudFormation stack
        const finalTags = (0, tags_1.applyTags)(this, props.tags);
        // Create Amplify App
        this.app = new amplify.CfnApp(this, "App", {
            name: props.appName,
            repository: props.repository,
            accessToken: accessToken,
            environmentVariables: this.formatEnvVars(props.environmentVariables),
            buildSpec: props.buildSpec || this.getDefaultBuildSpec(props.platform),
            platform: props.platform || "WEB_COMPUTE",
            customRules: props.customRules || this.getDefaultCustomRules(),
            autoBranchCreationConfig: {
                enableAutoBuild: props.buildSettings?.enableBranchAutoBuild ?? false,
                enableAutoBranchCreation: props.buildSettings?.enableAutoBranchCreation ?? false,
                enablePullRequestPreview: true,
            },
            basicAuthConfig: props.basicAuth
                ? {
                    enableBasicAuth: props.basicAuth.enableBasicAuth ?? true,
                    username: props.basicAuth.username,
                    password: props.basicAuth.password,
                }
                : undefined,
            tags: Object.keys(finalTags).length > 0
                ? (0, tags_1.formatTagsForCfn)(finalTags)
                : undefined,
        });
        // Create branches
        if (props.branches) {
            props.branches.forEach((branch, index) => {
                const cfnBranch = new amplify.CfnBranch(this, `Branch${index}`, {
                    appId: this.app.attrAppId,
                    branchName: branch.branchName,
                    stage: branch.stage || "PRODUCTION",
                    environmentVariables: this.formatEnvVars(branch.environmentVariables),
                    enableAutoBuild: branch.enableAutoBuild ?? true,
                    framework: branch.framework,
                    buildSpec: branch.buildSpec,
                });
                this.branches.push(cfnBranch);
            });
        }
        else {
            // Default main branch
            const mainBranch = new amplify.CfnBranch(this, "MainBranch", {
                appId: this.app.attrAppId,
                branchName: "main",
                stage: "PRODUCTION",
                enableAutoBuild: true,
            });
            this.branches.push(mainBranch);
        }
        // Custom domain setup
        if (props.customDomain) {
            this.domain = new amplify.CfnDomain(this, "Domain", {
                appId: this.app.attrAppId,
                domainName: props.customDomain.domainName,
                enableAutoSubDomain: props.customDomain.enableAutoSubdomain,
                autoSubDomainCreationPatterns: props.customDomain.autoSubdomainCreationPatterns,
                subDomainSettings: props.customDomain.subDomains?.map((sub) => ({
                    branchName: sub.branchName,
                    prefix: sub.prefix,
                })) || [
                    {
                        branchName: "main",
                        prefix: "",
                    },
                ],
            });
        }
        // Outputs
        new aws_cdk_lib_1.CfnOutput(this, "AppId", {
            value: this.app.attrAppId,
            description: "Amplify App ID",
        });
        new aws_cdk_lib_1.CfnOutput(this, "DefaultDomain", {
            value: this.app.attrDefaultDomain,
            description: "Amplify Default Domain",
        });
        if (this.domain) {
            new aws_cdk_lib_1.CfnOutput(this, "CustomDomain", {
                value: props.customDomain.domainName,
                description: "Amplify Custom Domain",
            });
        }
    }
    formatEnvVars(envVars) {
        if (!envVars)
            return undefined;
        return Object.entries(envVars).map(([name, value]) => ({ name, value }));
    }
    getDefaultCustomRules() {
        return [
            {
                source: "/<*>",
                target: "/index.html",
                status: "404-200",
            },
        ];
    }
    getGitHubAuthentication(props) {
        // Priority 1: Direct access token
        if (props.accessToken) {
            return props.accessToken;
        }
        // Priority 2: Fallback to GitHub CLI or environment variable
        return this.getGitHubToken();
    }
    getGitHubToken() {
        // Check if we're in a test environment
        const isTestEnvironment = process.env.NODE_ENV === "test" ||
            process.env.JEST_WORKER_ID !== undefined ||
            process.env.CI === "true";
        if (isTestEnvironment) {
            return process.env.MUFIN_PUSH_TOKEN || "mock-github-token-for-testing";
        }
        try {
            // Try to get token from gh CLI first
            const token = (0, child_process_1.execSync)("gh auth token", {
                encoding: "utf8",
                stdio: "pipe",
            })
                .toString()
                .trim();
            if (token && token.length > 0) {
                return token;
            }
        }
        catch (error) {
            // Silently continue to environment variable fallback
        }
        // Fallback to environment variable
        const envToken = process.env.MUFIN_PUSH_TOKEN;
        if (envToken && envToken.length > 0) {
            return envToken;
        }
        // Only log error if no token found anywhere
        console.log("❌ No GitHub token found! Please either:");
        console.log("   1. Run 'gh auth login' to authenticate with GitHub CLI, or");
        console.log("   2. Set the MUFIN_PUSH_TOKEN environment variable");
        console.log("   3. Pass accessToken directly in AmplifyAppProps");
        return undefined;
    }
    getDefaultBuildSpec(platform) {
        if (platform === "WEB_COMPUTE") {
            return `version: 1
frontend:
  phases:
    preBuild:
      commands:
        - yarn install
    build:
      commands:
        - yarn run build
  artifacts:
    baseDirectory: .next
    files:
      - '**/*'
  cache:
    paths:
      - .next/cache/**/*
      - node_modules/**/*`;
        }
        return `version: 1
frontend:
  phases:
    preBuild:
      commands:
        - npm ci
    build:
      commands:
        - npm run build
  artifacts:
    baseDirectory: dist
    files:
      - '**/*'
  cache:
    paths:
      - node_modules/**/*`;
    }
}
exports.AmplifyApp = AmplifyApp;
_a = JSII_RTTI_SYMBOL_1;
AmplifyApp[_a] = { fqn: "must-cdk.AmplifyApp", version: "0.0.105" };
//# sourceMappingURL=data:application/json;base64,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