"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebSocketApiGatewayToLambda = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const apigatewayv2 = require("aws-cdk-lib/aws-apigatewayv2");
const apigatewayv2_integrations = require("aws-cdk-lib/aws-apigatewayv2-integrations");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const route53 = require("aws-cdk-lib/aws-route53");
const targets = require("aws-cdk-lib/aws-route53-targets");
const constructs_1 = require("constructs");
const utils_1 = require("./utils");
const tags_1 = require("../common/tags");
class WebSocketApiGatewayToLambda extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaFunction = props.lambdaFunction;
        // Apply tags to both construct and CloudFormation stack
        (0, tags_1.applyTags)(this, props.tags);
        // Create WebSocket API
        this.webSocketApi = new apigatewayv2.WebSocketApi(this, "WebSocketApi", {
            apiName: props.apiName,
            description: `WebSocket API for ${props.apiName}`,
            ...props.apiProps,
        });
        // Create default integration for the primary lambda
        const defaultIntegration = new apigatewayv2_integrations.WebSocketLambdaIntegration("DefaultIntegration", this.lambdaFunction);
        // Add default route
        this.webSocketApi.addRoute("$default", {
            integration: defaultIntegration,
        });
        // Add custom routes if provided
        if (props.customRoutes) {
            this.addCustomRoutes(props.customRoutes);
        }
        // Create stage
        const stageName = props.stageName || "dev";
        this.webSocketStage = new apigatewayv2.WebSocketStage(this, "WebSocketStage", {
            webSocketApi: this.webSocketApi,
            stageName: stageName,
            autoDeploy: true,
        });
        // Setup CloudWatch logging
        if (props.enableLogging) {
            const loggingResources = (0, utils_1.setupLogging)(this, props.apiName, props.logGroupProps);
            this.apiGatewayLogGroup = loggingResources.logGroup;
        }
        // Setup custom domain
        if (props.customDomainName) {
            const domainResources = this.setupCustomDomain(props);
            this.certificate = domainResources.certificate;
            this.domain = domainResources.domain;
            this.aRecord = domainResources.aRecord;
        }
    }
    /**
     * Add custom routes to the WebSocket API
     */
    addCustomRoutes(routes) {
        routes.forEach((route) => {
            const integration = new apigatewayv2_integrations.WebSocketLambdaIntegration(`${route.routeKey}Integration`, route.handler);
            this.webSocketApi.addRoute(route.routeKey, {
                integration: integration,
                returnResponse: route.routeResponseSelectionExpression !== undefined,
            });
        });
    }
    /**
     * Setup custom domain with certificate and Route53 record
     */
    setupCustomDomain(props) {
        let certificate;
        // Use existing certificate or create new one
        if (props.existingCertificate) {
            certificate = props.existingCertificate;
        }
        else if (props.hostedZone) {
            certificate = new acm.Certificate(this, "Certificate", {
                domainName: props.customDomainName,
                validation: acm.CertificateValidation.fromDns(props.hostedZone),
            });
        }
        else {
            throw new Error("Either certificateArn or hostedZone must be provided for custom domain");
        }
        // Create custom domain for WebSocket API
        const domain = new apigatewayv2.DomainName(this, "CustomDomain", {
            domainName: props.customDomainName,
            certificate: certificate,
        });
        // Create API mapping
        new apigatewayv2.ApiMapping(this, "ApiMapping", {
            api: this.webSocketApi,
            domainName: domain,
            stage: this.webSocketStage,
        });
        let aRecord;
        // Create Route53 alias record if hosted zone provided
        if (props.hostedZone) {
            aRecord = new route53.ARecord(this, "CustomDomainAliasRecord", {
                zone: props.hostedZone,
                recordName: props.customDomainName,
                target: route53.RecordTarget.fromAlias(new targets.ApiGatewayv2DomainProperties(domain.regionalDomainName, domain.regionalHostedZoneId)),
            });
        }
        return { certificate, domain, aRecord };
    }
    /**
     * Add a custom route after construction (for dynamic route addition)
     */
    addRoute(route) {
        const integration = new apigatewayv2_integrations.WebSocketLambdaIntegration(`${route.routeKey}Integration`, route.handler);
        return this.webSocketApi.addRoute(route.routeKey, {
            integration: integration,
            returnResponse: route.routeResponseSelectionExpression !== undefined,
        });
    }
    /**
     * Get the WebSocket API URL (useful for outputs)
     */
    get webSocketUrl() {
        if (this.domain) {
            return `wss://${this.domain.name}`;
        }
        return this.webSocketStage.url;
    }
}
exports.WebSocketApiGatewayToLambda = WebSocketApiGatewayToLambda;
_a = JSII_RTTI_SYMBOL_1;
WebSocketApiGatewayToLambda[_a] = { fqn: "must-cdk.WebSocketApiGatewayToLambda", version: "0.0.105" };
//# sourceMappingURL=data:application/json;base64,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