"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.amplifyCommand = void 0;
exports.amplifyInitCommand = amplifyInitCommand;
const fs = require("fs");
const path = require("path");
const commander_1 = require("commander");
/**
 * Create the amplify command with subcommands
 */
exports.amplifyCommand = new commander_1.Command("amplify")
    .description("Amplify-related commands")
    .addCommand(new commander_1.Command("init")
    .description("Initialize a new Amplify project with React application template")
    .option("-d, --directory <dir>", "Target directory", process.cwd())
    .action(amplifyInitCommand));
/**
 * Initialize a new Amplify project by copying the Next.js application template
 */
async function amplifyInitCommand(options) {
    try {
        const { directory } = options;
        // Ensure the directory exists
        if (!fs.existsSync(directory)) {
            console.error(`❌ Directory does not exist: ${directory}`);
            process.exit(1);
        }
        // Get the path to the amplify template
        const templatePath = getAmplifyExamplePath();
        if (!fs.existsSync(templatePath)) {
            console.error(`❌ Amplify template not found at: ${templatePath}`);
            console.error("   Make sure must-cdk is properly installed.");
            console.error("");
            console.error("💡 Troubleshooting:");
            console.error("   • If installed globally: npm install -g must-cdk@latest");
            console.error("   • If installed locally: npm install must-cdk");
            console.error("   • Try reinstalling the package");
            console.error("");
            console.error("📚 For more help, visit: https://github.com/globalmsq/must-cdk");
            process.exit(1);
        }
        console.log(`📁 Initializing Amplify React application in: ${directory}`);
        console.log(`📂 Source: ${templatePath}`);
        console.log("");
        // Copy all files and directories recursively
        copyDirectoryRecursive(templatePath, directory);
        console.log("");
        console.log(`🎉 Successfully initialized project!`);
        console.log("");
        console.log("📝 Files initialized:");
        console.log("   • README.md - Project documentation and setup instructions");
        console.log("   • src/ - React application source code");
        console.log("   • public/ - Static assets and files");
        console.log("   • amplify/ - Amplify configuration files");
        console.log("   • package.json & package-lock.json - Dependencies");
        console.log("");
        console.log("🚀 Next steps:");
        console.log("   1. Review the README.md for setup instructions");
        console.log("   2. Install dependencies: npm install");
        console.log("   3. Configure your Amplify app settings");
        console.log("   4. Deploy using Amplify console or CLI");
        console.log("");
        console.log("📚 For more information, visit:");
        console.log("   https://github.com/globalmsq/must-cdk");
    }
    catch (error) {
        console.error("❌ Error during Amplify initialization:", error);
        process.exit(1);
    }
}
/**
 * Copy a directory and its contents recursively
 */
function copyDirectoryRecursive(source, target) {
    let fileCount = 0;
    let dirCount = 0;
    function copyRecursive(src, dest) {
        const items = fs.readdirSync(src);
        for (const item of items) {
            const srcPath = path.join(src, item);
            const destPath = path.join(dest, item);
            const stat = fs.statSync(srcPath);
            if (stat.isDirectory()) {
                // Skip node_modules and other common directories that shouldn't be copied
                if (item === "node_modules" ||
                    item === ".git" ||
                    item === ".next" ||
                    item === "dist") {
                    continue;
                }
                // Create directory if it doesn't exist
                if (!fs.existsSync(destPath)) {
                    fs.mkdirSync(destPath, { recursive: true });
                    dirCount++;
                }
                // Recursively copy directory contents
                copyRecursive(srcPath, destPath);
            }
            else {
                // Skip hidden files and lock files that might cause issues
                if (item.startsWith(".") &&
                    item !== ".gitignore" &&
                    item !== ".env.example") {
                    continue;
                }
                // Check if file already exists
                if (fs.existsSync(destPath)) {
                    continue;
                }
                // Copy the file
                fs.copyFileSync(srcPath, destPath);
                fileCount++;
            }
        }
    }
    copyRecursive(source, target);
    return { files: fileCount, directories: dirCount };
}
/**
 * Get the path to the amplify template directory
 */
function getAmplifyExamplePath() {
    const currentDir = __dirname;
    // When running from compiled lib/ directory (development)
    let templatePath = path.resolve(currentDir, "../../../templates/amplify");
    if (fs.existsSync(templatePath)) {
        return templatePath;
    }
    // When running from source src/ directory (development)
    templatePath = path.resolve(currentDir, "../../templates/amplify");
    if (fs.existsSync(templatePath)) {
        return templatePath;
    }
    // When installed as global package - try to find the package installation directory
    try {
        // Try to resolve the package.json of must-cdk
        const packageJsonPath = require.resolve("must-cdk/package.json");
        const packageDir = path.dirname(packageJsonPath);
        templatePath = path.join(packageDir, "templates/amplify");
        if (fs.existsSync(templatePath)) {
            return templatePath;
        }
    }
    catch (error) {
        // If require.resolve fails, continue with other attempts
    }
    // Try relative to the CLI script location (for global installs)
    // Global npm packages are typically in: /usr/local/lib/node_modules/must-cdk/
    // or ~/.npm-global/lib/node_modules/must-cdk/
    const possibleGlobalPaths = [
        path.resolve(currentDir, "../../templates/amplify"), // lib/cli/commands -> templates/amplify
        path.resolve(currentDir, "../../../templates/amplify"), // if nested deeper
    ];
    for (const possiblePath of possibleGlobalPaths) {
        if (fs.existsSync(possiblePath)) {
            return possiblePath;
        }
    }
    // Last resort: try relative to process.cwd() (shouldn't normally be needed)
    templatePath = path.resolve(process.cwd(), "templates/amplify");
    return templatePath;
}
//# sourceMappingURL=data:application/json;base64,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