import * as acm from "aws-cdk-lib/aws-certificatemanager";
import * as cloudfront from "aws-cdk-lib/aws-cloudfront";
import * as origins from "aws-cdk-lib/aws-cloudfront-origins";
import * as route53 from "aws-cdk-lib/aws-route53";
import * as s3 from "aws-cdk-lib/aws-s3";
import { TaggableProps } from "../common/tags";
/**
 * S3 origin information
 */
export interface S3OriginInfo {
    readonly id: string;
    readonly bucket: s3.IBucket;
}
/**
 * HTTP origin information
 */
export interface HttpOriginInfo {
    readonly id: string;
    readonly origin: origins.HttpOrigin;
}
export interface S3OriginConfig {
    /**
     * Unique identifier for this S3 origin
     */
    readonly id: string;
    /**
     * Existing S3 bucket to use as origin (required)
     */
    readonly bucket: s3.IBucket;
    /**
     * Origin path for S3 requests (e.g., "/static")
     */
    readonly originPath?: string;
    /**
     * Use legacy Origin Access Identity instead of modern Origin Access Control
     * @default false - uses OAC for better security
     */
    readonly useLegacyOAI?: boolean;
    /**
     * Existing Origin Access Identity (only used if useLegacyOAI is true)
     */
    readonly originAccessIdentity?: cloudfront.OriginAccessIdentity;
    /**
     * Additional S3 origin properties
     */
    readonly s3OriginProps?: origins.S3OriginProps;
}
export interface HttpOriginConfig {
    /**
     * Unique identifier for this HTTP origin
     */
    readonly id: string;
    /**
     * Domain name of the HTTP origin (required)
     */
    readonly domainName: string;
    /**
     * Origin path for HTTP requests (e.g., "/api/v1")
     */
    readonly originPath?: string;
    /**
     * Protocol policy for the origin
     * @default HTTPS_ONLY
     */
    readonly protocolPolicy?: cloudfront.OriginProtocolPolicy;
    /**
     * HTTP port (for HTTP protocol)
     * @default 80
     */
    readonly httpPort?: number;
    /**
     * HTTPS port (for HTTPS protocol)
     * @default 443
     */
    readonly httpsPort?: number;
    /**
     * Additional HTTP origin properties
     */
    readonly httpOriginProps?: origins.HttpOriginProps;
}
export interface CacheBehaviorConfig {
    /**
     * Path pattern for this behavior (e.g., "/api/*", "*.jpg")
     */
    readonly pathPattern: string;
    /**
     * Origin ID to route this pattern to
     */
    readonly originId: string;
    /**
     * Viewer protocol policy
     * @default REDIRECT_TO_HTTPS
     */
    readonly viewerProtocolPolicy?: cloudfront.ViewerProtocolPolicy;
    /**
     * Allowed HTTP methods
     * @default ALLOW_GET_HEAD for S3, ALLOW_ALL for HTTP
     */
    readonly allowedMethods?: cloudfront.AllowedMethods;
    /**
     * Methods to cache
     * @default CACHE_GET_HEAD_OPTIONS
     */
    readonly cachedMethods?: cloudfront.CachedMethods;
    /**
     * Enable compression
     * @default true
     */
    readonly compress?: boolean;
    /**
     * Cache policy ID (use AWS managed policies)
     */
    readonly cachePolicyId?: string;
    /**
     * Origin request policy ID
     */
    readonly originRequestPolicyId?: string;
    /**
     * Response headers policy ID
     */
    readonly responseHeadersPolicyId?: string;
    /**
     * Cache policy (alternative to cachePolicyId)
     */
    readonly cachePolicy?: cloudfront.ICachePolicy;
    /**
     * Origin request policy (alternative to originRequestPolicyId)
     */
    readonly originRequestPolicy?: cloudfront.IOriginRequestPolicy;
    /**
     * Response headers policy (alternative to responseHeadersPolicyId)
     */
    readonly responseHeadersPolicy?: cloudfront.IResponseHeadersPolicy;
}
export interface CloudFrontToOriginsProps extends TaggableProps {
    /**
     * S3 origins configuration
     */
    readonly s3Origins?: S3OriginConfig[];
    /**
     * HTTP origins configuration
     */
    readonly httpOrigins?: HttpOriginConfig[];
    /**
     * ID of the origin to use as default behavior
     * If not specified, will use the first HTTP origin, then first S3 origin
     */
    readonly defaultOriginId?: string;
    /**
     * Cache behaviors for specific path patterns
     */
    readonly cacheBehaviors?: CacheBehaviorConfig[];
    /**
     * Primary custom domain name for the CloudFront distribution
     */
    readonly customDomainName?: string;
    /**
     * Additional domain names (aliases) for the distribution
     * Note: All domains must be covered by the same certificate.
     * CloudFront only supports one certificate per distribution.
     */
    readonly additionalDomainNames?: string[];
    /**
     * Route53 hosted zone for the custom domain
     * Required for creating Route53 records
     */
    readonly hostedZone?: route53.IHostedZone;
    /**
     * Existing ACM certificate to use for the CloudFront distribution
     * Certificate must be in us-east-1 region for CloudFront.
     * The certificate must cover all domains (customDomainName + additionalDomainNames).
     * CloudFront only supports one certificate per distribution.
     */
    readonly certificate?: acm.ICertificate;
    /**
     * Default root object for the distribution
     * @default "index.html"
     */
    readonly defaultRootObject?: string;
    /**
     * Enable CloudFront access logging
     * @default true
     */
    readonly enableLogging?: boolean;
    /**
     * Existing S3 bucket for logs
     * If not provided and logging is enabled, a new bucket will be created
     */
    readonly logBucket?: s3.IBucket;
    /**
     * Prefix for log files
     * @default "cloudfront"
     */
    readonly logPrefix?: string;
    /**
     * Include cookies in access logs
     * @default false
     */
    readonly logIncludeCookies?: boolean;
    /**
     * Custom error page configurations
     * If not provided, intelligent defaults will be applied based on origin types
     */
    readonly errorPages?: cloudfront.ErrorResponse[];
    /**
     * Geographic restriction configuration
     */
    readonly geoRestriction?: cloudfront.GeoRestriction;
    /**
     * Web Application Firewall (WAF) web ACL ID
     */
    readonly webAclId?: string;
    /**
     * CloudFront distribution price class
     * @default PRICE_CLASS_100
     */
    readonly priceClass?: cloudfront.PriceClass;
    /**
     * Whether the distribution is enabled
     * @default true
     */
    readonly enabled?: boolean;
    /**
     * Comment for the distribution
     */
    readonly comment?: string;
    /**
     * Create Route53 records for all domain names
     * @default true if hostedZone is provided
     */
    readonly createRoute53Records?: boolean;
    /**
     * Enable IPv6 for the distribution
     * @default false
     */
    readonly enableIpv6?: boolean;
    /**
     * HTTP version to support
     * @default HttpVersion.HTTP2
     */
    readonly httpVersion?: cloudfront.HttpVersion;
}
