"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getTagsFromEnvironment = getTagsFromEnvironment;
exports.mergeTags = mergeTags;
exports.applyTags = applyTags;
exports.formatTagsForCfn = formatTagsForCfn;
const aws_cdk_lib_1 = require("aws-cdk-lib");
/**
 * Reads tags from TAGS environment variable and applies them to all resources.
 */
/**
 * Parses the TAGS environment variable and returns tags as an object.
 * Expected format: key1=value1,key2=value2
 *
 * @returns Parsed tags from environment variable
 */
function getTagsFromEnvironment() {
    const tagsEnv = process.env.TAGS;
    if (!tagsEnv || tagsEnv.trim() === "") {
        return {};
    }
    const tags = {};
    try {
        // Split by comma and parse each key=value pair
        const tagPairs = tagsEnv.split(",");
        for (const pair of tagPairs) {
            const trimmedPair = pair.trim();
            if (trimmedPair === "")
                continue;
            const [key, ...valueParts] = trimmedPair.split("=");
            if (key && valueParts.length > 0) {
                // Join back in case value contains '=' characters
                const value = valueParts.join("=");
                tags[key.trim()] = value.trim();
            }
        }
    }
    catch (error) {
        console.warn(`Failed to parse TAGS environment variable: ${error}`);
        return {};
    }
    return tags;
}
/**
 * Merges user-provided tags with environment tags.
 * Environment tags take precedence over user tags.
 *
 * @param userTags - Optional user-provided tags
 * @returns Merged tags with environment tags taking precedence
 */
function mergeTags(userTags) {
    const envTags = getTagsFromEnvironment();
    return {
        ...userTags,
        ...envTags,
    };
}
/**
 * Applies tags to both the construct and the CloudFormation stack.
 * This ensures tags are visible at both the resource and stack level.
 *
 * @param construct - The construct to apply tags to
 * @param userTags - Optional user-provided tags
 * @returns Final merged tags that were applied
 */
function applyTags(construct, userTags) {
    const finalTags = mergeTags(userTags);
    // Apply tags to the construct (inherits to child resources)
    Object.entries(finalTags).forEach(([key, value]) => {
        aws_cdk_lib_1.Tags.of(construct).add(key, value);
    });
    // Apply tags to the CloudFormation stack
    const stack = aws_cdk_lib_1.Stack.of(construct);
    Object.entries(finalTags).forEach(([key, value]) => {
        aws_cdk_lib_1.Tags.of(stack).add(key, value);
    });
    return finalTags;
}
/**
 * Converts tags object to AWS CDK tag format for CfnResource.
 *
 * @param tags - Tags object
 * @returns Array of tag objects in CDK format
 */
function formatTagsForCfn(tags) {
    return Object.entries(tags).map(([key, value]) => ({ key, value }));
}
//# sourceMappingURL=data:application/json;base64,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