"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsCodeDeploy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const ecs_components_1 = require("./ecs-components");
const tags_1 = require("../common/tags");
/**
 * A CDK construct that creates an ECS Fargate service with CodeDeploy blue-green deployment capability.
 * This construct provides a modular approach to deploy containerized applications with blue-green deployment.
 */
class EcsCodeDeploy extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Apply tags to the construct
        (0, tags_1.applyTags)(this, props.tags);
        // Determine the target port
        const targetPort = props.targetPort ?? props.containers[0].containerPort;
        // 1. Create the ECS Service resources (service, task definition, containers)
        this._serviceBuilder = new ecs_components_1.EcsServiceComponent(this, "EcsService", {
            vpc: props.vpc,
            cluster: props.cluster,
            serviceName: props.serviceName,
            containers: props.containers,
            taskRole: props.taskRole,
            taskExecRole: props.taskExecRole,
            taskCPU: props.taskCPU,
            memoryLimit: props.memoryLimit,
            securityGroups: props.securityGroups,
            subnets: props.taskSubnets,
            assignPublicIp: props.assignPublicIp,
            autoScaling: props.autoScaling,
            tags: props.tags,
            desiredCount: props.desiredCount || 1,
            healthCheckGracePeriod: props.healthCheckGracePeriod || 300,
            deploymentController: {
                type: aws_cdk_lib_1.aws_ecs.DeploymentControllerType.CODE_DEPLOY,
            },
        });
        // 2. Create the Application Load Balancer resources (ALB, listeners, target groups)
        this._albBuilder = new ecs_components_1.LoadBalancerComponent(this, "LoadBalancer", {
            vpc: props.vpc,
            serviceName: props.serviceName,
            service: this._serviceBuilder.service,
            targetPort: targetPort,
            loadBalancerConfig: props.loadBalancer,
            subnets: props.loadBalancer.subnets,
            tags: props.tags,
        });
        // 3. Create the CodeDeploy resources (application, deployment group)
        this._deploymentBuilder = new ecs_components_1.CodeDeployComponent(this, "CodeDeploy", {
            appName: props.serviceName,
            service: this._serviceBuilder.service,
            productionListener: this._albBuilder.productionListener,
            testListener: this._albBuilder.testListener,
            productionTargetGroup: this._albBuilder.productionTargetGroup,
            testTargetGroup: this._albBuilder.testTargetGroup,
            config: props.codeDeployConfig,
            tags: props.tags,
        });
        // Expose the actual AWS resources as public properties for direct use
        this.service = this._serviceBuilder.service;
        this.loadBalancer = this._albBuilder.loadBalancer;
        this.taskDefinition = this._serviceBuilder.taskDef;
        this.productionTargetGroup = this._albBuilder.productionTargetGroup;
        this.testTargetGroup = this._albBuilder.testTargetGroup;
        this.application = this._deploymentBuilder.application;
        this.containers = this._serviceBuilder.containers;
    }
    /**
     * Get the production listener
     */
    get productionListener() {
        return this._albBuilder.productionListener;
    }
    /**
     * Get the test listener
     */
    get testListener() {
        return this._albBuilder.testListener;
    }
    /**
     * Get all listeners from the load balancer
     */
    get listeners() {
        return this.loadBalancer.listeners;
    }
    /**
     * Get the load balancer DNS name
     */
    get loadBalancerDnsName() {
        return this.loadBalancer.loadBalancerDnsName;
    }
    /**
     * Get the service ARN
     */
    get serviceArn() {
        return this.service.serviceArn;
    }
    // For backward compatibility with tests
    blueListener() {
        return this.productionListener;
    }
    greenListener() {
        return this.testListener;
    }
    allListeners() {
        return this.listeners;
    }
}
exports.EcsCodeDeploy = EcsCodeDeploy;
_a = JSII_RTTI_SYMBOL_1;
EcsCodeDeploy[_a] = { fqn: "must-cdk.EcsCodeDeploy", version: "0.0.105" };
//# sourceMappingURL=data:application/json;base64,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