"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeDeployComponent = exports.LoadBalancerComponent = exports.EcsServiceComponent = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const tags_1 = require("../common/tags");
/**
 * Creates and configures an ECS Fargate service with task definition and containers
 */
class EcsServiceComponent extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.containers = [];
        // Apply tags
        if (props.tags) {
            (0, tags_1.applyTags)(this, props.tags);
        }
        // Create log group
        this.logGroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, "LogGroup", {
            logGroupName: `/aws/ecs/${props.serviceName}`,
            retention: aws_cdk_lib_1.aws_logs.RetentionDays.INFINITE,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        // Create or use roles
        const taskRole = props.taskRole ??
            new aws_cdk_lib_1.aws_iam.Role(this, "TaskRole", {
                roleName: `${props.serviceName}-task-role`,
                assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal("ecs-tasks.amazonaws.com"),
                description: `Task role for ${props.serviceName}`,
            });
        const taskExecRole = props.taskExecRole ??
            new aws_cdk_lib_1.aws_iam.Role(this, "TaskExecRole", {
                roleName: `${props.serviceName}-task-exec-role`,
                assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal("ecs-tasks.amazonaws.com"),
                description: `Task execution role for ${props.serviceName}`,
                managedPolicies: [
                    aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName("service-role/AmazonECSTaskExecutionRolePolicy"),
                ],
            });
        this.taskRole = taskRole;
        this.taskExecutionRole = taskExecRole;
        // Create task definition
        this.taskDef = new aws_cdk_lib_1.aws_ecs.FargateTaskDefinition(this, "TaskDefinition", {
            memoryLimitMiB: props.memoryLimit ?? 2048,
            cpu: props.taskCPU ?? 1024,
            executionRole: this.taskExecutionRole,
            taskRole: this.taskRole,
        });
        // Add containers
        this.addContainers(props.containers);
        // Create ECS service
        this.service = new aws_cdk_lib_1.aws_ecs.FargateService(this, "Service", {
            cluster: props.cluster,
            serviceName: props.serviceName,
            taskDefinition: this.taskDef,
            assignPublicIp: props.assignPublicIp ?? false,
            desiredCount: props.desiredCount ?? 1,
            healthCheckGracePeriod: props.healthCheckGracePeriod
                ? aws_cdk_lib_1.Duration.seconds(props.healthCheckGracePeriod)
                : aws_cdk_lib_1.Duration.seconds(300),
            deploymentController: props.deploymentController ?? {
                type: aws_cdk_lib_1.aws_ecs.DeploymentControllerType.CODE_DEPLOY,
            },
            vpcSubnets: props.subnets,
            securityGroups: props.securityGroups,
        });
        // Set up auto-scaling if configured
        if (props.autoScaling) {
            this.setupAutoScaling(props.autoScaling);
        }
    }
    /**
     * Extract the repository name from an ECR image URI
     */
    getRepositoryNameFromImage(image) {
        if (image instanceof aws_cdk_lib_1.aws_ecs.EcrImage) {
            // For ECR images, try to extract the repository name from the image URI
            // This is a bit of a hack as CDK doesn't expose the repository name directly
            const imageStr = image.toString();
            // Try to extract repo name from something like:
            // ${Token[TOKEN.139]} (repository URL: 123456789012.dkr.ecr.us-east-1.amazonaws.com/my-repo), tag: latest
            const repoMatch = /repository URL: [^/]+\/([^)]+)\)/.exec(imageStr);
            if (repoMatch && repoMatch[1]) {
                // Return just the repository name part without tag
                const repoName = repoMatch[1].split(":")[0];
                return repoName;
            }
        }
        return undefined;
    }
    /**
     * Add containers to the task definition
     */
    addContainers(containers) {
        containers.forEach((container, index) => {
            // Use the repository name from the ECR image if available, otherwise use provided name or default
            const repoName = this.getRepositoryNameFromImage(container.image);
            const containerName = container.name ?? repoName ?? `Container${index}`;
            const containerDef = this.taskDef.addContainer(containerName, {
                image: container.image,
                memoryReservationMiB: container.memoryReservation ?? container.memoryLimit ?? 1024,
                logging: container.logging ??
                    aws_cdk_lib_1.aws_ecs.LogDriver.awsLogs({
                        streamPrefix: "ecs",
                        logGroup: this.logGroup,
                    }),
                portMappings: [{ containerPort: container.containerPort }],
                healthCheck: container.healthCheck ?? {
                    command: [
                        "CMD-SHELL",
                        `curl -f http://localhost:${container.containerPort} || exit 1`,
                    ],
                    interval: aws_cdk_lib_1.Duration.seconds(30),
                    timeout: aws_cdk_lib_1.Duration.seconds(5),
                },
                // Environment variables and secrets
                environment: container.environment,
                secrets: container.secrets,
                // Additional container configuration options
                cpu: container.cpu,
                essential: container.essential ?? true,
                workingDirectory: container.workingDirectory,
                user: container.user,
                entryPoint: container.entryPoint,
                command: container.command,
            });
            // Store container reference
            this.containers.push(containerDef);
        });
    }
    /**
     * Set up auto-scaling for the service
     */
    setupAutoScaling(autoScaling) {
        const scalableTarget = this.service.autoScaleTaskCount({
            minCapacity: autoScaling.minCapacity,
            maxCapacity: autoScaling.maxCapacity,
        });
        // Setup CPU-based scaling if configured
        if (autoScaling.cpuScale) {
            scalableTarget.scaleOnCpuUtilization("CpuScaling", autoScaling.cpuScale);
        }
        // Setup memory-based scaling if configured
        if (autoScaling.memoryScale) {
            scalableTarget.scaleOnMemoryUtilization("MemoryScaling", autoScaling.memoryScale);
        }
    }
}
exports.EcsServiceComponent = EcsServiceComponent;
/**
 * Creates and configures an Application Load Balancer with listeners and target groups
 */
class LoadBalancerComponent extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Apply tags
        if (props.tags) {
            (0, tags_1.applyTags)(this, props.tags);
        }
        const config = props.loadBalancerConfig || {};
        const useHttps = config.certificates && config.certificates.length > 0;
        // Use existing load balancer or create a new one
        if (config.existingLoadBalancer) {
            this.loadBalancer =
                config.existingLoadBalancer;
        }
        else {
            this.loadBalancer = new aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationLoadBalancer(this, "LoadBalancer", {
                vpc: props.vpc,
                loadBalancerName: `${props.serviceName}-alb`,
                internetFacing: config.internetFacing ?? false,
                vpcSubnets: config.subnets || props.subnets,
                securityGroup: config.securityGroups?.[0],
            });
            // Add HTTP to HTTPS redirect only if certificates are provided
            if (useHttps) {
                this.loadBalancer.addRedirect();
            }
        }
        // Determine protocol and ports
        const protocol = useHttps
            ? aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationProtocol.HTTPS
            : aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationProtocol.HTTP;
        const productionPort = config.productionPort ?? (useHttps ? 443 : 80);
        const testPort = config.testPort ?? 8080;
        // Create or use existing production listener
        try {
            // Try to find an existing listener at the production port
            this.productionListener = this.findOrCreateListener("ProductionListener", productionPort, protocol, config.certificates);
        }
        catch (error) {
            // Create a new listener if one doesn't exist or there was an error
            this.productionListener = this.loadBalancer.addListener("ProductionListener", {
                port: productionPort,
                protocol,
                ...(useHttps && { certificates: config.certificates }),
            });
        }
        // Create test listener
        this.testListener = this.loadBalancer.addListener("TestListener", {
            port: testPort,
            protocol,
            ...(useHttps && { certificates: config.certificates }),
        });
        // Create health check config
        const healthCheck = this.createHealthCheck(config.healthCheck);
        // Create target groups
        this.productionTargetGroup = this.productionListener.addTargets("ProductionTarget", {
            port: props.targetPort,
            protocol: aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationProtocol.HTTP,
            targets: [
                props.service,
            ],
            healthCheck,
            stickinessCookieDuration: aws_cdk_lib_1.Duration.days(1),
        });
        this.testTargetGroup = this.testListener.addTargets("TestTarget", {
            port: props.targetPort,
            protocol: aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationProtocol.HTTP,
            targets: [], // Empty for test deployment
            healthCheck,
            stickinessCookieDuration: aws_cdk_lib_1.Duration.days(1),
        });
    }
    /**
     * Find an existing listener or create a new one
     */
    findOrCreateListener(id, port, protocol, certificates) {
        // Check if a listener already exists at this port
        for (const listener of this.loadBalancer.listeners) {
            // This is a rough heuristic - in practice you'd want more reliable detection
            if (listener.toString().includes(`Port:${port}`)) {
                return listener;
            }
        }
        // Create a new listener if none exists
        return this.loadBalancer.addListener(id, {
            port,
            protocol,
            ...(certificates && certificates.length > 0 && { certificates }),
        });
    }
    /**
     * Create a health check configuration from the provided config or defaults
     */
    createHealthCheck(config) {
        return {
            path: config?.path ?? "/",
            port: config?.port ?? "traffic-port",
            interval: config?.interval ?? aws_cdk_lib_1.Duration.seconds(30),
            timeout: config?.timeout ?? aws_cdk_lib_1.Duration.seconds(10),
            healthyThresholdCount: config?.healthyThresholdCount ?? 2,
            unhealthyThresholdCount: config?.unhealthyThresholdCount ?? 5,
            healthyHttpCodes: config?.healthyHttpCodes,
        };
    }
}
exports.LoadBalancerComponent = LoadBalancerComponent;
/**
 * Creates and configures a CodeDeploy application and deployment group
 */
class CodeDeployComponent extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Apply tags
        if (props.tags) {
            (0, tags_1.applyTags)(this, props.tags);
        }
        // Create CodeDeploy application
        this.application = new aws_cdk_lib_1.aws_codedeploy.EcsApplication(this, "Application", {
            applicationName: props.appName,
        });
        // Create deployment group
        this.deploymentGroup = new aws_cdk_lib_1.aws_codedeploy.EcsDeploymentGroup(this, "DeploymentGroup", {
            application: this.application,
            service: props.service,
            deploymentGroupName: props.appName,
            deploymentConfig: this.getDeploymentConfig(props.config),
            blueGreenDeploymentConfig: {
                blueTargetGroup: props.productionTargetGroup,
                greenTargetGroup: props.testTargetGroup,
                listener: props.productionListener,
                testListener: props.testListener,
            },
        });
    }
    getDeploymentConfig(_config) {
        // Use custom deployment config or default to CodeDeployDefault.EcsAllAtOnce
        // In a more sophisticated implementation, you could create custom deployment configs
        // based on the provided parameters
        return aws_cdk_lib_1.aws_codedeploy.EcsDeploymentConfig.ALL_AT_ONCE;
    }
}
exports.CodeDeployComponent = CodeDeployComponent;
//# sourceMappingURL=data:application/json;base64,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