import { aws_ec2 as ec2, aws_ecs as ecs, aws_iam as iam, aws_elasticloadbalancingv2 as elbv2, Duration } from "aws-cdk-lib";
import { ICertificate } from "aws-cdk-lib/aws-certificatemanager";
import { TaggableProps } from "../common/tags";
/**
 * Configuration for the ECS Fargate task definition and container.
 */
export interface ContainerProps {
    /** Optional container name (default: Container{index}). */
    readonly name?: string;
    /** Hard memory limit in MiB for the task (default: 2048). */
    readonly memoryLimit?: number;
    /** Soft memory reservation in MiB for the container (default: 1024). */
    readonly memoryReservation?: number;
    /** Container image to deploy. */
    readonly image: ecs.ContainerImage;
    /** Optional container health check configuration. */
    readonly healthCheck?: ecs.HealthCheck;
    /** The port number the container listens on. */
    readonly containerPort: number;
    /** Environment variables to set in the container. */
    readonly environment?: {
        [key: string]: string;
    };
    /** Secrets to inject into the container from AWS Systems Manager Parameter Store or AWS Secrets Manager. */
    readonly secrets?: {
        [key: string]: ecs.Secret;
    };
    /** CPU units for this container (default: uses task-level CPU allocation). */
    readonly cpu?: number;
    /** Whether this container is essential (default: true). */
    readonly essential?: boolean;
    /** Working directory inside the container. */
    readonly workingDirectory?: string;
    /** User to run the container as. */
    readonly user?: string;
    /** Entry point for the container. */
    readonly entryPoint?: string[];
    /** Command to run in the container. */
    readonly command?: string[];
    /** Optional log driver configuration for the container. */
    readonly logging?: ecs.LogDriver;
}
/**
 * Configuration for ECS service auto-scaling.
 */
export interface AutoScalingProps {
    /** Minimum number of tasks to run. */
    readonly minCapacity: number;
    /** Maximum number of tasks to run. */
    readonly maxCapacity: number;
    /** Scale task based on CPU utilization. */
    readonly cpuScale?: ecs.CpuUtilizationScalingProps;
    /** Scale task based on memory utilization. */
    readonly memoryScale?: ecs.MemoryUtilizationScalingProps;
}
/**
 * Health check configuration for target groups
 */
export interface HealthCheckConfig {
    /** The path for health checks (default: "/") */
    readonly path?: string;
    /** The port for health checks (default: "traffic-port") */
    readonly port?: string;
    /** The interval between health checks (default: 30 seconds) */
    readonly interval?: Duration;
    /** The timeout for health checks (default: 5 seconds) */
    readonly timeout?: Duration;
    /** The number of successful health checks before considering healthy (default: 3) */
    readonly healthyThresholdCount?: number;
    /** The number of failed health checks before considering unhealthy (default: 3) */
    readonly unhealthyThresholdCount?: number;
    /** The HTTP codes to consider healthy (default: "200") */
    readonly healthyHttpCodes?: string;
}
/**
 * Configuration for the load balancer component
 */
export interface LoadBalancerConfig {
    /** Security groups for the ALB */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /** Select which subnets the ALB will be placed on */
    readonly subnets?: ec2.SubnetSelection;
    /** Whether the load balancer should be internet-facing (default: false). */
    readonly internetFacing?: boolean;
    /** Existing load balancer to use (if not provided, a new one will be created) */
    readonly existingLoadBalancer?: elbv2.IApplicationLoadBalancer;
    /** Optional ACM certificates for HTTPS termination. If not provided, HTTP listeners will be used. */
    readonly certificates?: ICertificate[];
    /** Health check configuration for target groups */
    readonly healthCheck?: HealthCheckConfig;
    /** The production listener port (default: 443 for HTTPS, 80 for HTTP) */
    readonly productionPort?: number;
    /** The test listener port (default: 8080) */
    readonly testPort?: number;
}
/**
 * Configuration for CodeDeploy deployment
 */
export interface CodeDeployConfig {
    /** The percentage of traffic to shift initially (default: 10%) */
    readonly trafficRoutingStartPercent?: number;
    /** The interval between traffic shifting increments (default: 1 minute) */
    readonly trafficRoutingInterval?: Duration;
    /** The termination wait time after deployment (default: 30 minutes) */
    readonly terminationWaitTime?: Duration;
}
/**
 * Properties for the EcsCodeDeploy construct.
 */
export interface EcsCodeDeployProps extends TaggableProps {
    /** VPC in which to deploy ECS and ALB resources. */
    readonly vpc: ec2.IVpc;
    /** Security group config for ECS service */
    readonly securityGroups: ec2.ISecurityGroup[];
    /** Select which subnets the ECS tasks will be placed on */
    readonly taskSubnets: ec2.SubnetSelection;
    /** ECS Cluster where the service will run. */
    readonly cluster: ecs.ICluster;
    /** Task role for the ECS task */
    readonly taskRole?: iam.IRole;
    /** Task execution role for the ECS task */
    readonly taskExecRole?: iam.IRole;
    /** Base name used for resources like log groups, roles, services, etc. */
    readonly serviceName: string;
    /** CPU units for the task (default: 1024). */
    readonly taskCPU?: number;
    /** Memory limit for the task in MiB (default: 2048). */
    readonly memoryLimit?: number;
    /** Configuration related to the task definition and container. */
    readonly containers: ContainerProps[];
    /** The port to expose on the target group (defaults to first container's port) */
    readonly targetPort?: number;
    /** Whether the Fargate tasks should be assigned public IP addresses (default: false).
     * This is required if your tasks need to access the internet and are in a public subnet.
     */
    readonly assignPublicIp?: boolean;
    /** Optional auto-scaling configuration. */
    readonly autoScaling?: AutoScalingProps;
    /** Load balancer configuration */
    readonly loadBalancer: LoadBalancerConfig;
    /** CodeDeploy configuration */
    readonly codeDeployConfig?: CodeDeployConfig;
    /** Desired count of tasks (default: 1) */
    readonly desiredCount?: number;
    /** Health check grace period in seconds (default: 300) */
    readonly healthCheckGracePeriod?: number;
}
