# Standard library imports
import os, sys         # File system and system-level operations
import itertools       # Advanced iteration tools
import re              # Regular expressions for text parsing
import shutil          # File operations like copy/move

# Visualization & array handling
import matplotlib.pyplot as plt  # For displaying images and plots
import numpy as np               # For numerical operations and array manipulation

# Image processing
from PIL import Image            # For opening and manipulating image files
import cv2                       # OpenCV for advanced image preprocessing

# Terminal output styling
from termcolor import colored    # For colored terminal messages

# OCR engine
from pytesseract import image_to_string  # Tesseract OCR for text extraction from images

# Import known OCR noise patterns from centralized database
from .OCR_NOISE_DB import (noisy_cases_for_1μπ, noisy_cases_for_2μπ, 
        noisy_cases_for_10μ, prefixes_for_10μ, noisy_cases_for_20μπ,
        noisy_cases_for_100μπ, noisy_cases_for_1mm,
        known_corrupted_substrings, known_noisy_prefixes)




# SEMOCR Class Initialization
class SEMOCR(object):
    def __init__(self, sem_pixel_size=None, Notaccess=None, THIN_THRESHOLD=10,
                refimage_path=None, user_patterns_path=None, user_words_path=None):

        """
        Initializes the SEMOCR class for OCR-based SEM image analysis.

        Parameters:
            sem_pixel_size (dict): Dictionary mapping image IDs to pixel sizes.
            Notaccess (list): List of image IDs that could not be accessed or processed.
            THIN_THRESHOLD (int): Threshold value used for scale detection logic.
            refimage_path (str): Path to the reference image for scale detection.
            user_patterns_path (str): Path to the Tesseract user patterns file.
            user_words_path (str): Path to the Tesseract user words file.
        """

        super(SEMOCR, self).__init__()

        # Threshold for detecting thin structures via OCR
        self.THIN_THRESHOLD = THIN_THRESHOLD

        # Dictionary mapping image IDs to pixel sizes
        self.sem_pixel_size = sem_pixel_size if sem_pixel_size is not None else {}

        # List of image IDs that couldn't be accessed or processed
        self.Notaccess = Notaccess if Notaccess is not None else []

        # Resolve the absolute path to the reference image inside sem_package/ref_scale
        module_dir = os.path.dirname(os.path.abspath(__file__))
        self.refimage = refimage_path or os.path.join(module_dir, "ref_scale", "scale_reference.jpg")

        # Build full paths to model files inside sem_package/model
        self.user_patterns_path = user_patterns_path or os.path.join(module_dir, "model", "ell.user-patterns")
        self.user_words_path = user_words_path or os.path.join(module_dir, "model", "ell.user-words")

        # Define OCR configs
        self.config_psm11 = f'--psm 11 --psm 12 --psm 6 --oem 3 \
                    --user-patterns {self.user_patterns_path} \
                    --user-words    {self.user_words_path} \
                    -c load_system_dawg=0 \
                    -c language_model_penalty_non_freq_dict_word=0.2 \
                    -c language_model_penalty_non_dict_word=0.3 \
                    -c tessedit_char_whitelist=0123456789\u03BC\u03C0'

        self.config_psm10 = f'--psm 10 --oem 3 \
                    --user-patterns {self.user_patterns_path} \
                    --user-words    {self.user_words_path} \
                    -c load_system_dawg=0 \
                    -c language_model_penalty_non_freq_dict_word=0.1 \
                    -c language_model_penalty_non_dict_word=0.15 \
                    -c tessedit_char_whitelist=0123456789\u03BC\u03C0'

        # Group configs into a dictionary for easy access
        self.ocr_configs = {"psm10": self.config_psm10, "psm11": self.config_psm11}                  
                    


    def ReadImage(self, image):

        """
        Loads a SEM image from disk in grayscale format.

        Parameters:
            image (str): Path to the image file.

        Returns:
            np.ndarray: Grayscale image array suitable for OCR and preprocessing.

        Raises:
            FileNotFoundError: If the image cannot be loaded from the given path.
        """

        # Read the image using OpenCV in grayscale mode (flag = 0)
        img = cv2.imread(image, 0)

        # Raise an error if the image couldn't be loaded
        if img is None:
            raise FileNotFoundError(f"Image not found or unreadable: {image}")
        return img



    # Apply Thresholding to SEM Image
    def GetThresh(self, img, thr, maxv, targ):

        """
        Applies thresholding to a grayscale SEM image.

        Parameters:
            img (np.ndarray): Grayscale image array.
            thr (int): Threshold value used to binarize the image.
            maxv (int): Maximum value to assign to pixels exceeding the threshold.
            targ (int): OpenCV thresholding type (e.g., cv2.THRESH_BINARY).

        Returns:
            np.ndarray: Thresholded binary image.
        """

        # Apply thresholding using OpenCV
        # Pixels > thr are set to maxv; others to 0 (or based on targ)
        _, thresh = cv2.threshold(img, thresh=thr, maxval=maxv, type=targ)
        return thresh



    # Detect Contours in Thresholded SEM Image
    def GetContour(self, thresh):
        """
        Detects contours in a thresholded SEM image.

        Parameters:
            thresh (np.ndarray): Binary image produced by thresholding.

        Returns:
            list: A list of contours, where each contour is an array of points.
        """

        # Find contours using OpenCV
        # RETR_TREE retrieves all contours and reconstructs the full hierarchy
        # CHAIN_APPROX_SIMPLE compresses horizontal, vertical, and diagonal segments
        contours, _ = cv2.findContours(thresh, mode=cv2.RETR_TREE, method=cv2.CHAIN_APPROX_SIMPLE)  
        return contours



    # Convert SEM Image to Grayscale
    def GetGrayScale(self, img):
        """
        Converts a color SEM image to grayscale.

        Parameters:
            img (np.ndarray): Input image in BGR format (as read by OpenCV).

        Returns:
            np.ndarray: Grayscale image array.
        """

        # Convert BGR image to grayscale using OpenCV
        gray_image = cv2.cvtColor(img, cv2.COLOR_BGR2GRAY)
        return gray_image



    # Detect Uniform Color Stripe in SEM Image
    def ColorDetector(self, img):

        """
        Detects a uniform color stripe in the SEM image by scanning small boxes
        and checking for zero variance. Used as a fallback when primary detection fails.
        
        Parameters:
            img (np.ndarray): Input image array (grayscale or color).

        Returns:
            int or list: Pixel value at the detected stripe location,
                         or an empty list if detection fails.
        """

        try:
            # Define box dimensions for scanning
            w_box = img.shape[1] // 32
            h_box = w_box

            # Generate grid coordinates for scanning
            X = np.arange(0, img.shape[1] - w_box, w_box)
            Y = np.arange(img.shape[0] // 2, img.shape[0] - h_box, h_box)

            # Scan each box and check for zero variance
            for x, y in itertools.product(X, Y):
                box = np.array(img[y:y + h_box, x:x + w_box]).reshape((h_box * w_box,))
                if np.cov(box) == 0:
                    return img[y, x]  # Return pixel value at stripe location

        except Exception as e:
            # Handle corrupted image or unexpected failure
            print("corrupted image stripe")
            print(colored(f"Error details: {e}", "yellow"))
            return None





    # Extract Info Bar from SEM Image
    def SelectInfoBar(self, img, contours, indx):

        """
        Extracts the info bar region from a SEM image using contour analysis.

        Parameters:
            img (np.ndarray): Grayscale SEM image.
            contours (list): List of contours detected in the image.
            indx (int): Divider used to crop the info bar width.

        Returns:
            str or list: Filename of the saved info bar image, or an empty list if not found.
        """
        
        info_bar = []  # Initialize early to avoid undefined reference

        try:
            for cnt in contours:
                # Get bounding box for each contour
                x_infobar, y_infobar, w_infobar, h_infobar = cv2.boundingRect(cnt)

                # Heuristic: Select contours likely to be the info bar
                if (w_infobar > img.shape[1]*0.25 and img.shape[0]*0.125 > h_infobar > img.shape[0]*0.01):                        

                    # Crop the region suspected to be the info bar
                    infobar_matrix = img[y_infobar:y_infobar + h_infobar,x_infobar:x_infobar + w_infobar//indx]                     

                    # Apply thresholding to clean the info bar
                    thresh_b = self.GetThresh(infobar_matrix, thr=200, maxv=255, targ=cv2.THRESH_BINARY)                                       

                    # Optional check for thin structures
                    if h_infobar > self.THIN_THRESHOLD or w_infobar > self.THIN_THRESHOLD:                    
                        pass  # Currently unused — placeholder for future logic

                    # Save the extracted info bar image
                    info_bar = 'myinfobar.jpg'
                    cv2.imwrite(info_bar, thresh_b)
            return info_bar
        
        except Exception as e:            
            # Handle unexpected errors
            print(colored('[No info bar is found] from SelectInfoBar', 'red'))
            print(colored(f"Error details: {e}", "yellow"))
            return info_bar




    def SearchInfobar(self, img, info_bar=None, retries=0, max_retries=3):

        """
        Attempts to locate the info bar in a SEM image using contour and color analysis.
        This function is triggered only if SelectInfoBar fails to detect the info bar.

        Parameters:
            img (np.ndarray): Grayscale SEM image.
            info_bar (any): Placeholder or previous result from SelectInfoBar.
            retries (int): Current retry count.
            max_retries (int): Maximum allowed retries.

        Returns:
            np.ndarray or list: Thresholded info bar image if found, or an empty list if not.
        """
        info_bar = []

        if img is None or not isinstance(img, np.ndarray):
            print(colored('[Invalid image input]', 'red'))
            return info_bar

        if retries > max_retries:
            print(colored('[Max retries reached] SearchInfobar aborted', 'red'))
            return info_bar

        try:
            # Apply binary thresholding to the image
            thresh = self.GetThresh(img, thr=254, maxv=255, targ=cv2.THRESH_BINARY)

            # Detect contours in the thresholded image
            contours = self.GetContour(thresh)

            if not contours:
                print(colored('[No contours found]', 'yellow'))
                return info_bar

            # Extract bounding boxes
            boxes = [cv2.boundingRect(cnt) for cnt in contours if cnt is not None]
            if not boxes:
                return info_bar

            # Identify widest contour
            w_max = max([w for _, _, w, _ in boxes])
            clean_infobar = None

            for x, y, w, h in boxes:
                if w == w_max:
                    infobar_matrix = img[y:y + h, x:x + w]
                    if infobar_matrix is not None and infobar_matrix.size > 0:
                        clean_infobar = self.GetThresh(infobar_matrix, thr=200, maxv=255, targ=cv2.THRESH_BINARY)
                    break

            # If multiple contours exist, apply color-based correction
            if len(contours) != 1:
                stripe_color = self.ColorDetector(img)
                if stripe_color is not None:
                    if stripe_color < 50:
                        # Invert image if stripe is dark
                        img = 255 - img
                    else:
                        # Mask stripe color and retry
                        img[img == stripe_color] = 255
                    return self.SearchInfobar(img, info_bar=None, retries=retries + 1)

            return clean_infobar if clean_infobar is not None else []

        except Exception as e:
            print(colored('[No info bar is found] from SearchInfobar', 'red'))
            print(colored(f"Error details: {e}", "yellow"))
            return info_bar




    # Extract Reference Contour from Scale SEM Image
    @property
    def ReferenceImage(self):

        """
        Processes a reference SEM image and extracts its primary contour.

        Parameters:
            refimage (str): Path to the reference image file.

        Returns:
            np.ndarray: The first contour detected in the thresholded reference image.
        """

        # Load the reference image in color
        ref_img = cv2.imread(self.refimage)
        if ref_img is None:
            raise FileNotFoundError(f"Reference image not found at: {self.refimage}")

        # Convert to grayscale for preprocessing
        gray_ref_img = self.GetGrayScale(ref_img)

        # Apply inverse binary + Otsu thresholding
        tresh_ref_img = self.GetThresh(gray_ref_img, thr=100, maxv=255, targ=cv2.THRESH_BINARY_INV+cv2.THRESH_OTSU)

        # Detect contours in the thresholded image
        cont_ref_img = self.GetContour(tresh_ref_img)

        # Select the first contour as the reference
        cnt_ref_img = cont_ref_img[0]
        return cnt_ref_img



    # Detect and Extract Scale Bar from SEM Info Bar
    def SelectScale(self, contours_img_info_bar, img_info_bar):

        """
        Detects and extracts the scale bar from the info bar region of a SEM image.

        Parameters:
            contours_img_info_bar (list): List of contours detected in the info bar region.
            img_info_bar (np.ndarray): Image matrix of the info bar.

         Returns:
            tuple:
                - info_bar_norectangle (str or None): Filename of the unannotated info bar image,
                or `None` if no valid scale bar was found.
                - semtext (str or None): Filename of the cropped text region,
                or `None` if extraction failed.
                - scale_width (int or None): Width of the extracted scale bar in pixels,
                or `None` if no scale bar was detected.
        """

        # Initialize bounding box and tracking variables
        max_bounding = (0, 0, 0, 0)
        max_area = 0
        cont_list, similarity_list, cont_area = [], [], []
        x_list, y_list, w_list, h_list = [], [], [], []

        # Load reference contour from known scale image
        cnt_ref_img = self.ReferenceImage

        # Analyze each contour in the info bar
        for cont in contours_img_info_bar:
            x_cont, y_cont, w_cont, h_cont = cv2.boundingRect(cont)

            # Filter out small contours that are unlikely to be scale bars
            if h_cont >= 25 or w_cont >= 25:
                areacont = cv2.contourArea(cont)
                is_same_shape = cv2.matchShapes(cnt_ref_img, cont, 1, 0.0)

                # Store metrics for later comparison
                similarity_list.append(is_same_shape)
                cont_list.append(cont)
                cont_area.append(areacont)
                w_list.append(w_cont)
                h_list.append(h_cont)
                x_list.append(x_cont)
                y_list.append(y_cont)

        # Fallback if no valid contours were collected
        if not cont_list:
            print(colored("[SelectScale] No valid contours found for scale bar.", "red"))
            return None, None, None

        # Identify best match: either largest area or closest shape similarity
        MinValue = min(similarity_list)
        MaxContArea = max(cont_area)

        for carea, similarity, w, h, x, y in zip(cont_area, similarity_list, w_list, h_list, x_list, y_list):
            if carea == MaxContArea or similarity == MinValue:
                area = w * h
                max_bounding = (x, y, w, h)
                max_area = area

                # Extract text and scale bar regions from the info bar 
                text_matrix = img_info_bar[0:max_area, 0:max_area]
                scale_matrix = img_info_bar[y:y + h, x:x + w - 2]
                scale_width = scale_matrix.shape[1]

                # Save cropped and annotated images only if scale bar is valid
                if scale_matrix.size > 0:
                    # Save cropped scale bar image
                    cv2.imwrite('seg_crop_1.jpg', scale_matrix)

                    # Save cropped text region
                    semtext = 'text.jpg'
                    cv2.imwrite(semtext, text_matrix)

                    # Save unannotated info bar
                    info_bar_norectangle = "info_bar_no_rectangle.jpg"
                    cv2.imwrite(info_bar_norectangle, img_info_bar)

                    # Draw rectangle around detected scale bar and save annotated image
                    cv2.rectangle(img_info_bar, (x, y), (x + w - 2, y + h), (0, 0, 255), 2)
                    barname = 'seg_cont_1.jpg'
                    cv2.imwrite(barname, img_info_bar)
                    return info_bar_norectangle, semtext, scale_width
                else:
                    # Warn if scale bar extraction failed due to empty matrix
                    print(colored("[SelectScale] Warning: scale_matrix is empty - skipping save.", "yellow"))
                    return None, None, None

 

    # Read Digit-Unit Label from SEM Text Region
    # Extract the raw digit-unit string (e.g., "200μ", "500π") from the OCR-processed SEM scale bar text
    def ReadDigitUnit(self, semtext): 

        """
        Extracts and normalizes digit-unit labels from a SEM text image using OCR.

        Parameters:
            semtext (str): Path to the image file containing the SEM scale text.

        Returns:
            str: A standardized digit-unit string (e.g., "1μπ", "200π", "10μ"), or "-1" if extraction fails.
        """

        try:   
            # Load and preprocess the image
            img = cv2.imread(semtext)
            gray = cv2.cvtColor(img, cv2.COLOR_BGR2GRAY) 
            gray = cv2.medianBlur(gray, 3) 
            _, textimage = cv2.threshold(gray, 128, 255, cv2.THRESH_BINARY | cv2.THRESH_OTSU)  

            digitunit = image_to_string(textimage, lang='eng+grc') 
            #print("digitunit BEFORE using dict",digitunit, "length: 1", len(digitunit))

            # Handle noisy OCR outputs for "1μπ" — common misreads and corrupted patterns
            if "1pm" in digitunit or "tum" in digitunit or "1am\"" in digitunit or "1 μπὶ" in digitunit:
                digitunit="1μπ"
                return digitunit

            # Handle noisy OCR outputs for "2μπ" — multiple variants and partial matches
            if "2ym" in digitunit or digitunit=="2pm" or "2 ΜΠῚ" in digitunit or digitunit.startswith("2m") or "2 um\"" in digitunit\
            or "2um" in digitunit or "2 um" in digitunit:            
                digitunit="2μπ"
                return digitunit

            # Handle noisy OCR outputs for "3μπ" — multiple variants and partial matches
            if digitunit=="3um" or "3pm EHT =" in digitunit or "3 um\"" in digitunit:
               digitunit="3μπ" 
               return digitunit

            # Handle noisy OCR outputs for "10μπ" - multiple variants and partial matches
            if "10 um" in digitunit or "10 pm" in digitunit or "10 ym" in digitunit or "10 μαὶ" in digitunit or "10 μαι" in digitunit\
            or "10 μην’" in digitunit or "10 μαι" in digitunit:
                digitunit="10μπ"
                return digitunit
            
            # Handle noisy OCR outputs for "10μπ" - multiple variants and partial matches    
            if "104m" in digitunit:
                 digitunit="10μπ"
                 return digitunit    
            
            # Handle noisy OCR outputs for "20 nm" - multiple variants and partial matches    
            if "201m EHT" in digitunit or "20m EHT" in digitunit or "20 nm’" in digitunit or "20 nm" in digitunit:
                digitunit="20π"
                return digitunit

            # Handle noisy OCR outputs for "20μπ" - multiple variants and partial matches
            if "20 um" in digitunit or "20 wm" in digitunit or "20 pm" in digitunit or "20 ym\"" in digitunit\
            or "20 ym" in digitunit:
                digitunit="20μπ"
                return digitunit 

            # Handle noisy OCR outputs for "30μπ" - multiple variants and partial matches
            if "30 um" in digitunit or "30um" in digitunit or "30 ym" in digitunit or "30pm" in digitunit or "30 pm" in digitunit:            
                digitunit="30μπ"
                return digitunit

            # Handle noisy OCR outputs for "100μπ" - multiple variants and partial matches
            if digitunit=="100 p" or digitunit=="100 pt" or "100 pm" in digitunit or "100 um" in digitunit or "100 ym" in digitunit\
            or "100 μπὶ" in digitunit or "100 pum" in digitunit or digitunit=="100 um’" or digitunit=="100 um’" or "100um" in digitunit:
                digitunit="100μπ"
                return digitunit 

            # Handle noisy OCR outputs for "200μ" - multiple variants and partial matches
            if "200 pm" in digitunit or "200 pm*" in digitunit or "200 um" in digitunit or "200 ym EHT" in digitunit or "200 bm" in digitunit\
            or "200 ym\"" in digitunit or "200 ym\"" in digitunit or "200 μητ'" in digitunit: 
                digitunit="200μ"
                return digitunit
            
            # Handle noisy OCR outputs for "300μπ" - multiple variants and partial matches    
            if "300 um" in digitunit or "300 ym" in digitunit:
                digitunit="300μπ"
                return digitunit
            
            # Handle noisy OCR outputs for "300nm" - multiple variants and partial matches    
            if "300 nm" in digitunit:
                digitunit="300ππ"
                return digitunit 
            
            # Handle noisy OCR outputs for "100nm" - multiple variants and partial matches     
            if "100 nm EHT" in digitunit or "100 nm’" in digitunit or "100 nm" in digitunit:
                digitunit="100π"
                return digitunit

            # Handle noisy OCR outputs for "200nm" - multiple variants and partial matches   
            if "200 nm" in digitunit or "200 nm EHT" in digitunit:
                digitunit="200π"
                return digitunit
            
            # Handle noisy OCR outputs for "30nm" - multiple variants and partial matches    
            if "30 nm" in digitunit or "30m EHT" in digitunit:
                 digitunit="30π"
                 return digitunit
                 
            # Handle noisy OCR outputs for "1mm" - multiple variants and partial matches    
            if "tem:" in digitunit or "1mm’" in digitunit or digitunit=="1mm":  
                digitunit="1mm" 
                return digitunit

            # Apply OCR with your custom-trained configuration
            else:
                textimage=Image.open(semtext)
                digitunit = image_to_string(textimage, lang='ell+grc', config=self.ocr_configs["psm11"]) 

                #print("From ReadDigitUnit function digitunit after using dict", digitunit)
                all_digitunit = re.sub(r"\W", "",digitunit)
                #print("From ReadDigitUnit function all_digitunit and digitunit after using dict",all_digitunit, digitunit, 
                #    "length: 1", len(all_digitunit), "length: 2", len(digitunit))

                
                # Handle extreme OCR noise cases that consistently map to "1μπ"         
                # Check against noisy list
                for i in noisy_cases_for_1μπ:
                    if all_digitunit == i:
                        digitunit = "1μπ"
                        return digitunit 


                # Handle extreme OCR noise cases that consistently map to "2μπ"
                # Check against noisy list
                for i in noisy_cases_for_2μπ:
                    if all_digitunit == i:
                        digitunit = "2μπ"
                        return digitunit    
                

                 # Handle extreme OCR noise cases that consistently map to "3μπ"   
                if all_digitunit=="μπ75π52241π80":
                    digitunit="3μπ"
                    return digitunit
                    
                
                # Handle extreme OCR noise cases that consistently map to "20μπ"
                if all_digitunit=="20π" and digitunit=="20π":                
                    digitunit="20μπ"
                    return digitunit
            
                                
                # Handle extreme OCR noise cases that consistently map to "20μπ"
                # Check against noisy list
                for i in noisy_cases_for_20μπ:
                    if all_digitunit == i:
                        digitunit = "20μπ"
                        return digitunit 

                    
                # Handle extreme OCR noise cases that consistently map to "30μπ"
                if all_digitunit =="3μπ" and digitunit =="3μπ":    
                    digitunit="30μπ"
                    return digitunit 
                    
                # Handle extreme OCR noise cases that consistently map to "30nm"    
                if all_digitunit=="1000ππ40ππ70801" or all_digitunit=="1000ππ40π111484":
                    digitunit="30ππ"
                    return digitunit 
                         
                # Handle extreme OCR noise cases that consistently map to "3μπ" 
                if all_digitunit =="μπ5" and "μπ" not in digitunit:
                    digitunit="3μπ" 
                    return digitunit 


                # Handle extreme OCR noise cases that consistently map to "100μπ"
                # Check against noisy list
                for i in noisy_cases_for_100μπ:
                    if all_digitunit == i:
                        digitunit = "100μπ"
                        return digitunit


                # Handle extreme OCR noise cases that consistently map to "200μπ"
                if "π1710417" in all_digitunit and "200π" in digitunit or all_digitunit.startswith("200π2001πππ"):
                    digitunit="200μ"
                    return digitunit

                
                # Special case: structured check for "10μπ" with exact length
                if "10μπ" in all_digitunit and "10μπ" in digitunit and len(all_digitunit) == 26:
                    digitunit = "10μ"
                    return digitunit


                # Handle noisy OCR outputs that start with known prefixes           
                for prefix in prefixes_for_10μ:
                    if all_digitunit.startswith(prefix):
                        digitunit = "10μ"
                        return digitunit

                
                # Handle extreme OCR noise cases that consistently map to "10μ"       
                # Check against noisy list
                for i in noisy_cases_for_10μ:
                    if all_digitunit == i:
                        digitunit = "10μ"
                        return digitunit


                # Handle rare OCR noise patterns that resolve to "100μ"
                if all_digitunit=="10μπ3014ππ6πππ498" or all_digitunit=="10μπ5004ππ64ππ8250" or all_digitunit=="00μπ600π522πππ08":
                    digitunit="100μ"
                    return digitunit
                    
                # Handle rare and highly corrupted OCR outputs that resolve to "100 nm"
                if all_digitunit=="10ππ5ππ86ππ28512π5" or all_digitunit=="6272012πππππ1415043530000" or all_digitunit=="00ππ100π5πμ6098217ππ":
                    digitunit="100ππ"
                    return digitunit

              
                # Handle extreme OCR noise cases that consistently map to "1mm"
                # Check against noisy list
                for i in noisy_cases_for_1mm:
                    if all_digitunit == i:
                        digitunit = "1mm"
                        return digitunit
                

                # Check for embedded numeric fragment "50181" with strict length constraints
                is_fragment_match = '50181' in digitunit and len(all_digitunit) == 23 and len(digitunit) == 29

                # Check for specific length combination known to produce corrupted outputs
                is_length_match = len(all_digitunit) == 26 and len(digitunit) == 32

                # Match known corrupted substrings from noisy SEM image batches
                is_known_corruption = any(sub in all_digitunit for sub in known_corrupted_substrings)
    
                # Detect semantic noise from misread UI overlays or metadata
                is_overlay_noise = "EHT" in digitunit and "WO" in digitunit

                # Match structured prefixes tied to specific SEM image patterns
                is_prefix_match = any(all_digitunit.startswith(p) for p in known_noisy_prefixes)    
                
                # Check for unit-like suffix pattern
                is_suffix_match = all_digitunit.endswith("π51")

                # Final condition: apply all heuristics only if digitunit is non-numeric and non-alpha
                if (digitunit and not digitunit.isdigit() and not digitunit.isalpha() and (is_fragment_match 
                    or is_length_match or is_known_corruption or is_overlay_noise or is_prefix_match or is_suffix_match)): 

    
                    # Extract digit-unit string using a specialized OCR configuration
                    # This block uses a different dictionary and character whitelist than the first OCR pass
                    digitunit = image_to_string(textimage, lang='ell+grc', config=self.ocr_configs["psm10"])

                    # Return the extracted digit-unit string for further processing
                    return digitunit

                # Final validation and correction for ambiguous "200π" cases
                # If the initial OCR result is "200π", we re-run OCR with a broader language model to verify its true meaning 
                if all_digitunit and digitunit =="200π":

                    # Re-run OCR using English + Greek to catch alternate unit representations like "um" or "nm"
                    digitunit=image_to_string(textimage, lang='eng+grc')  

                    # If OCR returns "200 um" or a misread variant like "200 ym", correct it to "200μ"
                    if digitunit=="200 um" or "200 ym" in digitunit:
                        digitunit="200μ"
                        return digitunit

                    # If OCR returns "200 nm" or variants with metadata noise (e.g., "EHT", "EH", "Er"), confirm "200π"
                    if digitunit=="200 nm" or "200 nm EHT" in digitunit or "200 nm EH" or "200 nm Er" in digitunit:
                        digitunit="200π"
                        return digitunit  

                # If no special handling is needed, return the digitunit as-is                     
                else:
                    return digitunit            
                
        # Catch-all exception handler for unexpected OCR failures 
        except:            
            error = sys.exc_info()[0]   # Capture the exception type for debugging (optional)
            digitunit = '-1'            # Use "-1" as a sentinel value for unrecoverable OCR errors
            print("rare case: after catching all possible exceptions:",digitunit)
            return digitunit 



    # Extract Numeric Value from Digit-Unit String
    # Isolate the first numeric sequence from OCR output, e.g., "200μ" or "500π"
    def GetDigit(self, digitunit):

        """
        Extracts the numeric portion from a digit-unit string using regular expression splitting.

        Parameters:
            digitunit (str): A string containing a numeric value followed by a unit (e.g., "200μ", "500π").

        Returns:
            str or list: The extracted numeric value as a string if found; otherwise, an empty list if no digits are present.
        """
        try:
            digit = re.split(r'(\d+)', digitunit)[1]
            return digit
        except IndexError:
            digit = []
            return digit



    # Extract scale bar region, interpret digit-unit label, and calculate pixel size in physical units
    def GetPixelSize(self, info_bar, contours):

        """
        Calculates the pixel size from a SEM image's info bar by extracting and interpreting the scale bar label.

        Parameters:
            info_bar (str): Path to the SEM info bar image containing the scale bar and text.
            contours (list): List of contours detected in the image, used to locate the scale bar region.

        Returns:
            dict or str: A dictionary containing the computed pixel size with its unit (e.g., {'Pixel size': '0.428 µm'}),
                         or a string flag ('-1', '-2') indicating unresolved or malformed OCR output.
        """

        # Initialize dictionary to store the final pixel size result
        pixel_size_dict = {}

        # Read the full SEM info bar image from disk
        img_info_bar = cv2.imread(info_bar)

        # Split the info bar into left and right halves for focused processing
        height, width, channels = img_info_bar.shape
        half_info_bar = width // 2
        left_part_info_bar = img_info_bar[:, :half_info_bar]
        right_part_info_bar = img_info_bar[:, half_info_bar:]

        # Convert the left half to grayscale for contour detection
        gray_img_info_bar = self.GetGrayScale(left_part_info_bar)

        # Apply thresholding to isolate high-contrast regions (e.g., scale bar)
        thresh_img_info_bar = self.GetThresh(gray_img_info_bar, thr=254, maxv=255, targ=cv2.THRESH_BINARY_INV + cv2.THRESH_OTSU)
        
        # Detect contours in the thresholded image
        contours_img_info_bar = self.GetContour(thresh_img_info_bar)

        # Select the scale bar region and extract its width and associated text
        info_bar_norectangle, semtext, scale_width = self.SelectScale(contours_img_info_bar, left_part_info_bar)
 

        # Read the digit-unit label from the extracted SEM text region
        digitunit = self.ReadDigitUnit(semtext)

        # If the label starts with a digit, attempt to parse the unit
        if digitunit[0].isdigit():
            unit = re.split(r'(\d+)', digitunit)[2]
            unit = re.sub(r"\W", "", unit)  # Remove non-alphanumeric characters

            # Handle rare case where unit is missing or malformed
            if not unit:
                unit = digitunit[5:]
                '\u03C0\u03C0' in unit  # Check for ππ pattern (Greek pi)
                if not unit:
                    digitunit = '-1'
                    return digitunit

            # If unit starts with a letter, proceed with pixel size calculation
            if unit[0].isalpha():
                digit = self.GetDigit(digitunit)

                # Special hardcoded case for corrupted OCR output
                if ("22201350182ππ" in digitunit and "π181810" in digitunit                    
                    and "π8820" in digitunit
                    and digit == "22201350182"):
            
                    pixel_size_dict['Pixel size'] = "0.428 µm"
                    return pixel_size_dict

                # Define known character patterns for nanometers and micrometers
                char_list_nano = ['\u03A0\u03A0', '\u03B7\u03C0', '\u03B7', '\u03B7\u03C0\u03B9', '\u03B7\u03C0\u03B7', '\u03B8']
                char_list_micro = ['\u03B8\u03BC\u03C0\u03B9', '.\u03BC\u03C0\u03B9', '\u03C1\u03B5', '\u03B1\u03BC\u03C0\u03B1', '\u03B1\u03BC\u03C0\u03B7']

                # Case: nanometer unit
                if digitunit[0].isdigit() and unit[0] == '\u03C0' or unit in char_list_nano:
                    scale_digit_unit = digit + ' nm'
                    pixel_div = int(digit) / scale_width
                    pixel_size_dict['Pixel size'] = str(pixel_div) + ' nm'
                    return pixel_size_dict

                # Case: micrometer unit
                if digitunit[0].isdigit() and unit[0] == '\u03BC' or unit in char_list_micro:
                    scale_digit_unit = digit + ' µm'
                    pixel_div = int(digit) / scale_width
                    pixel_size_dict['Pixel size'] = str(pixel_div) + ' µm'
                    return pixel_size_dict

                # Case: picometer unit
                if digitunit[0].isdigit() and unit[0] == 'pm':
                    scale_digit_unit = digit + ' pm'
                    pixel_div = int(digit) / scale_width
                    pixel_size_dict['Pixel size'] = str(pixel_div) + ' pm'
                    return pixel_size_dict

                # Case: millimeter unit or corrupted mm-like pattern
                if unit[0].isdigit() or unit == 'mm' or '\u03C0\u03C0\u03C0\u03C0' in re.sub(r"\W", "", digitunit):
                    scale_digit_unit = '1 mm'
                    pixel_div = int(digit) / scale_width
                    pixel_size_dict['Pixel size'] = str(pixel_div) + ' mm'
                    return pixel_size_dict

            # If unit is not alphabetic, mark as unresolved
            else:
                digitunit = '-1'
                return digitunit

        # Special case: unit-only label (e.g., "μπππ") or starts with a letter
        elif digitunit[0].isalpha() or digitunit == 'μπππ':
            scale_digit_unit = '1 µm'
            pixel_div = 1 / scale_width
            pixel_size_dict['Pixel size'] = str(pixel_div) + ' µm'
            return pixel_size_dict

        # Fallback: unrecognized or malformed digit-unit label
        else:
            digitunit = '-2'
            return digitunit



    # Reattempt scale bar detection and pixel size calculation when OCR returns unresolved flags (-1, -2)
    def Verify(self, img, contours, semimage):

        """
        Attempts to verify and recover pixel size information from a SEM image when initial OCR fails.

        Parameters:
            img (str): Path to the SEM image being processed.
            contours (list): List of contours detected in the image, used to locate the info bar.
            semimage (str): Identifier or filename of the SEM image, used for logging/debugging.

        Returns:
            dict or str: A dictionary containing the pixel size if successfully extracted,
                         or a string flag ('-1', '-2') indicating unresolved OCR output.
        """        
        try:
            # Start with the first index for selecting the info bar region
            indx = 1

            # Attempt to select the info bar from the image using contours
            info_bar = self.SelectInfoBar(img, contours, indx)

            # If no info bar is found, fallback to a default image and clean it
            if not info_bar:
                info_bar = 'myinfobar.jpg'
                clean_infobar = self.SearchInfobar(img, info_bar)

                if isinstance(clean_infobar, np.ndarray) and clean_infobar.size > 0:    
                    cv2.imwrite(info_bar, clean_infobar)
                else:
                    print(colored('[Invalid image] Cannot save info bar', 'red'))
                    return {}

            # If an info bar is available, attempt to extract pixel size
            if info_bar:
                digitunit = self.GetPixelSize(info_bar, contours)

                # If OCR returns unresolved flags, retry with different index values
                if digitunit == '-1' or digitunit == '-2':
                    for _ in np.arange(1, 9, 1):
                        indx += 1
                        info_bar = self.SelectInfoBar(img, contours, indx)
                        digitunit = self.GetPixelSize(info_bar, contours)

                        # If a valid result is found, return it immediately
                        if digitunit != '-1':
                            return digitunit

                    # If all retries fail, log the issue for review
                    if digitunit == '-1' or digitunit == '-2' or not digitunit:
                        print("text bug from ocr", semimage)

                # If OCR succeeded on the first try, return the result
                else:
                    pixel_size_dict = digitunit
                    return pixel_size_dict

        # Catch-all exception handler for unexpected failures        
        except Exception as e: 
            info_bar = []           
            pixel_size_dict = {}
            print(semimage, "has no info bar or hard to select it")
            print(colored(f"Error details: {e}", "yellow"))
            return pixel_size_dict



    # Execute OCR Pipeline on SEM Image
    # Process SEM image, extract scale bar, compute pixel size, and handle access failures
    def RunOCR(self, semimage):

        """
        Executes the full OCR pipeline on a SEM image to compute its pixel size.

        Parameters:
            semimage (str): Path to the SEM image file to be processed.

        Returns:
            tuple: A tuple containing:
                - comp_pixel_size (str): The computed pixel size (e.g., "0.428 µm") or "Not accessible" if failed.
                - self.Notaccess (list): A list of image filenames that failed OCR or pixel size extraction.
        """

        # Read the SEM image from disk
        img = self.ReadImage(semimage)

        # Apply binary thresholding to prepare for contour detection
        thresh = self.GetThresh(img, thr=254, maxv=255, targ=cv2.THRESH_BINARY)

        # Detect contours in the thresholded image
        contours = self.GetContour(thresh)

        # Attempt to verify and extract pixel size from the image
        pixel_size_dict = self.Verify(img, contours, semimage)

        # If pixel size is successfully computed and not corrupted, store it
        if pixel_size_dict and pixel_size_dict['Pixel size'] != '2775168772.75 nm':
            comp_pixel_size = [(k, v) for k, v in pixel_size_dict.items()][0][1]
            self.sem_pixel_size["ComputedPixelSize"] = 'Pixel Size = ' + comp_pixel_size

        # If pixel size is inaccessible or corrupted, log the failure
        else:
            comp_pixel_size = 'Not accessible'
            self.sem_pixel_size["ComputedPixelSize"] = 'Pixel Size = ' + comp_pixel_size
            self.Notaccess.append(semimage)

        # Print the final computed pixel size dictionary for review
        print("Computed pixel size", self.sem_pixel_size)

        # Return the result and list of inaccessible images
        return comp_pixel_size, self.Notaccess
