from typing import Any, Dict, List, Optional

from pipelex import log
from pipelex.cogt.image.prompt_image import PromptImage
from pipelex.cogt.llm.llm_job import LLMJob
from pipelex.cogt.llm.llm_job_components import LLMJobConfig, LLMJobParams
from pipelex.cogt.llm.llm_prompt import LLMPrompt
from pipelex.cogt.llm.llm_prompt_factory_abstract import LLMPromptFactoryAbstract
from pipelex.cogt.llm.llm_prompt_template import LLMPromptTemplate
from pipelex.config import get_config
from pipelex.pipeline.job_metadata import JobCategory, JobMetadata


class LLMJobFactory:
    # straightforward: the prompt is provided
    @classmethod
    def make_llm_job(
        cls,
        llm_prompt: LLMPrompt,
        llm_job_params: LLMJobParams,
        llm_job_config: Optional[LLMJobConfig] = None,
        job_metadata: Optional[JobMetadata] = None,
    ) -> LLMJob:
        config = get_config()
        llm_config = config.cogt.llm_config
        if job_metadata:
            job_metadata.update(
                JobMetadata(
                    job_category=JobCategory.LLM_JOB,
                )
            )
        else:
            job_metadata = JobMetadata(
                job_category=JobCategory.LLM_JOB,
            )
        job_params = llm_job_params
        job_config = llm_job_config or llm_config.llm_job_config

        return LLMJob(
            job_metadata=job_metadata,
            llm_prompt=llm_prompt,
            job_params=job_params,
            job_config=job_config,
        )

    # powerful: the prompt is generated by a prompt factory using prompt arguments
    # but this one is using the generic llm_prompt_factory.make_llm_prompt_from_args()
    @classmethod
    async def make_llm_job_from_prompt_factory(
        cls,
        llm_prompt_factory: LLMPromptFactoryAbstract,
        llm_prompt_arguments: Dict[str, Any],
        llm_job_params: LLMJobParams,
        llm_job_config: Optional[LLMJobConfig] = None,
        job_metadata: Optional[JobMetadata] = None,
    ) -> LLMJob:
        log.debug(llm_prompt_arguments, title="llm_prompt_arguments")
        llm_prompt = await llm_prompt_factory.make_llm_prompt_from_args(**llm_prompt_arguments)

        return cls.make_llm_job(
            job_metadata=job_metadata,
            llm_prompt=llm_prompt,
            llm_job_params=llm_job_params,
            llm_job_config=llm_job_config,
        )

    # powerful: the prompt is generated by a prompt template using prompt arguments
    @classmethod
    async def make_llm_job_from_prompt_template(
        cls,
        llm_prompt_template: LLMPromptTemplate,
        llm_prompt_arguments: Dict[str, Any],
        llm_job_params: LLMJobParams,
        llm_job_config: Optional[LLMJobConfig] = None,
        job_metadata: Optional[JobMetadata] = None,
    ) -> LLMJob:
        log.debug(llm_prompt_arguments, title="llm_prompt_arguments")
        llm_prompt = await llm_prompt_template.make_llm_prompt_from_args(**llm_prompt_arguments)

        return cls.make_llm_job(
            job_metadata=job_metadata,
            llm_prompt=llm_prompt,
            llm_job_params=llm_job_params,
            llm_job_config=llm_job_config,
        )

    @classmethod
    def make_llm_job_from_prompt_contents(
        cls,
        llm_job_params: LLMJobParams,
        user_text: Optional[str] = None,
        system_text: Optional[str] = None,
        user_images: Optional[List[PromptImage]] = None,
        llm_job_config: Optional[LLMJobConfig] = None,
        job_metadata: Optional[JobMetadata] = None,
    ) -> LLMJob:
        config = get_config()
        llm_config = config.cogt.llm_config
        if job_metadata:
            job_metadata.update(
                JobMetadata(
                    job_category=JobCategory.LLM_JOB,
                )
            )
        else:
            job_metadata = JobMetadata(
                job_category=JobCategory.LLM_JOB,
            )
        llm_prompt = LLMPrompt(
            system_text=system_text,
            user_text=user_text,
            user_images=user_images or [],
        )
        job_params = llm_job_params
        job_config = llm_job_config or llm_config.llm_job_config

        return LLMJob(
            job_metadata=job_metadata,
            llm_prompt=llm_prompt,
            job_params=job_params,
            job_config=job_config,
        )
