"""
This module exposes exceptions used by refinery.
"""
from __future__ import annotations

from typing import Collection


class RefineryImportMissing(ModuleNotFoundError):
    """
    A special variant of the `ModuleNotFoundError` exception which is raised when a dependency of a
    refinery unit is not installed in the current environment. The exception also provides hints
    about what package has to be installed in order to make that module available.
    """
    def __init__(self, missing: str, dependencies: Collection[str] = (), more: str | None = None):
        super().__init__()
        import shlex
        self.missing = missing
        self.install = ' '.join(shlex.quote(dist) for dist in dependencies)
        self.more = more
        self.dependencies = dependencies or (missing,)

    def __repr__(self):
        return F'{self.__class__.__name__}({self.missing!r})'


class RefineryCriticalException(RuntimeError):
    """
    If this exception is thrown, processing of the entire input stream
    is aborted instead of just aborting the processing of the current
    chunk.
    """
    pass


class RefineryPotentialUserError(RuntimeError):
    """
    This exception can be raised by a unit to inform the user about a
    suspected input error.
    """
    pass


class RefineryException(RuntimeError):
    """
    This is an exception that was not generated by an external library.
    """
    pass
