from __future__ import annotations

from refinery.units import Unit
from refinery.lib.structures import MemoryFile


class qr(Unit):
    """
    Extract information from bar codes, especially QR codes. This unit is a thin proxy around the
    pyzbar library, which itself only provides Python bindings for the ZBar library.
    """
    @Unit.Requires('pyzbar', ['formats', 'extended', 'all'],
        more='you also have to install the zbar shared library separately')
    def _pyzbar():
        import pyzbar
        import pyzbar.pyzbar
        return pyzbar

    @Unit.Requires('Pillow', ['formats', 'extended', 'all'])
    def _image():
        from PIL import Image
        return Image

    def process(self, data):
        img = self._image.open(MemoryFile(data, output=bytes))
        bar = self._pyzbar.pyzbar.decode(img)
        for data in bar:
            self.log_debug(data)
            if not (data := getattr(data, 'data', None)):
                continue
            if isinstance(data, str):
                data = data.encode(self.codec)
            if isinstance(data, (bytes, bytearray)):
                yield data
                continue
            self.log_warn(
                F'skipping unknown data generated by zbar: {data!r}', clip=True)
