"""Fish shell completion handler."""

from typing import List

from .base import CompletionContext, CompletionHandler


class FishCompletionHandler(CompletionHandler):
  """Fish-specific completion handler."""

  def generate_script(self, prog_name: str) -> str:
    """Generate fish completion script."""
    script = f'''# Fish completion for {prog_name}
# Generated by freyja

function __{prog_name}_complete
    # Set up completion environment
    set -x _FREYA_COMPLETE fish
    set -x COMP_WORDS_STR (commandline -cp)
    set -x COMP_CWORD_NUM (count (commandline -cp))

    # Get completions from the program
    {prog_name} --_complete 2>/dev/null
end

# Register completions for {prog_name}
complete -f -c {prog_name} -a '(__{prog_name}_complete)'

# Add option completions
complete -c {prog_name} -l help -d "Show help message"
complete -c {prog_name} -l verbose -d "Enable verbose output"  
complete -c {prog_name} -l no-color -d "Disable colored output"
complete -c {prog_name} -l install-completion -d "Install shell completion"
'''
    return script

  def get_completions(self, context: CompletionContext) -> List[str]:
    """Get fish-specific completions."""
    # Reuse bash completion logic for now
    from .bash import BashCompletionHandler
    bash_handler = BashCompletionHandler(self.cli)
    return bash_handler._get_generic_completions(context)

  def install_completion(self, prog_name: str) -> bool:
    """Install fish completion."""
    from .installer import CompletionInstaller
    installer = CompletionInstaller(self, prog_name)
    return installer.install('fish')
