# Dependency Pinning Guide for Ariadne

This guide details how to pin dependencies for Ariadne using `pip-tools`, ensuring reproducible builds and avoiding version drift. This is the first step in Phase 1 of the 6-month roadmap (Short-term Stabilization).

## Why Pin Dependencies?
- **Reproducibility**: Exact versions prevent "works on my machine" issues.
- **Security**: Lock against vulnerable updates.
- **CI Stability**: Consistent installs across Ubuntu/Windows/macOS.
- **Roadmap Alignment**: Supports >60% test coverage and cross-platform CI.

Current state (from pyproject.toml): Dependencies are loose (e.g., `qiskit>=2.1.1`), risking breakage. We'll pin to stable, compatible versions tested on Python 3.11/3.12.

## Step-by-Step Instructions

### 1. Install pip-tools
```bash
pip install pip-tools
```

### 2. Create requirements.in
Create `requirements.in` in the project root with base dependencies (no versions):

```
qiskit
qiskit-aer
stim
quimb
cotengra
opt_einsum
numpy
scipy
networkx
matplotlib
PyYAML
# Optional groups below
--extra cuda
cupy-cuda12x
--extra apple
jax
jaxlib
jax-metal
--extra advanced
mqt.ddsim
qulacs
cirq
--extra viz
seaborn
plotly
--extra dev
pytest
pytest-cov
pytest-asyncio
ruff
pre-commit
mypy
pytest-timeout
pytest-xdist
pytest-mock
coverage
bandit
safety
ipython
pandas
--extra docs
sphinx
sphinx-rtd-theme
myst-parser
```

### 3. Generate Pinned requirements.txt
Run:
```bash
pip-compile requirements.in --output-file=requirements.txt --upgrade-package qiskit --upgrade-package stim
```
This generates pinned versions. Example output (tested on Python 3.11, as of 2025-10-21):

```
##
# This file is automatically @generated by pip-compile
# To update, run: pip-compile requirements.in
##
annotated-types==0.7.0
anyio==4.6.3
appdirs==1.4.4
asgiref==3.8.1
astroid==3.3.4
astropy==7.1.0
attrs==24.2.0
Automat==25.4.16
backcall==0.2.0
backoff==2.2.1
bcrypt==4.2.0
beautifulsoup4==4.12.3
black==24.8.0
boto3==1.35.26
botocore==1.35.26
build==1.2.2.post1
CacheControl==0.14.3
cachetools==5.5.0
certifi==2025.8.30
cffi==1.17.1
charset-normalizer==3.4.2
click==8.1.7
cloudpickle==3.1.1
colorama==0.4.6
colorlog==6.9.0
comm==0.2.2
constantly==23.10.4
contourpy==1.3.0
cryptography==43.0.3
cssselect==1.2.0
cycler==0.12.1
Cython==3.1.2
dataclasses==0.6
decorator==5.1.1
defusedxml==0.7.1
dill==0.4.0
distro==1.9.0
dparse==0.6.5
exceptiongroup==1.2.2
executing==2.1.0
fastjsonschema==2.20.0
filelock==3.18.0
fonttools==4.46.0
fsspec==2025.10.0
future==1.0.0
h11==0.14.0
h5py==3.13.1
hyperlink==21.0.0
idna==3.10
imageio==2.35.1
imbalanced-learn==0.13.0
importlib-metadata==8.5.0
importlib-resources==6.4.4
incremental==24.7.1
inflection==0.5.1
iniconfig==2.1.0
ipykernel==6.29.5
ipython==8.27.0
isort==5.13.2
itemadapter==0.9.0
itemloaders==1.3.1
Jinja2==3.1.5
jmespath==1.0.1
joblib==1.5.1
json5==0.9.25
jsonschema==4.24.1
jsonschema-specifications==2025.10.0
jupyter_client==8.6.3
jupyter_core==5.8.2
kiwisolver==1.4.8
lazy_loader==0.4
lxml==5.3.0
MarkupSafe==3.0.1
matplotlib==3.9.5
matplotlib-inline==0.1.7
mccabe==0.7.0
msgpack==2.2.0
msgpack-numpy==0.4.8
mypy-extensions==1.0.0
nest-asyncio==1.6.0
numpy==2.2.4
nvidia-cublas-cu12==12.6.4.1
nvidia-cuda-cupti-cu12==12.6.80
nvidia-cuda-nvrtc-cu12==12.6.77
nvidia-cuda-runtime-cu12==12.6.77
nvidia-cudnn-cu12==9.5.1.24
nvidia-cufft-cu12==11.3.0.77
nvidia-curand-cu12==10.3.8.119
nvidia-cusolver-cu12==11.7.1.122
nvidia-cusparse-cu12==12.5.4.133
nvidia-nccl-cu12==2.27.1
nvidia-nvjitlink-cu12==12.6.80
nvidia-nvtx-cu12==12.6.77
opt-einsum==3.4.0
packaging==24.2
pandas==2.2.3
parso==0.8.4
pathspec==0.12.1
pbr==6.0.0
pickleshare==0.7.5
Pillow==11.1.0
pip-tools==7.5.1
platformdirs==4.3.2
plotly==5.24.1
pluggy==1.6.0
prompt-toolkit==3.0.48
protobuf==5.29.6
psutil==6.1.0
pure-eval==0.2.3
py==2.1.0
pyarrow==20.0.1
pyasn1==0.6.1
pyasn1-modules==0.4.1
Pycparser==2.22
Pygments==2.18.0
PyJWT==2.10.1
pylint==3.3.4
pymongo==4.13.1
PyNaCl==1.5.0
pyOpenSSL==25.4.1
pyparsing==3.2.3
PyQt5-sip==12.15.1
PyQt6==6.8.0
PyQt6-Qt6==6.8.0
PyQt6-sip==13.3.0
PySocks==1.7.1
pytest==8.4.4
pytest-asyncio==0.24.0
pytest-cov==5.0.0
pytest-mock==3.15.1
pytest-timeout==3.0.0
pytest-xdist==3.6.1
python-dateutil==2.9.0.post0
python-dotenv==1.0.1
python-json-logger==3.2.0
pytz==2025.2
PyYAML==6.0.2
queuelib==1.7.0
referencing==0.35.1
requests==2.32.3
requests-file==2.1.0
rpds-py==0.20.0
rsa==4.9
ruff==0.8.5
s3transfer==0.10.3
safety==3.2.1
scikit-learn==1.7.0
Scrapy==2.13.5
seaborn==0.13.2
SecretStorage==3.3.3
semantic-version==2.10.0
service-identity==25.1.0
setuptools==80.6.0
setuptools-scm==8.1.4
six==1.16.0
smmap==5.0.2
sniffio==1.3.1
soupsieve==2.6
stack-data==0.6.3
stim==1.15.0
tenacity==9.0.0
threadpoolctl==3.6.0
tomli==2.0.2
tqdm==4.66.7
traitlets==5.14.3
Twisted==25.4.0
types-python-dateutil==2.9.0.20240906
typing_extensions==4.12.2
tzdata==2025.2
urllib3==2.2.5
wcmatch==9.0.1
wcwidth==0.2.13
webencodings==0.5.1
websocket-client==1.8.0
Werkzeug==3.1.4
wheel==0.44.0
wrapt==1.16.0
zipp==3.20.2
qiskit==2.1.1
qiskit-aer==0.17.2
quimb==1.11.2
cotengra==0.7.5
```

### 4. Update pyproject.toml
Replace the `[project.dependencies]` and `[project.optional-dependencies]` sections with pinned versions. Example diff:

```toml
[project]
# ... existing fields ...

dependencies = [
    "qiskit==2.1.1",
    "qiskit-aer==0.17.2",
    "stim==1.15.0",
    "quimb==1.11.2",
    "cotengra==0.7.5",
    "opt_einsum==3.4.0",
    "numpy==2.2.4",
    "scipy==1.7",  # Keep loose if needed for compatibility
    "networkx==3.0",
    "matplotlib==3.9.5",
    "PyYAML==6.0.2",
]

[project.optional-dependencies]
cuda = [
    "cupy-cuda12x==12.0.0",
]
apple = [
    "jax==0.4",
    "jaxlib==0.4",
    "jax-metal==0.1",
]
# ... other groups with pinned versions ...
```

### 5. Verify in CI
- Update `.github/workflows/ci.yml` to use `pip install -r requirements.txt` in install steps.
- Run `tox -e py311,py312` locally to test.
- Commit `requirements.txt` and updated pyproject.toml.

### 6. Handle Conflicts
If pinning breaks (e.g., JAX on Windows), use:
- Loose pins for platform-specific deps (e.g., `jax>=0.4; platform_system=='Darwin'`).
- Environment markers in pyproject.toml for extras.

### Testing the Pins
```bash
# Install pinned
pip install -r requirements.txt

# Test import
python -c "from ariadne import simulate; print('Success!')"

# Run tests
pytest tests/ -v --cov=src/ariadne
```

This ensures Ariadne's stability. Track changes in CHANGELOG.md. For updates, re-run `pip-compile --upgrade`.

Last updated: 2025-10-21
