from pathlib import Path
import typer
from rich import print
import os
import sys
import subprocess

packageFileName = "requirements.txt"
preset = "windows" if os.name == "nt" else "default"
debugPreset="debug-windows" if os.name=="nt" else "debug"
execulableExtention=".exe" if os.name =="nt" else ""
app = typer.Typer(no_args_is_help=True)
buildType="Release"

@app.command()
def about():
    """Show information about the Sage project and CLI."""

    print("[bold cyan]Sage by Viuv Labs[/bold cyan]: A unified build and dependency system for C/C++")
    print("[faint white]Use '--help' to explore commands[/faint white]")

def _change_directory(path: str):
    if path:
        try:
            os.chdir(path)
        except FileNotFoundError:
            print(f"[bold red]Directory not found: {path}[/bold red]")
            raise typer.Exit()

@app.command()
def compile(path: str = typer.Option(None, "--path", "-p", help="Path to the project directory.")):
    """Compile the project using the current preset and build directory."""
    _change_directory(path)
    onCompile()

def onCompile():
    if not os.path.isfile("CMakeLists.txt"):
        print("[bold red]Missing CMakeLists.txt[/bold red]")
        return
    
    build_dir = f"build/{preset}"
    if not os.path.isdir(build_dir):
        runInstall()
        result = subprocess.run(["cmake", "--preset", preset])
        if result.returncode != 0:
            print("[bold red]CMake configuration failed[/bold red]")
            return
    
    subprocess.run(["cmake", "--build", build_dir, "--parallel"])

@app.command()
def run(args: list[str], path: str = typer.Option(None, "--path", "-p", help="Path to the project directory.")):
    """
    Run one or more compiled executables from the build directory.

    Arguments should match target names generated by the build.
    Example:
        sage run main test_game
    Searches inside build/{preset}/ for matching executables.
    """
    _change_directory(path)
    onRun(args)

def onRun(args:list[str]=[]):
    if not args:
        print("[bold red]Specify at least one app to build/run[/bold red]")
        return

    for arg in args:
        path_direct = f"build/{preset}/{arg}{execulableExtention}"
        path_nested = (f"build/{preset}/{arg}/{arg}{execulableExtention}")
        if os.path.isfile(path_direct):
            subprocess.run([path_direct])
        elif os.path.isfile(path_nested):
            subprocess.run([path_nested])
        else:
            print(f"[bold red]Executable '{arg}' not found in {path_direct} or {path_nested}[/bold red]")

@app.command()
def build(args:list[str], path: str = typer.Option(None, "--path", "-p", help="Path to the project directory.")):
    """
    Build and Run one or more compiled executables from the build directory.

    Arguments should match target names generated by the build.
    Example:
        sage build main test_game
    Searches inside build/{preset}/ for matching executables.
    """
    _change_directory(path)
    onCompile()
    onRun(args=args)

@app.command()
def install(
    package: str = typer.Option(None, "--package", "-p", help="Package name to install"),
    version: str = typer.Option(None, "--version", "-v", help="Optional package version"),
    path: str = typer.Option(None, "--path", help="Path to the project directory.")
):
    """Install dependencies from requirements.txt or add new ones , sage install --help"""
    _change_directory(path)
    runInstall(package,version)


@app.command()
def doctor():
    """
    Run diagnostics and automatically configure your development environment.

    This command will:
    - Detect missing tools (e.g. compiler, CMake, Conan, Ninja) and install them if necessary.
    - Validate your toolchain setup across platforms.
    - Configure a working Conan profile tailored for Clang with sane defaults.
    - Inspect CMake presets and verify project integrity.
    - Ensure compatibility for cross-platform development with static linking options.

    [💡] Future versions will include environment summary, version checks,
    and auto-fix suggestions for broken configs.
    """
    tools = {
        "cmake": {"version_cmd": ["cmake", "--version"], "install": {
            "Windows": "winget install Kitware.CMake",
            "Linux": "sudo apt-get install cmake",
            "Darwin": "brew install cmake"
        }},
        "ninja": {"version_cmd": ["ninja", "--version"], "install": {
            "Windows": "winget install Ninja-build.Ninja",
            "Linux": "sudo apt-get install ninja-build",
            "Darwin": "brew install ninja"
        }},
        "clang": {"version_cmd": ["clang", "--version"], "install": {
            "Windows": "winget install LLVM.LLVM",
            "Linux": "sudo apt-get install clang",
            "Darwin": "brew install llvm"
        }},
        "conan": {"version_cmd": ["conan", "--version"], "install": {
            "Windows": "pip install conan",
            "Linux": "pip install conan",
            "Darwin": "pip install conan"
        }},
    }

    if os.name == "nt":
        tools["msvc"] = {"version_cmd": ["vswhere", "-latest", "-property", "displayName"], "install": {
            "Windows": "Install Visual Studio Build Tools: https://aka.ms/vs/17/release/vs_BuildTools.exe"
        }}

    print("[bold blue]Running CppSage Doctor...[/bold blue]")
    
    os_map = {
        "win32": "Windows",
        "linux": "Linux",
        "darwin": "Darwin"
    }
    current_os = os_map.get(sys.platform)

    for tool, details in tools.items():
        try:
            result = subprocess.run(details["version_cmd"], capture_output=True, text=True, check=True, shell=True)
            version = result.stdout.strip().splitlines()[0]
            print(f"[green]V {tool} found: {version}[/green]")
        except (subprocess.CalledProcessError, FileNotFoundError):
            print(f"[bold red]X {tool} not found.[/bold red]")
            if current_os:
                install_cmd = details["install"].get(current_os)
                if install_cmd:
                    print(f"  To install, run: [bold cyan]{install_cmd}[/bold cyan]")
                else:
                    print(f"  Installation instructions not available for {current_os}")
            else:
                print(f"  Installation instructions not available for your OS: {sys.platform}")

    if subprocess.run(["pip","install","conan"],capture_output=True).returncode!=0:
        print("failed to install conan")
    code=subprocess.run(["conan","profile","detect"],capture_output=True).returncode
    if code==0 or code==1:
        print("default conan profile created!")
    else:
        print("error while creating conan profile!")
        return
    conan_profile_path=subprocess.run(["conan","profile","path","default"],capture_output=True).stdout.strip()
    modifyConanProfile(conan_profile_path)

@app.command()
def create():
    """
    Generate a new starter C/C++ project with predefined structure.

    Prompts for project name and programming language (C or C++).
    Creates folders, CMake configuration, Clang-format/tidy files, presets,
    and a minimal entry point with Conan-ready setup.
    """

    project_name = typer.prompt("Enter project name").strip()
    lang = typer.prompt("Choose language (C/C++)").strip().lower()

    if not project_name:
        print("[bold red]Project name cannot be empty[/bold red]")
        raise typer.Exit()
    if lang not in ["c", "c++"]:
        print("[bold red]Invalid language. Must be 'C' or 'C++'[/bold red]")
        raise typer.Exit()

    ext = "c" if lang == "c" else "cpp"
    root = Path(project_name)
    print(f"[bold green]Scaffolding {lang.upper()} project: {project_name}[/bold green]")

    # Create directory structure
    for folder in [
         "cmake", "res",
        f"{project_name}/src", f"{project_name}/include", "packages"
    ]:
        (root / folder).mkdir(parents=True, exist_ok=True)

    # Root CMakeLists.txt
    cmake_root = f"""#Auto Generated Root CMake file by Sage
#Copyright(c) 2025 None.All rights reerved.
cmake_minimum_required(VERSION 3.6...3.31)
project({project_name} VERSION 0.1.0 LANGUAGES CXX C)
include(cmake/config.cmake)
#@add_find_package Warning: Do not remove this line

#@add_subproject Warning: Do not remove this line
add_subdirectory({project_name})
"""
    (root / "CMakeLists.txt").write_text(cmake_root)

    # config.cmake
    config_cmake = f"""#Auto Generated Root CMake file by Sage
#None
set(CMAKE_CXX_STANDARD 20)
set(CMAKE_CXX_STANDARD_REQUIRED True)
set(CMAKE_EXPORT_COMPILE_COMMANDS ON)
option(STATIC_LINK "Enable static linking" ON)
option(ENABLE_TESTS "GTests" OFF)
if(STATIC_LINK)
  set(BUILD_SHARED_LIBS OFF)
  if (WIN32)
      set(CMAKE_MSVC_RUNTIME_LIBRARY "MultiThreaded$<$<CONFIG:Debug>:Debug>")
  else()
      set(CMAKE_EXE_LINKER_FLAGS "${{CMAKE_EXE_LINKER_FLAGS}} -static")
  endif()
else()
  set(BUILD_SHARED_LIBS ON)
  if(WIN32)
    set(CMAKE_MSVC_RUNTIME_LIBRARY "MultiThreaded$<$<CONFIG:Debug>:Debug>DLL")
  endif()
endif()
set(COMPANY "None")
string(TIMESTAMP CURRENT_YEAR "%Y")
set(COPYRIGHT "Copyright(c) ${{CURRENT_YEAR}} ${{COMPANY}}.")
include_directories(${{CMAKE_BINARY_DIR}} ${{CMAKE_SOURCE_DIR}})
configure_file(res/config.h.in {project_name}config.h)
if (CMAKE_CXX_COMPILER_ID MATCHES "Clang")
    message(STATUS "Enabling secure coding features for Clang")
    add_compile_options(
        -Xclang
        -Wall -Wextra -Wpedantic
        -Wshadow -Wold-style-cast
        -Wcast-align -Wnull-dereference
        -Wformat=2 -Wformat-security
        -fstack-protector-strong
        -D_FORTIFY_SOURCE=2
        -fno-common
        #-Werror
    )
endif()
"""
    (root / "cmake/config.cmake").write_text(config_cmake)

    # res/config.h.in
    config_h = f"""#ifndef __{project_name}__
    #define __{project_name}__
#include <string_view>
namespace Project {{
    constexpr std::string_view VERSION_STRING = "@{project_name}_VERSION_MAJOR@.@{project_name}_VERSION_MINOR@.@{project_name}_VERSION_PATCH@";
    constexpr std::string_view COMPANY_NAME = "@COMPANY@";
    constexpr std::string_view COPYRIGHT_STRING = "@COPYRIGHT@";
    constexpr std::string_view PROJECT_NAME = "@PROJECT_NAME@";
}}
#endif
"""
    (root / "res/config.h.in").write_text(config_h)

    # Subproject CMakeLists.txt
    subproject_cmake = f"""add_executable({project_name} src/main.{ext}) # Add your Source Files here
#@add_target_link_libraries Warning: Do not remove this line
"""
    (root / project_name / "CMakeLists.txt").write_text(subproject_cmake)

    # Other essential files
    (root / "CMakePresets.json").write_text("""

{
  "version": 2,
  "configurePresets": [
    {
      "name": "default",
      "generator": "Ninja",
      "binaryDir": "${sourceDir}/build/",
      "cacheVariables": {
        "CMAKE_TOOLCHAIN_FILE": "packages/install/conan_toolchain.cmake",
        "STATIC_LINK": false,
        "CMAKE_BUILD_TYPE":"Release",
        "CMAKE_CXX_COMPILER":"clang++",
        "CMAKE_C_COMPILER":"clang"
      }
    },
    {"name": "debug",
    "inherits":"default",
    "cacheVariables": {
      "CMAKE_BUILD_TYPE":"Debug"
    }
    },
    {
      "name": "windows",
      "inherits": "default",
      "cacheVariables": {
        "CMAKE_CXX_COMPILER":"clang-cl",
        "CMAKE_C_COMPILER":"clang-cl"
      }
    },
    {
      "name": "debug-windows",
      "inherits":"windows",
      "cacheVariables": {
        "CMAKE_BUILD_TYPE":"Debug"
      }
    }
  ]
}""")
    (root / "packages/requirements.txt").write_text("[requires]\n[generators]\nCMakeDeps\nCMakeToolchain\n")
    (root / project_name / f"src/main.{ext}").write_text("int main() {\n    return 0;\n}")

    for name, content in {
        ".clang-format": "BasedOnStyle: Google",
        ".clangd": "CompileFlags:\n CompilationDatabase: build\n",
        ".editorconfig": "root = true",
        ".gitignore": "build/\npackages/install/\n.vscode/",
    }.items():
        (root / name).write_text(content)

    print(f"[bold cyan]{project_name} has been created successfully![/bold cyan]")





import re

def _process_conan_output(output: str):
    """Parses the output of a conan install command to find package information."""
    find_packages = []
    target_link_libraries = []
    
    find_package_regex = re.compile(r"find_package\(([^)]+)\)")
    target_link_regex = re.compile(r"target_link_libraries\(\.\.\. ([^)]+)\)")

    for line in output.splitlines():
        find_match = find_package_regex.search(line)
        if find_match:
            find_packages.append(f"find_package({find_match.group(1)})")

        target_match = target_link_regex.search(line)
        if target_match:
            target_link_libraries.append(target_match.group(1))
            
    return find_packages, target_link_libraries

def runInstall(package:str=None,version:str=None):
    req_path = f"packages/{packageFileName}"

    if not package:
        if os.path.isfile(req_path):
            print("[bold yellow]Installing dependencies from requirements.txt[/bold yellow]")
            subprocess.run(["conan", "install", req_path, "--output-folder", "packages/install", "--build=missing","-c","tools.cmake.cmaketoolchain:generator=Ninja","-s",f"build_type={buildType}"])
            return
        else:
            print("[bold red]Missing requirements.txt[/bold red]")
        return
    
    req_dir=os.path.dirname(req_path)
    os.makedirs(req_dir,exist_ok=True)

    if not os.path.isfile(req_path):
        with open(req_path, "w") as f:
            f.write("[requires]\n[generators]\nCMakeDeps\nCMakeToolchain\n")

    if not version:
        print(f"[bold green]No version provided for {package}. Fetching latest...[/bold green]")
        search = subprocess.run(["conan", "search", package], capture_output=True, text=True)
        if search.returncode != 0:
            print(f"[bold red]Conan search failed for {package}[/bold red]")
            return
        lines = search.stdout.strip().splitlines()
        version = lines[-1].split("/")[1] if lines else ""

    if not version:
        print(f"[bold red]No available versions found for {package}[/bold red]")
        return

    full_package = f"{package}/{version}"
    if len(full_package)>0:
        print(f"[bold yellow]Installing: {full_package}[/bold yellow]")

        with open(req_path, "r") as f:
            lines = f.readlines()

        if full_package + "\n" not in lines:
            for i, line in enumerate(lines):
                if line.strip() == "[requires]":
                    lines.insert(i + 1, full_package + "\n")
                    break
            with open(req_path, "w") as f:
                f.writelines(lines)
        else:
            print(f"[bold yellow]{full_package} is already listed[/bold yellow]")


    result = subprocess.run(["conan", "install", req_path, "--output-folder", "packages/install", "--build=missing"], capture_output=True, text=True)
    if result.returncode != 0:
        print("[bold red]Conan installation failed.[/bold red]")
        print(result.stderr)
        return

    # --- Auto-update CMakeLists.txt from conan output ---
    project_name = ""
    with open("CMakeLists.txt", "r") as f:
        for line in f:
            if line.strip().startswith("project("):
                project_name = line.strip().split("(")[1].split()[0]
                break
    
    if not project_name:
        print("[bold red]Could not find project name in CMakeLists.txt[/bold red]")
        return

    find_packages, target_link_libraries = _process_conan_output(result.stderr)
    
    if find_packages:
        with open("CMakeLists.txt", "r+") as f:
            content = f.read()
            f.seek(0)
            lines = content.splitlines()
            # Find the placeholder and insert the new packages
            for i, line in enumerate(lines):
                if "#@add_find_package" in line:
                    for pkg in find_packages:
                        if pkg not in content:
                            lines.insert(i + 1, pkg)
                    break
            f.write('\n'.join(lines))


    if target_link_libraries:
        subproject_cmake_path = f"{project_name}/CMakeLists.txt"
        if os.path.isfile(subproject_cmake_path):
            with open(subproject_cmake_path, "r+") as f:
                content = f.read()
                f.seek(0)
                lines = content.splitlines()
                for i, line in enumerate(lines):
                    if "#@add_target_link_libraries" in line:
                        for lib in target_link_libraries:
                            link_str = f"target_link_libraries({project_name} PRIVATE {lib})"
                            if link_str not in content:
                                lines.insert(i + 1, link_str)
                        break
                f.write('\n'.join(lines))
        else:
            print(f"[bold red]Subproject CMakeLists.txt not found at {subproject_cmake_path}[/bold red]")

def onCompile():
    if not os.path.isfile("CMakeLists.txt"):
        print("[bold red]Missing CMakeLists.txt[/bold red]")
        return
    
    build_dir = f"build"
    if not os.path.isdir(build_dir):
        runInstall()
        result = subprocess.run(["cmake", "--preset", preset])
        if result.returncode != 0:
            print("[bold red]CMake configuration failed[/bold red]")
            return
    
    subprocess.run(["cmake", "--build", build_dir, "--parallel"])

def onRun(args:list[str]=[]):
    if not args:
        print("[bold red]Specify at least one app to build/run[/bold red]")
        return

    for arg in args:
        path_direct = f"build/{arg}{execulableExtention}"
        path_nested = (f"build/{arg}/{arg}{execulableExtention}")
        if os.path.isfile(path_direct):
            subprocess.run([path_direct])
        elif os.path.isfile(path_nested):
            subprocess.run([path_nested])
        else:
            print(f"[bold red]Executable '{arg}' not found in {path_direct} or {path_nested}[/bold red]")

@app.command()
def debug(path: str = typer.Option(None, "--path", "-p", help="Path to the project directory.")):
    """Run the debug build of the project."""
    _change_directory(path)
    global buildType
    buildType="Debug"
    global preset
    preset=debugPreset
    req_path = f"packages/{packageFileName}"
    if os.path.isfile(req_path):
        subprocess.run(["conan", "install", req_path, "--output-folder", "packages/install", "--build=missing","-c","tools.cmake.cmaketoolchain:generator=Ninja","-s",f"build_type={buildType}"])
    else:
        print(f"[bold red]Missing {req_path}[/bold red]")
        return
    onCompile()


def modifyConanProfile(conan_profile_path:str):
    if not os.path.isfile(conan_profile_path):
        print(f"{conan_profile_path} doesn't exist!")
        return
    with open(conan_profile_path, "r") as file:
        file_data = file.readlines()

    # Set compiler.cppstd to 20
    found_cppstd = False
    for index, line in enumerate(file_data):
        if line.strip().startswith("compiler.cppstd"):
            file_data[index] = "compiler.cppstd=20\n"
            found_cppstd = True
            break
    if not found_cppstd:
        file_data.append("compiler.cppstd=20\n")

    # Add &:compiler=clang if not present
    found_compiler = False
    for line in file_data:
        if line.strip() == "&:compiler=clang":
            found_compiler = True
            break
    if not found_compiler:
        file_data.append("&:compiler=clang\n")

    with open(conan_profile_path, "w") as file:
        file.writelines(file_data)
    
def processConanLogAndFindPackageStr(result:str):
    find_packages=[]
    target_link:str=""
    for line in result.splitlines():
        if line.strip().find("find_package"):
            find_packages.append(line.strip())
        if line.strip().find("target_link"):
            target_link=line.strip()

    return [find_packages,target_link]