---
description: Initialize a new development plan bundle with idea, product, and features structure.
---
# SpectFact Plan Init Command (greenfield & brownfield)

## User Input

```text
$ARGUMENTS
```

You **MUST** consider the user input before proceeding (if not empty).

## ⚠️ CRITICAL: CLI Usage Enforcement

**YOU MUST ALWAYS USE THE SPECFACT CLI**. Never create artifacts directly or implement functionality.

### Rules

1. **ALWAYS execute CLI first**: Run `specfact plan init` before any plan creation - execute the CLI command before any other operations
2. **NEVER write code**: Do not implement plan initialization logic - the CLI handles this
3. **NEVER create YAML/JSON directly**: All plan bundles must be CLI-generated
4. **NEVER bypass CLI validation**: CLI ensures schema compliance and metadata - use it, don't bypass its validation
5. **Use CLI output as grounding**: Parse CLI output, don't regenerate or recreate it - use the CLI output as the source of truth
6. **NEVER manipulate internal code**: Do NOT use Python code to directly modify PlanBundle objects, Feature objects, or any internal data structures. The CLI is THE interface - use it exclusively.
7. **No internal knowledge required**: You should NOT need to know about internal implementation details (PlanBundle model, Feature class, etc.). All operations must be performed via CLI commands.

### What Happens If You Don't Follow This

- ❌ Artifacts may not match CLI schema versions
- ❌ Missing metadata and telemetry
- ❌ Format inconsistencies
- ❌ Validation failures
- ❌ Works only in Copilot mode, fails in CI/CD
- ❌ Breaks when CLI internals change
- ❌ Requires knowledge of internal code structure

## ⏸️ Wait States: User Input Required

**When user input is required, you MUST wait for the user's response.**

### Wait State Rules

1. **Never assume**: If input is missing, ask and wait
2. **Never continue**: Do not proceed until user responds
3. **Be explicit**: Clearly state what information you need
4. **Provide options**: Give examples or default suggestions

### Example Wait States

#### Missing Required Argument

```text
❌ WRONG: "Assuming --out is '.specfact/plans/main.bundle.yaml' and continuing..."
✅ CORRECT: 
"What output path would you like to use for the plan bundle? 
(default: .specfact/plans/main.bundle.yaml)
[WAIT FOR USER RESPONSE - DO NOT CONTINUE]"
```

#### Confirmation Required

```text
❌ WRONG: "Proceeding with interactive mode..."
✅ CORRECT:
"Will execute: specfact plan init --interactive --out .specfact/plans/main.bundle.yaml
Continue? (y/n)
[WAIT FOR USER RESPONSE - DO NOT CONTINUE]"
```

## Goal

Create a new development plan bundle. The plan bundle includes idea, business context, product structure (themes and releases), and initial features with stories.

**Two Approaches:**

1. **Greenfield** - Start from scratch with manual plan creation (interactive prompts)
2. **Brownfield** - Scan existing codebase to import structure (`specfact import from-code`), then refine interactively

The user should choose their approach at the beginning of the interactive flow.

## Operating Constraints

**STRICTLY READ-ONLY**: Do **not** modify the codebase. All plan bundles must be generated by the specfact CLI.

**Command**: `specfact plan init`

**Mode Auto-Detection**: The CLI automatically detects operational mode (CI/CD or CoPilot) based on environment. No need to specify `--mode` flag. Mode is detected from:

- Environment variables (`SPECFACT_MODE`)
- CoPilot API availability
- IDE integration (VS Code/Cursor with CoPilot)
- Defaults to CI/CD mode if none detected

## 🔄 Dual-Stack Workflow (Copilot Mode)

When in copilot mode, follow this three-phase workflow:

### Phase 1: CLI Grounding (REQUIRED)

**ALWAYS execute CLI first** to get structured, validated output:

```bash
specfact plan init --interactive --out <output_path>
```

**Note**: Mode is auto-detected by the CLI. No need to specify `--mode` flag.

**Capture from CLI output**:

- CLI-generated plan bundle (`.specfact/plans/main.bundle.yaml` or specified path)
- Metadata (timestamps, validation results)
- Telemetry (execution time, feature/story counts)

### Phase 2: LLM Enrichment (OPTIONAL, Copilot Only)

**Purpose**: Add semantic understanding to CLI output

**What to do**:

- Read CLI-generated plan bundle
- Research codebase for additional context (for brownfield approach)
- Suggest improvements to features/stories
- Extract business context from code comments/docs

**What NOT to do**:

- ❌ Create YAML/JSON artifacts directly
- ❌ Modify CLI artifacts directly
- ❌ Bypass CLI validation

**Output**: Generate enrichment report (Markdown) with insights

### Phase 3: CLI Artifact Creation (REQUIRED)

**Final artifacts MUST be CLI-generated**:

**If enrichment was generated**:

- Present CLI output + enrichment report side-by-side
- User can manually apply enrichments via CLI or interactive mode

**If no enrichment**:

- Use CLI-generated artifacts as-is

**Result**: All artifacts are CLI-generated (ensures format consistency, metadata, telemetry)

## Execution Steps

### 1. Parse Arguments

Extract arguments from user input:

- `--interactive/--no-interactive` - Interactive mode with prompts (default: interactive)
- `--out PATH` - Output plan bundle path (optional, default: `.specfact/plans/main.bundle.yaml`)
- `--scaffold/--no-scaffold` - Create complete `.specfact/` directory structure (default: scaffold)

For single quotes in args like "I'm Groot", use escape syntax: e.g `'I'\''m Groot'` (or double-quote if possible: `"I'm Groot"`).

### 2. Ensure Directory Structure

If `--scaffold` is enabled (default):

- Create `.specfact/` directory structure:
  - `.specfact/plans/` - Plan bundles
  - `.specfact/protocols/` - Protocol definitions (FSM)
  - `.specfact/reports/` - Analysis and comparison reports
  - `.specfact/reports/brownfield/` - Brownfield analysis reports
  - `.specfact/reports/comparison/` - Plan comparison reports
  - `.specfact/reports/enforcement/` - Enforcement reports

If `--no-scaffold`, ensure minimum structure exists.

### 3. Interactive vs Non-Interactive Mode

#### Non-Interactive Mode (`--no-interactive`)

Create a minimal plan bundle with:

- `version: "1.0"`
- Empty `idea` (None)
- Empty `business` (None)
- Empty `product` (themes: [], releases: [])
- Empty `features` ([])

Write to output path and exit.

#### Interactive Mode (default)

### 2. Choose Plan Creation Approach

**WAIT STATE**: Ask the user which approach they want and **WAIT for their response**:

```text
Plan Creation Approach:
1. Greenfield - Start from scratch (manual plan creation)
2. Brownfield - Import from existing codebase (scan repository first)

Choose option (1 or 2): _
[WAIT FOR USER RESPONSE - DO NOT CONTINUE]
```

**If user chooses option 2 (Brownfield)**:

1. **Execute CLI brownfield analysis first** (REQUIRED):

   ```bash
   specfact import from-code --repo . --name <plan_name> --confidence 0.7
   ```

   **WAIT STATE**: If `--name` is not provided, ask user for plan name and **WAIT**:

   ```text
   "What name would you like to use for the brownfield analysis? 
   (e.g., 'my-project', 'API Client v2')
   [WAIT FOR USER RESPONSE - DO NOT CONTINUE]"
   ```

   - This CLI command analyzes the codebase and generates an auto-derived plan bundle
   - Plan is saved to: `.specfact/plans/<name>-<timestamp>.bundle.yaml` (where `<name>` is the sanitized plan name)
   - **Capture CLI output**: Plan bundle path, feature/story counts, metadata

2. **Load the CLI-generated auto-derived plan**:
   - Read the CLI-generated plan bundle from brownfield analysis
   - Extract features, themes, and structure from the auto-derived plan

3. **Execute CLI plan init with brownfield data**:

   ```bash
   specfact plan init --interactive --out <output_path>
   ```

   - CLI will use the auto-derived plan as starting point
   - Guide user through interactive prompts to refine/add:
     - Idea section (title, narrative, target users, metrics)
     - Business context (if needed)
     - Product themes (confirm/add to auto-derived themes)
     - Features (confirm/refine auto-derived features, add stories if missing)

4. **CLI merges and finalizes**:
   - CLI merges refined idea/business sections with auto-derived features
   - CLI writes final plan bundle to output path

**If user chooses option 1 (Greenfield)**:

- Execute CLI plan init directly:

  ```bash
  specfact plan init --interactive --out <output_path>
  ```

- CLI will guide user through interactive prompts starting with Section 1 (Idea)

### 3. CLI Interactive Flow (Greenfield or Brownfield)

**The CLI handles all interactive prompts**. Your role is to:

- Execute the CLI command
- Present CLI prompts to the user
- Wait for user responses
- Continue CLI execution based on user input

**For reference, the CLI interactive flow includes**:

#### Section 1: Idea

**For Brownfield approach**: Pre-fill with values from auto-derived plan if available (extract from plan bundle's `idea` section or from README/pyproject.toml analysis).

Prompt for:

- **Project title** (required) - If brownfield, suggest from auto-derived plan or extract from README/pyproject.toml
- **Project narrative** (required) - Brief description - If brownfield, suggest from auto-derived plan or README
- **Optional details**:
  - Target users (list) - If brownfield, suggest from auto-derived plan
  - Value hypothesis (text) - If brownfield, suggest from README value proposition
  - Success metrics (dict: `{"metric_name": "target_value"}`) - Suggest based on project type

### 6. Section 2: Business Context (Optional)

Ask if user wants to add business context:

- **Market segments** (list)
- **Problems you're solving** (list)
- **Your solutions** (list)
- **How you differentiate** (list)
- **Business risks** (list)

### 7. Section 3: Product - Themes and Releases

**For Brownfield approach**: Pre-fill themes from auto-derived plan (extract from plan bundle's `product.themes`).

Prompt for:

- **Product themes** (list, e.g., "AI/ML", "Security", "Performance") - If brownfield, pre-fill with themes from auto-derived plan
- **Releases** (optional, interactive loop):
  - Release name (e.g., "v1.0 - MVP")
  - Release objectives (list)
  - Feature keys in scope (list, e.g., `["FEATURE-001", "FEATURE-002"]`) - If brownfield, suggest feature keys from auto-derived plan
  - Release risks (list)
  - Ask if user wants to add another release

### 8. Section 4: Features

**For Brownfield approach**: Pre-fill with features from auto-derived plan. For each feature:

- Show auto-derived feature details (key, title, outcomes, acceptance criteria)
- Ask user to confirm, refine, or add stories
- If features have no stories, prompt to add them interactively

Interactive loop to add features:

- **Feature key** (required, e.g., "FEATURE-001")
- **Feature title** (required)
- **Expected outcomes** (list)
- **Acceptance criteria** (list)
- **Optional details**:
  - Constraints (list)
  - Confidence (0.0-1.0, float)
  - Draft flag (boolean)
- **Stories** (optional, interactive loop):
  - Story key (required, e.g., "STORY-001")
  - Story title (required)
  - Acceptance criteria (list)
  - Optional details:
    - Tags (list, e.g., `["critical", "backend"]`)
    - Confidence (0.0-1.0, float)
    - Draft flag (boolean)

### 9. Sensitive Information Disclosure Gate

**BEFORE generating the final plan bundle**, perform a disclosure gate check to identify potentially sensitive information that should not be published publicly:

1. **Review Business Section** (if provided):
   - **Risks**: Check for internal business concerns (e.g., "Market competition", "Open source sustainability", "Proprietary competition") - **Remove these** as they contain internal strategy
   - **Segments**: Check for specific targeting strategies - **Generalize if needed** (e.g., "GitHub Spec-Kit community" → "Open source developers")
   - **Differentiation**: Check for competitive positioning details - **Keep public differentiators only** (remove strategic positioning)
   - **Problems/Solutions**: Keep only information already published in public docs (README, public guides)

2. **Review Idea Section**:
   - **Metrics**: Check for internal KPIs - **Keep only public success metrics**
   - **Value Hypothesis**: Keep only public value proposition

3. **Review Features Section**:
   - **Features**: Technical implementation details are generally safe to publish (already in codebase)
   - **Stories**: Implementation details are safe

4. **Display Disclosure Warning**:

   ```text
   ⚠️  Disclosure Gate Check
   ============================================================
   
   Potentially Sensitive Information Detected:
   - Business risks: [list of risks]
   - Market segments: [list of segments]
   - Competitive differentiation: [list of differentiators]
   
   This information may contain internal strategy that should not
   be published publicly. Consider:
   1. Removing business section entirely (it's optional)
   2. Sanitizing business section (remove risks, generalize segments)
   3. Keeping as-is if already published in public docs
   
   Proceed with sanitized plan? (y/n)
   ```

5. **If user confirms sanitization**, remove or generalize sensitive information before proceeding.

6. **If user chooses to keep sensitive info**, warn them that it will be included in the plan bundle.

**Note**: For brownfield plans, business context may have been extracted from internal docs. Always review before finalizing.

### 4. CLI Generates Plan Bundle (REQUIRED)

**The CLI generates the plan bundle** with:

- `version: "1.0"`
- `idea`: Idea object (or None if not provided) - From CLI interactive prompts
- `business`: Business object (or None if not provided) - From CLI interactive prompts (after disclosure gate)
- `product`: Product object with themes and releases - From CLI interactive prompts
- `features`: List of Feature objects with stories - From CLI interactive prompts (for brownfield: merged with auto-derived features)

**For Brownfield approach**: CLI merges the auto-derived plan's features with the refined idea/business/product sections from interactive prompts.

### 5. CLI Validates Plan Bundle (REQUIRED)

**The CLI validates the generated plan bundle**:

- Schema validation (Pydantic models)
- Required fields check
- Data type validation
- Report validation results

**If validation fails**: CLI reports errors and does not write the plan bundle.

### 6. CLI Writes Plan Bundle (REQUIRED)

**The CLI writes the plan bundle** to output path:

- Creates parent directories if needed
- Writes YAML with proper formatting
- Reports success with file path

**Final Disclosure Reminder**: Before committing or publishing, verify that the plan bundle does not contain sensitive internal strategy (business risks, specific competitive positioning, internal targets).

### 7. Display Summary

Show plan summary:

- Title
- Themes count
- Features count
- Releases count
- Business context included (yes/no) - warn if sensitive info detected

**Note**: Plans created with this command can later be exported to Spec-Kit format using `specfact sync spec-kit --bidirectional`. The export will generate fully compatible Spec-Kit artifacts (spec.md, plan.md, tasks.md) with all required fields including INVSEST criteria, Scenarios, Constitution Check, and Phase organization.

**Prerequisites for Spec-Kit Sync**: Before running `specfact sync spec-kit --bidirectional`, ensure you have:

- Constitution (`.specify/memory/constitution.md`) created via `/speckit.constitution` command
- The constitution must be populated (not just template placeholders)

## Output Format

### Plan Bundle Structure

```yaml
version: "1.0"
idea:
  title: "My Awesome Project"
  narrative: "A project that does amazing things"
  target_users: ["Developers", "Data Scientists"]
  value_hypothesis: "Users will save 50% of their time"
  metrics:
    user_satisfaction: "> 4.5/5"
    time_saved: "50%"
business:
  segments: ["Enterprise", "SMB"]
  problems: ["Complex workflows", "Time-consuming tasks"]
  solutions: ["Automation", "AI assistance"]
  differentiation: ["Better UX", "Lower cost"]
  # Note: 'risks' field removed - contains internal strategy, not suitable for public disclosure
product:
  themes: ["AI/ML", "Security"]
  releases:
    - name: "v1.0 - MVP"
      objectives: ["Core features", "Basic security"]
      scope: ["FEATURE-001", "FEATURE-002"]
      risks: ["Scope creep", "Timeline delays"]
features:
  - key: "FEATURE-001"
    title: "User Authentication"
    outcomes: ["Secure login", "Session management"]
    acceptance: ["Users can log in", "Sessions persist"]
    constraints: ["Must use OAuth2"]
    confidence: 1.0
    draft: false
    stories:
      - key: "STORY-001"
        title: "Login API"
        acceptance: ["API returns JWT token"]
        tags: ["critical", "backend"]
        confidence: 1.0
        draft: false
```

## Guidelines

### Feature Keys

- Use format: `FEATURE-###` (e.g., `FEATURE-001`, `FEATURE-002`)
- Keys should be unique within a plan
- Sequential numbering recommended

### Story Keys

- Use format: `STORY-###` (e.g., `STORY-001`, `STORY-002`)
- Keys should be unique within a feature
- Sequential numbering recommended

### Confidence Scores

- Range: 0.0-1.0
- Default: 1.0 for manually created plans
- Lower confidence indicates uncertainty or draft status

### Validation

- All required fields must be present
- Data types must match schema
- Feature keys must be unique
- Story keys must be unique within their feature

## Summary

**Key Decision Point**: Always ask the user first whether they want:

1. **Greenfield** - Start from scratch with interactive prompts (standard CLI behavior)
2. **Brownfield** - Import existing codebase structure using `specfact import from-code`, then refine interactively

**For Brownfield**:

- **Execute CLI first**: Run `specfact import from-code --repo . --name <name> --confidence 0.7`
- **Wait for user input**: If `--name` is missing, ask and wait for response
- Load CLI-generated auto-derived plan from `.specfact/plans/<name>-<timestamp>.bundle.yaml`
- **Execute CLI plan init**: Run `specfact plan init --interactive --out <path>`
- CLI uses auto-derived features, themes, and structure as pre-filled suggestions in interactive prompts
- User can confirm, refine, or add to auto-derived content via CLI interactive prompts
- CLI merges refined idea/business sections with auto-derived features
- **CLI performs disclosure gate check** before finalizing (business context may contain internal strategy)

**For Greenfield**:

- **Execute CLI directly**: Run `specfact plan init --interactive --out <path>`
- CLI proceeds with interactive prompts (no pre-filling)
- Standard CLI command behavior

**Disclosure Gate** (handled by CLI):

- **CLI performs disclosure gate check** before generating final plan bundle
- CLI reviews business section for sensitive information (risks, competitive positioning, targeting strategy)
- CLI sanitizes or removes internal strategy information before publishing
- CLI warns user if sensitive information is detected
- CLI gets user confirmation before including sensitive information in plan bundle

**CRITICAL**: All plan bundles MUST be generated by the CLI. Never create YAML/JSON artifacts directly.

## Context

{ARGS}
