#!/usr/bin/env python3
"""
Test the complete round-trip process:
string -> TypeScript compression -> encoding -> Python -> decoding -> decompression -> string

This verifies that the exact same string comes out after the full process.
"""

import base64
import gzip
import json
import sys


def decompress_gzip_base64(compressed_data: str) -> str:
    """
    Python function to decompress data created by TypeScript compressString().
    
    This exactly mirrors the TypeScript decompressString() function:
    1. Decode from base64
    2. Decompress using gzip  
    3. Convert back to UTF-8 string
    """
    try:
        # Step 1: Decode from base64
        compressed_bytes = base64.b64decode(compressed_data)
        
        # Step 2: Decompress using gzip
        decompressed_bytes = gzip.decompress(compressed_bytes)
        
        # Step 3: Convert back to UTF-8 string
        original_string = decompressed_bytes.decode('utf-8')
        
        return original_string
    except Exception as e:
        raise Exception(f"Failed to decompress: {e}")


def test_roundtrip_with_real_data():
    """Test the round-trip process with the actual generated test data."""
    
    print("🔄 Testing Complete Round-Trip Process")
    print("=" * 50)
    print("Process: string → TS compress → Python decompress → string")
    print("=" * 50)
    
    try:
        # Load the test data generated by Node.js (which simulates TypeScript compression)
        with open('compression_test_data.json', 'r', encoding='utf-8') as f:
            test_data = json.load(f)
        
        print(f"📊 Testing {len(test_data['testCases'])} cases")
        print()
        
        all_passed = True
        
        for i, test_case in enumerate(test_data['testCases'], 1):
            print(f"{i}. {test_case['name']}")
            
            if 'error' in test_case:
                print(f"   ❌ SKIP: {test_case['error']}")
                all_passed = False
                continue
            
            # Original string (input)
            original = test_case['original']
            
            # Compressed data (from TypeScript/Node.js)
            compressed = test_case['compressed']
            
            # Python decompression (output)
            try:
                decompressed = decompress_gzip_base64(compressed)
                
                # Verify perfect round-trip
                if decompressed == original:
                    print(f"   ✅ PASS: Round-trip successful")
                    
                    # Show some stats
                    original_len = len(original.encode('utf-8'))
                    compressed_len = len(compressed.encode('utf-8'))
                    
                    if original_len > 0:
                        ratio = (1 - compressed_len / original_len) * 100
                        print(f"   📊 Size: {original_len} → {compressed_len} bytes ({ratio:.1f}% change)")
                    
                    # Show a preview of the data
                    preview = repr(original[:50])
                    if len(original) > 50:
                        preview += "..."
                    print(f"   📝 Data: {preview}")
                    
                else:
                    print(f"   ❌ FAIL: Round-trip failed!")
                    print(f"      Original:    {repr(original[:100])}")
                    print(f"      Decompressed: {repr(decompressed[:100])}")
                    all_passed = False
                    
            except Exception as e:
                print(f"   ❌ FAIL: Decompression error: {e}")
                all_passed = False
            
            print()
        
        print("=" * 50)
        if all_passed:
            print("🎉 SUCCESS: All round-trip tests passed!")
            print("✅ TypeScript compression → Python decompression works perfectly")
            return True
        else:
            print("❌ FAILURE: Some round-trip tests failed")
            return False
            
    except FileNotFoundError:
        print("❌ Test data file not found!")
        print("Run: node generate_test_data.js")
        return False
    except Exception as e:
        print(f"❌ Error: {e}")
        return False


def demonstrate_process():
    """Demonstrate the exact process step by step."""
    
    print("\n" + "=" * 50)
    print("🔍 Step-by-Step Process Demonstration")
    print("=" * 50)
    
    # Use a simple example
    original_string = "Hello, World! 🌍 Testing compression with émojis and spëcial chars"
    
    print(f"1. Original string:")
    print(f"   {repr(original_string)}")
    print(f"   Length: {len(original_string.encode('utf-8'))} bytes")
    print()
    
    # Load compressed version from test data
    try:
        with open('compression_test_data.json', 'r', encoding='utf-8') as f:
            test_data = json.load(f)
        
        # Find our test case (should be the Unicode String case)
        compressed_data = None
        for case in test_data['testCases']:
            if 'émojis and spëcial chars' in case.get('original', ''):
                compressed_data = case['compressed']
                break
        
        if compressed_data:
            print(f"2. After TypeScript compression (base64-encoded gzip):")
            print(f"   {compressed_data[:80]}...")
            print(f"   Length: {len(compressed_data)} characters")
            print()
            
            print(f"3. Python decompression process:")
            print(f"   a) Decode base64 → binary gzip data")
            print(f"   b) Decompress gzip → UTF-8 bytes")
            print(f"   c) Decode UTF-8 → original string")
            print()
            
            decompressed = decompress_gzip_base64(compressed_data)
            
            print(f"4. Final result:")
            print(f"   {repr(decompressed)}")
            print(f"   Length: {len(decompressed.encode('utf-8'))} bytes")
            print()
            
            if decompressed == original_string:
                print("✅ PERFECT MATCH: Original and decompressed strings are identical!")
            else:
                print("❌ MISMATCH: Strings differ!")
                
        else:
            print("❌ Could not find test case in data")
            
    except Exception as e:
        print(f"❌ Error demonstrating process: {e}")


def main():
    """Main function."""
    success = test_roundtrip_with_real_data()
    demonstrate_process()
    
    sys.exit(0 if success else 1)


if __name__ == '__main__':
    main()
